"""
Copyright © 2019-2022 NVIDIA CORPORATION & AFFILIATES. ALL RIGHTS RESERVED.
 
This software product is a proprietary product of Nvidia Corporation and its affiliates
(the "Company") and all right, title, and interest in and to the software
product, including all associated intellectual property rights, are and
shall remain exclusively with the Company.

This software product is governed by the End User License Agreement
provided with the software product.
"""
import re

from services.modules.base_module import BaseModule
from services.modules.run_command.command_manager import CommandManager


class Crictl(BaseModule):
    """
    Crictl class
    """
    def get_pods(self):
        """
        get pods with data
        :return:list of data
        """
        data: list = []
        result = CommandManager.run_command(connection=self.connection, command='crictl pods')
        if result.rc == 0:
            dict_keys = []
            # output example
            # # crictl pods
            # POD ID              CREATED             STATE               NAME                               NAMESPACE           ATTEMPT             RUNTIME
            # 1253147c9e8b5       4 days ago          Ready               doca-url-filter-qa-h-vrt-009-oob   default             0                   (default)
            for line in result.output.splitlines():
                if 'POD ID' in line:
                    res = re.split(r'\s{2,}', line)
                    if len(res) == 7:
                        for a in res:
                            dict_keys.append(a.strip())
                    else:
                        break
                else:
                    res = re.split(r'\s{2,}', line)
                    if len(res) == 7:
                        pod_data: dict = {}
                        for index in range(7):
                            pod_data.update({dict_keys[index]: res[index]})

                        data.append(pod_data)
            if not data:
                result.update_logger_fail_to_parse_output()
        return data

    def get_containers(self, name: str = ''):
        """
        get container
        :return: list of data
        """
        data: list = []
        cmd = 'crictl ps --all'
        if name:
            cmd = f'{cmd} --name {name}'
        result = CommandManager.run_command(connection=self.connection, command=cmd)
        if result.rc == 0:
            dict_keys = []
            # output example
            # crictl ps --all
            # CONTAINER           IMAGE               CREATED             STATE               NAME                ATTEMPT             POD ID
            # 712d58064a380       946edaad5c704       5 days ago          Running             doca-url-filter     0                   1253147c9e8b5
            for line in result.output.splitlines():
                if 'CONTAINER' in line:
                    res = re.split(r'\s{2,}', line)
                    if len(res) >= 7:
                        for a in res:
                            dict_keys.append(a.strip())
                    else:
                        break
                else:
                    res = re.split(r'\s{2,}', line)
                    if len(res) >= 7 or not dict_keys:
                        container_data: dict = {}
                        for index in range(len(res)):
                            container_data.update({dict_keys[index]: res[index]})

                        data.append(container_data)
            if not data:
                result.update_logger_fail_to_parse_output()
        return data

    def get_container_logs(self, container_id: str):
        """
        get container logs
        :param container_id:
        :return:
        """
        log: str = ''
        result = CommandManager.run_command(connection=self.connection, command=f'crictl logs {container_id} ')
        if result.rc == 0:
            # output example
            # crictl  logs acfa13705172e
            # 2022-02-24 14:17:28,796 INFO Set uid to user 0 succeeded
            # 2022-02-24 14:17:28,807 INFO supervisord started with pid 8
            # 2022-02-24 14:17:29,813 INFO spawned: 'clx' with pid 10
            log = result.output
        return log
