"""
Copyright © 2019-2022 NVIDIA CORPORATION & AFFILIATES. ALL RIGHTS RESERVED.
 
This software product is a proprietary product of Nvidia Corporation and its affiliates
(the "Company") and all right, title, and interest in and to the software
product, including all associated intellectual property rights, are and
shall remain exclusively with the Company.

This software product is governed by the End User License Agreement
provided with the software product.
"""
import logging

logger = logging.getLogger('blueman_models')

class VerifierConfig:
    """
    verifier config class
    """
    def __init__(self,verifier_config_file_path:str = ''):
        self._file_path = verifier_config_file_path
        self._kernel_modules_ethernet: list = []
        self._kernel_modules_infiniband: list = []
        self._dpu_temperature_pass_range: list = []
        self._dpu_temperature_warning_range: list = []
        self._dpu_temperature_critical_larger_than: int = 95
        self._disk_usages_pass_range: list = []
        self._disk_usages_warning_range: list = []
        self._disk_usages_failed_range: list = []
        self._disk_wearout_pass_range: list = []
        self._disk_wearout_warning_range: list = []
        self._disk_wearout_failed_range: list = []
        self._memory_usages_pass_range: list = []
        self._memory_usages_warning_range: list = []
        self._memory_usages_failed_range: list = []
        self._cpu_usages_pass_range: list = []
        self._cpu_usages_warning_range: list = []
        self._cpu_usages_failed_range: list = []
        self._exclude_failed_services: list = []
        if self._file_path:
            self.__build_data_from_file()

    def __build_data_from_file(self):
        try:
            import configparser
            import os
            config = configparser.ConfigParser()
            config.read(os.path.abspath(self._file_path))
            if config:
                self._kernel_modules_ethernet = [x.strip() for x in config.get('Health:Kernel_modules','modules_ethernet').split(',')]
                self._kernel_modules_infiniband = [x.strip() for x in
                                                 config.get('Health:Kernel_modules', 'modules_infiniband').split(',')]
                self._dpu_temperature_pass_range = [int(x) for x in config.get('Health:DPU_temperature:Pass', 'range').split(',')]
                self._dpu_temperature_warning_range = [int(x) for x in config.get('Health:DPU_temperature:Warning', 'range').split(',')]
                self._dpu_temperature_critical_larger_than = int(config.get('Health:DPU_temperature:Critical', 'larger_than'))
                self._disk_usages_pass_range = [int(x) for x in config.get('Health:Disk_Usages:Pass', 'range').split(',')]
                self._disk_usages_warning_range = [int(x) for x in config.get('Health:Disk_Usages:Warning', 'range').split(',')]
                self._disk_usages_failed_range = [int(x) for x in config.get('Health:Disk_Usages:Failed', 'range').split(',')]
                self._memory_usages_pass_range = [int(x) for x in config.get('Health:Memory_Usages:Pass', 'range').split(',')]
                self._memory_usages_warning_range = [int(x) for x in config.get('Health:Memory_Usages:Warning', 'range').split(',')]
                self._memory_usages_failed_range = [int(x) for x in config.get('Health:Memory_Usages:Failed', 'range').split(',')]
                self._cpu_usages_pass_range = [int(x) for x in config.get('Health:CPU_Usages:Pass', 'range').split(',')]
                self._cpu_usages_warning_range = [int(x) for x in config.get('Health:CPU_Usages:Warning', 'range').split(',')]
                self._cpu_usages_failed_range = [int(x) for x in config.get('Health:CPU_Usages:Failed', 'range').split(',')]
                self._disk_wearout_pass_range = [int(x) for x in
                                                config.get('Health:Disk_Wearout:Pass', 'range').split(',')]
                self._disk_wearout_warning_range = [int(x) for x in
                                                   config.get('Health:Disk_Wearout:Warning', 'range').split(',')]
                self._disk_wearout_failed_range = [int(x) for x in
                                                  config.get('Health:Disk_Wearout:Failed', 'range').split(',')]
                self._exclude_failed_services = [x.strip() for x in
                                                 config.get('Health:System_services', 'exclude_failed_services').split(',')]

        except Exception as exp:
            logger.error(f'failed to build verifier config {exp}')

    @property
    def kernel_modules_ethernet(self) -> list:
        if not self._kernel_modules_ethernet:
            kernel_modules = 'act_gact, act_mirred, act_skbedit, aes_neon_blk, aes_neon_bs, async_memcpy, async_pq,' \
                             'async_raid6_recov, async_tx, async_xor, auxiliary, bluefield_edac, btrfs, cls_flower,' \
                             'crc_ccitt, crct10dif_ce, dm_multipath, gpio_mlxbf2, i2c_mlxbf, ib_cm, ib_core, ib_ipoib,' \
                             'ib_umad, ib_uverbs, ip_tables, ipmi_devintf, ipmi_msghandler, ipv6, iw_cm, llc,' \
                             ' mlx5_core,mlx5_ib, mlx_bootctl, mlx_compat, mlx_trio, mlxbf_gige, mlxbf_pmc, mlxdevm,' \
                             ' mlxfw,mst_pciconf, nf_conncount, nf_nat, nfnetlink, nfnetlink_cttimeout, nsh,' \
                             ' openvswitch, psample,raid0, raid1, raid10, raid456, raid6_pq, rdma_cm, rdma_ucm,' \
                             ' sbsa_gwdt, sch_fq_codel,sch_ingress, stp, uio, uio_pdrv_genirq, xfrm_algo, xfrm_user,' \
                             ' xor, xor_neon, zstd_compress'
            self._kernel_modules_ethernet = [x.strip() for x in kernel_modules.split(',')]
        return self._kernel_modules_ethernet

    @property
    def kernel_modules_infiniband(self) -> list:
        if not self._kernel_modules_infiniband:
            kernel_modules = 'aes_neon_blk, aes_neon_bs, async_memcpy, async_pq,' \
                             'async_raid6_recov, async_tx, async_xor, auxiliary, bluefield_edac, btrfs,' \
                             'crc_ccitt, crct10dif_ce, dm_multipath, gpio_mlxbf2, i2c_mlxbf, ib_cm, ib_core, ib_ipoib,' \
                             'ib_umad, ib_uverbs, ip_tables, ipmi_devintf, ipmi_msghandler, ipv6, iw_cm, llc,' \
                             ' mlx5_core,mlx5_ib, mlx_bootctl, mlx_compat, mlx_trio, mlxbf_gige, mlxbf_pmc, mlxdevm,' \
                             ' mlxfw,mst_pciconf, nf_conncount, nf_nat, nfnetlink, nsh,' \
                             ' openvswitch, psample,raid0, raid1, raid10, raid456, raid6_pq, rdma_cm, rdma_ucm,' \
                             ' sbsa_gwdt, sch_fq_codel, stp, uio, uio_pdrv_genirq, xfrm_algo, xfrm_user,' \
                             ' xor, xor_neon, zstd_compress'
            self._kernel_modules_infiniband = [x.strip() for x in kernel_modules.split(',')]
        return self._kernel_modules_infiniband

    @property
    def cpu_usages_pass_range(self) -> list:
        if not self._cpu_usages_pass_range:
            self._cpu_usages_pass_range = [0,80]
        return self._cpu_usages_pass_range

    @property
    def cpu_usages_warning_range(self) -> list:
        if not self._cpu_usages_warning_range:
            self._cpu_usages_warning_range = [80,90]
        return self._cpu_usages_warning_range

    @property
    def cpu_usages_failed_range(self) -> list:
        if not self._cpu_usages_failed_range:
            self._cpu_usages_failed_range = [90,100]
        return self._cpu_usages_failed_range

    @property
    def memory_usages_pass_range(self) -> list:
        if not self._memory_usages_pass_range:
            self._memory_usages_pass_range = [0,80]
        return self._memory_usages_pass_range

    @property
    def memory_usages_warning_range(self) -> list:
        if not self._memory_usages_warning_range:
            self._memory_usages_warning_range = [80,90]
        return self._memory_usages_warning_range

    @property
    def memory_usages_failed_range(self) -> list:
        if not self._memory_usages_failed_range:
            self._memory_usages_failed_range = [90,100]
        return self._memory_usages_failed_range

    @property
    def disk_usages_pass_range(self) -> list:
        if not self._disk_usages_pass_range:
            self._disk_usages_pass_range = [0,80]
        return self._disk_usages_pass_range

    @property
    def disk_usages_warning_range(self) -> list:
        if not self._disk_usages_warning_range:
            self._disk_usages_warning_range = [80,90]
        return self._disk_usages_warning_range

    @property
    def disk_usages_failed_range(self) -> list:
        if not self._disk_usages_failed_range:
            self._disk_usages_failed_range = [90,100]
        return self._disk_usages_failed_range

    @property
    def disk_wearout_pass_range(self) -> list:
        if not self._disk_wearout_pass_range:
            self._disk_wearout_pass_range = [0, 70]
        return self._disk_wearout_pass_range

    @property
    def disk_wearout_warning_range(self) -> list:
        if not self._disk_wearout_warning_range:
            self._disk_wearout_warning_range = [70, 90]
        return self._disk_wearout_warning_range

    @property
    def disk_wearout_failed_range(self) -> list:
        if not self._disk_wearout_failed_range:
            self._disk_wearout_failed_range = [90, 110]
        return self._disk_wearout_failed_range

    @property
    def dpu_temperature_pass_range(self) -> list:
        if not self._dpu_temperature_pass_range:
            self._dpu_temperature_pass_range = [-40,70]
        return self._dpu_temperature_pass_range

    @property
    def dpu_temperature_warning_range(self) -> list:
        if not self._dpu_temperature_warning_range:
            self._dpu_temperature_warning_range = [70,95]
        return self._dpu_temperature_warning_range

    @property
    def dpu_temperature_critical_larger_than(self) -> int:
        if not self._dpu_temperature_critical_larger_than:
            self._dpu_temperature_critical_larger_than = 95
        return self._dpu_temperature_critical_larger_than

    @property
    def failed_services_to_exclude(self) -> list:
        if not self._exclude_failed_services:
            exclude_services = 'NetworkManager-wait-online.service,systemd-networkd-wait-online.service'
            self._exclude_failed_services = [x.strip() for x in exclude_services.split(',')]
        return self._exclude_failed_services