"""
Copyright © 2019-2022 NVIDIA CORPORATION & AFFILIATES. ALL RIGHTS RESERVED.
 
This software product is a proprietary product of Nvidia Corporation and its affiliates
(the "Company") and all right, title, and interest in and to the software
product, including all associated intellectual property rights, are and
shall remain exclusively with the Company.

This software product is governed by the End User License Agreement
provided with the software product.
"""
import crypt, spwd
import secrets
import jwt
from datetime import datetime, timedelta
import hashlib

from services.blueman.models.model import Model, logger


class LoginCredential:
    """
    LoginCreadential class
    """
    secrets_key_for_hashing = secrets.token_hex(256)
    permitted_users: dict = {}

    @classmethod
    def verify_credential(cls, username: str, password: str) -> bool:
        """
        verify credential of login username and password
        :param username:
        :param password:
        :return: true if success false if not
        """
        try:
            if cls.__verify_if_exist_in_hash_table(username=username, password=password):
                return True
            else:
                permitted = False
                if not Model.connection:
                    encoded_password = spwd.getspnam(username).sp_pwdp
                    permitted = crypt.crypt(password, encoded_password) == encoded_password
                else:
                    permitted = Model.connection.verify_login_credentials(username=username, password=password)
                if permitted:
                    cls.__add_to_hash_table(username=username, password=password)
                    return True
            return False
        except Exception as exp:
            logger.error(exp)
            return False

    @classmethod
    def __add_to_hash_table(cls, username: str, password: str):
        """
        add usernam and password hash to permitted users with jwt_payload and exp time
        :param username:
        :param password:
        :return:
        """
        user_and_password = username + password
        user_and_password_string_hash = hashlib.sha256(user_and_password.encode()).hexdigest()
        jwt_payload = jwt.encode({
            'exp': datetime.utcnow() + timedelta(minutes=10)
        }, cls.secrets_key_for_hashing, algorithm="HS256")
        cls.permitted_users.update({user_and_password_string_hash: {'jwt_payload': jwt_payload}})

    @classmethod
    def __verify_if_exist_in_hash_table(cls, username: str, password: str):
        """
        verify if username and password hash exist in permitted users and not expired
        :param username:
        :param password:
        :return:
        """
        user_and_password = username + password
        user_and_password_string_hash = hashlib.sha256(user_and_password.encode()).hexdigest()
        if cls.permitted_users.get(user_and_password_string_hash):
            user_info = cls.permitted_users.get(user_and_password_string_hash)
            jwt_payload = user_info.get('jwt_payload')
            if jwt_payload:
                try:
                    jwt.decode(jwt_payload, cls.secrets_key_for_hashing, algorithms="HS256")
                    return True
                except jwt.exceptions.ExpiredSignatureError as exp:
                    return False
        return False