"""
Copyright © 2019-2022 NVIDIA CORPORATION & AFFILIATES. ALL RIGHTS RESERVED.
 
This software product is a proprietary product of Nvidia Corporation and its affiliates
(the "Company") and all right, title, and interest in and to the software
product, including all associated intellectual property rights, are and
shall remain exclusively with the Company.

This software product is governed by the End User License Agreement
provided with the software product.
"""
import re
from services.blueman.models.model import Model
from services.modules.system.cpu.cpu import CPU
from services.modules.system.general.general import General
from services.modules.system.logs.logs import Logs
from services.modules.system.memory.memory import Memory
from services.modules.system.storage.storage import Storage


class SystemHealth:
    """
    SystemHealth class
    """
    @classmethod
    def verify_cpu_usages(cls):
        """
        verify cpu usages and core process
        :return:
        """
        # cpu usages and percentage
        cpu_obj = CPU(Model.connection)
        pid_stats_data = cpu_obj.get_pid_stat()
        cores_usages_data = {}
        verify_config_obj = Model.verifier_config()
        for core in sorted(pid_stats_data.keys()):
            core_usages = 0.0
            for processes in pid_stats_data.get(core).get('Processes'):
                process_core_usage = processes.get('%CPU')
                if process_core_usage:
                    core_usages = core_usages + float(process_core_usage)
            value = float(core_usages)
            status = 'failed'
            if  verify_config_obj.cpu_usages_pass_range[0] <= value <= verify_config_obj.cpu_usages_pass_range[1]:
                status = 'pass'
            elif verify_config_obj.cpu_usages_warning_range[0] <= value <= verify_config_obj.cpu_usages_warning_range[1]:
                status = 'warning'
            elif verify_config_obj.cpu_usages_failed_range[0] <= value <= verify_config_obj.cpu_usages_failed_range[1]:
                status = 'failed'
            cores_usages_data.update({core:{'usages': str(core_usages), 'status': status,'Processes':pid_stats_data.get(core).get('Processes')}})
        return cores_usages_data

    @classmethod
    def verify_memory_usages(cls):
        """
        verify memory usages
        :return:
        """
        res = Memory(Model.connection).get_memory_info()
        if res:
            value = int(res['usages']['value'])
            verify_config_obj = Model.verifier_config()
            status = 'failed'
            if verify_config_obj.memory_usages_pass_range[0] <= value <= verify_config_obj.memory_usages_pass_range[1]:
                status = 'pass'
            elif verify_config_obj.memory_usages_warning_range[0] <= value <= verify_config_obj.memory_usages_warning_range[1]:
                status = 'warning'
            elif verify_config_obj.memory_usages_failed_range[0] <= value <= verify_config_obj.memory_usages_failed_range[1]:
                status = 'failed'
            res['status'] = status
        return res

    @classmethod
    def verify_disk_usages(cls):
        """
        verify disk usages
        :return:
        """
        res = Storage(Model.connection).get_hard_disk_info()
        if res:
            value = int(res['Usages']['value'])
            status = 'failed'
            verify_config_obj = Model.verifier_config()
            if verify_config_obj.disk_usages_pass_range[0] <= value <= verify_config_obj.disk_usages_pass_range[1]:
                status = 'pass'
            elif verify_config_obj.disk_usages_warning_range[0] <= value < \
                    verify_config_obj.disk_usages_warning_range[1]:
                status = 'warning'
            elif verify_config_obj.disk_usages_failed_range[0] <= value <= \
                    verify_config_obj.disk_usages_failed_range[1]:
                status = 'failed'
            res['status'] = status
        return res

    @classmethod
    def verify_dpu_temperature(cls):
        data: dict = {}
        # get temperature of dpu
        temperature, unit = General(Model.connection).get_temperature_of_dpu()
        temperature_data = {}
        temperature_data['temperature'] = {'value': temperature, 'unit': unit}
        if temperature:
            temperature = int(temperature)
            temperature_data['status'] = 'critical'
            verify_config_obj = Model.verifier_config()
            if verify_config_obj.dpu_temperature_pass_range[1] >= temperature >\
                    verify_config_obj.dpu_temperature_pass_range[0]:
                temperature_data['status'] = 'normal'
            elif verify_config_obj.dpu_temperature_warning_range[0] < temperature <=\
                    verify_config_obj.dpu_temperature_warning_range[1]:
                temperature_data['status'] = 'warning'
            elif verify_config_obj.dpu_temperature_pass_range[0] >= temperature >=\
                    verify_config_obj.dpu_temperature_critical_larger_than:  # in case of more than 95 or less than -40
                temperature_data['status'] = 'critical'

        data = temperature_data

        return data

    @classmethod
    def verify_logs(cls):
        """
        verify logs
        :return:
        """
        data: dict = {}
        #dmesg
        log_obj = Logs(Model.connection)
        res = log_obj.verify_dmesg_error_messages()
        if res:
            data['dmesg'] = {'value':res,'status':'failed'}
        else:
            data['dmesg'] = {'value': res, 'status': 'pass'}
        #system log
        res = log_obj.verify_system_logs_error_messages()
        if res:
            data['system_log'] = {'value':res,'status':'failed'}
        else:
            data['system_log'] = {'value': res, 'status': 'pass'}

        #ovs log
        res = log_obj.verify_ovs_error_messages()
        if res:
            data['ovs_log'] = {'value': res, 'status': 'failed'}
        else:
            data['ovs_log'] = {'value': res, 'status': 'pass'}

        return data

    @classmethod
    def show_dmesg_log(cls):
        """
        show dmesg logs
        :return: dict
        """
        return Logs(Model.connection).show_dmesg()

    @classmethod
    def verify_disk_wearout(cls):
        """
        verify disk wear out
        return : list of dict
        """
        data = []
        verify_config_obj = Model.verifier_config()
        res = Storage(Model.connection).get_disk_wearout_info()
        for info in res:
            for info_data_name, info_data_value in info.items():
                if isinstance(info_data_value,dict):
                    # Pre-EOL section
                    for pre_eol_sevirity,pre_eol_meaning in info_data_value.items():
                        data.append({'Description':info_data_name,'Status':pre_eol_meaning,'Severity':pre_eol_sevirity})
                        break
                else:
                    match = re.search(r'\d+%-(\d+)%',info_data_value)
                    if match:
                        value = int(match.group(1))
                        severity = ''
                        if value <= verify_config_obj.disk_wearout_pass_range[1]:
                            severity = 'Normal'
                        elif value > verify_config_obj.disk_wearout_warning_range[0] \
                                and value <= verify_config_obj.disk_wearout_warning_range[1]:
                            severity = 'Warning'
                        elif value > verify_config_obj.disk_wearout_failed_range[0]:
                            severity = 'Urgent'

                        data.append(
                            {'Description': info_data_name, 'Status': info_data_value, 'Severity': severity})

        return data

    @classmethod
    def verify_cpu_frequency(cls) -> (int,str,str):
        """
        verify cpu frequency
        """
        data = (0,'','')

        cpu_frequency = CPU(Model.connection).get_cpu_frequency()
        if cpu_frequency:
            value = cpu_frequency[0]
            unit = cpu_frequency[1]
            if cpu_frequency[1] == 'MHz':
                # TODO: define normal/warning/critical once have all the info
                status = 'normal'
                return value,unit,status
        return data

    @classmethod
    def verify_total_power(cls) -> (int,str,str):
        """
        verify total power
        """
        total_power_and_unit = General(Model.connection).get_total_power_consumption()
        if total_power_and_unit and total_power_and_unit[0]:
            value = total_power_and_unit[0]
            unit = total_power_and_unit[1]
            if value:
                # TODO: define normal/warning/critical once have all the info
                status = 'normal'
                return value,unit,status
        return ()

    @classmethod
    def show_rshim_log(cls):
        """
        show DPU rshim logs
        :return: list
        """
        return Logs(Model.connection).show_bf_rshim_log()

    @classmethod
    def show_nvme_devices_smart_log(cls):
        """
        show nvme devices smart log
        :return: dict
        """
        return Storage(Model.connection).get_nvme_smart_log()