/*
* Copyright (C) Mellanox Technologies Ltd. 2018.  ALL RIGHTS RESERVED.
*
* See file LICENSE for terms.
*/
#include <sys/types.h>
#include <sys/stat.h>
#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <glob.h>
#include <stdbool.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "common/logger.h"
#include "common/status.h"
#include "counters/counter_data.h"
#include "counters/counter_group.h"
#include "common/utils.h"

#define EXAMPLE_STR_LEN 32
// General explanation
//
// Counter porvider contains one or several components and implements counter group callbacks.
//
// - Components:
//     Provider contains all available components. Components may be enabled/disabled via `clx_config.ini`.
//     Component contains several counters of type `clx_counter_info_t`, not the data.
//     Components include all available counters to collect.
// - Counter group:
//     Counter group is reassembled from enabled components as follows:
//     - Make list of all the names from enabled components
//     - If name filtering is enabled, add counters with matching names to the group.
//     - With no filtering enabled, add all counters from the list to the group.
//     Counter groups for all counters providers will be converted to a single counters schema.
//     This process is handled by clx collector outside of provider.
//
// Even though counter group is created outside of provider, user can modify group callbacks
// if the implementation specific structures should be updated.
//


typedef enum example_counter_id {
    EXAMPLE_COUNTER_1_ID = 0,
    EXAMPLE_COUNTER_2_ID,
    EXAMPLE_COUNTER_STRING_ID,
    NUM_EXAMPLE_COUNTERS
} example_counter_id;


/// Example_counters_data_t this is placeholder gor generic data
/// It is updated on each iteration.
typedef struct example_counters_data_t {
    int64_t example_data_1;  // this structure shows
    int64_t example_data_2;
    char    example_data_string[EXAMPLE_STR_LEN];
} example_counters_data_t;


/** @struct example_group_t
 * @brief In this example example_group_t is inherited from clx_counter_group_t.
 * Add fields and customize if needed. Use custom group to cast and access clx_counter_group_t from group callbacks.
 */
typedef struct example_group_t {
    clx_counter_group_t  base;
} example_group_t;


/**
 * @struct example_provider_context_t
 * @brief example of provider implementation-specific data
 * @var example_provider_context_t::comp_idx_to_ptr
 *      Member 'comp_idx_to_ptr' maps component counter index to the address of generic data (i.e. for all available counters)
 * @var example_provider_context_t::group_idx_to_ptr
 *      Member 'group_idx_to_ptr' maps group counter index to the address of generic data (i.e. for all included counters)
 * @var example_provider_context_t::generic_data
 *      Member 'generic_data' contains generic data source. Example provider will read the data from this source. On each iteration values will be updated.
 * @var example_provider_context_t::num_reads
 *      Member 'num_reads' tracks how many reads were made
 */
typedef struct example_provider_context_t {
    // for all available counters
    void*                   comp_idx_to_ptr[NUM_EXAMPLE_COUNTERS];
    // for counters selected to group
    void*                   group_idx_to_ptr[NUM_EXAMPLE_COUNTERS];

    example_counters_data_t generic_data;
    int                     num_reads;

    bool                    example_boolean_param;
} example_provider_context_t;


static void example_finalize_provider(clx_provider_t* provider) {
    log_error("Finalize example counter provider");
    for (int i = 0; i < provider->num_components; i++) {
        clx_free_component_info(provider->components[i]);
    }
    free(provider->components);
    free(provider->extra);
    provider->components     = NULL;
    provider->num_components = 0;
    provider->version        = -1;
}

static bool example_read_counter_group_values(clx_counter_group_t* group, clx_counter_value_t* buffer);

static bool init_example_counters(clx_provider_t* provider, clx_component_t* comp) {
    example_provider_context_t* provider_ctx = (example_provider_context_t*)(provider->extra);

    clx_counter_info_t** counters = (clx_counter_info_t**) calloc(NUM_EXAMPLE_COUNTERS, sizeof(clx_counter_info_t*));

    if (counters == NULL) {
        log_error("Failed to re-allocate array of %d counters: %s", comp->num_counters + 1, strerror(errno));
        return false;
    }
    comp->counters = counters;

    // Allocate fisrt counter object with numberic int64_t value type
    clx_counter_info_t* counter_1 = (clx_counter_info_t*) calloc(1, sizeof(clx_counter_info_t));
    if (counter_1 == NULL) {
        log_error("Failed to allocate counter object: %s", strerror(errno));
        return false;
    }
    counter_1->id            = EXAMPLE_COUNTER_1_ID;
    counter_1->name          = strdup("example_counter:fast");
    counter_1->description   = strdup("Fast growing example_counter");
    counter_1->units         = strdup("");
    counter_1->data_type     = CLX_DATA_TYPE_INT64;
    counter_1->length        = 8;
    counter_1->counting_type = CLX_COUNTING_TYPE_RUNNING_SUM;

    // Store counter 1 and example data 1 address to comp_idx_to_ptr map
    comp->counters[comp->num_counters] = counter_1;
    provider_ctx->comp_idx_to_ptr[comp->num_counters] = &provider_ctx->generic_data.example_data_1;
    comp->num_counters += 1;

    // Allocate second counter object with numeric int64_t value type
    clx_counter_info_t* counter_2 = (clx_counter_info_t*) calloc(1, sizeof(clx_counter_info_t));
    if (counter_2 == NULL) {
        log_error("Failed to allocate counter object: %s", strerror(errno));
        return false;
    }
    counter_2->id            = EXAMPLE_COUNTER_2_ID;
    counter_2->name          = strdup("example_counter:slow");
    counter_2->description   = strdup("Slow-growing example counter");
    counter_2->units         = strdup("");
    counter_2->data_type     = CLX_DATA_TYPE_INT64;
    counter_2->length        = 8;
    counter_2->counting_type = CLX_COUNTING_TYPE_RUNNING_SUM;

    // Store counter 2 and example data 2 address to comp_idx_to_ptr map
    comp->counters[comp->num_counters] = counter_2;
    provider_ctx->comp_idx_to_ptr[comp->num_counters] = &provider_ctx->generic_data.example_data_2;
    comp->num_counters += 1;

    // Allocate 3 counter object, which will contain string value
    clx_counter_info_t* counter_3 = (clx_counter_info_t*) calloc(1, sizeof(clx_counter_info_t));
    if (counter_3 == NULL) {
        log_error("Failed to allocate counter object: %s", strerror(errno));
        return false;
    }
    counter_3->id            = EXAMPLE_COUNTER_STRING_ID;
    counter_3->name          = strdup("example_text");
    counter_3->description   = strdup("Text value");
    counter_3->units         = strdup("");
    counter_3->data_type     = CLX_DATA_TYPE_STRING;
    counter_3->length        = EXAMPLE_STR_LEN;  // Note: only even positive lengths are allowed
    counter_3->counting_type = CLX_COUNTING_TYPE_RUNNING_SUM;

    // Store string counter and example data string address to comp_idx_to_ptr map
    comp->counters[comp->num_counters] = counter_3;
    provider_ctx->comp_idx_to_ptr[comp->num_counters] = &provider_ctx->generic_data.example_data_string;
    comp->num_counters += 1;

    return true;
}


static bool init_example_component(clx_provider_t* provider) {
    ///
    /// Initialize counter porvider component
    ///

    /// 1. Allocate component
    clx_component_t* comp = (clx_component_t*) calloc(1, sizeof(clx_component_t));
    if (comp == NULL) {
        log_error("Failed to allocate example provider counters_example component: %s",
                  strerror(errno));
        return NULL;
    }

    /// 2. Fill component fields and add all counters
    comp->id            = provider->num_components;
    comp->name          = strdup("counters_example");
    comp->description   = strdup("Predictably changing counters");
    comp->enabled       = true;
    comp->status_string = NULL;
    comp->counters      = NULL;
    comp->num_counters  = 0;

    if (!init_example_counters(provider, comp)) {
        log_error("Failed initialize counters of example counters_example component");
        clx_free_component_info(comp);
        return false;
    }

    /// 3. Add counters to component
    provider->components[provider->num_components] = comp;
    provider->num_components++;
    log_debug("example component '%s' has been initialized.", comp->name);

    /// 4. Add more components if needed
    return true;
}


static bool init_components(clx_provider_t* provider, const struct clx_options_t* app_opts) {
    ///
    /// Reimplement with real components
    ///

    /// 1. Allocate components and store the component info into provider
    clx_component_t** components = (clx_component_t**) calloc(1, sizeof(clx_component_t*));
    if (components == NULL) {
        log_error("Failed to allocate example provider components array: %s", strerror(errno));
        return false;
    }
    provider->components     = components;
    provider->num_components = 0;

    /// 2. Check if component was enabled/disabled in user configuration (app_opts). Use several components if needed.
    bool enabled = clx_comp_is_enabled("counters_example", app_opts);
    if (!enabled) {
        return true;
    }

    /// 3. Initialize only enabled components
    if (!init_example_component(provider)) {
        log_error("example provider failed to initialize counters_example component");
        example_finalize_provider(provider);
        return false;
    }
    return true;
}


static bool example_init_provider(clx_provider_t* provider, const struct clx_options_t* app_opts) {
    ///
    /// Implement `.initialize` provider callback.
    /// Allocate provider-specific structures and create of one or several components.
    /// Provider components are accosiated with counter groups.
    /// Components can be turned on/off from user defuned options (app_opts).
    ///

    provider->extra = (example_provider_context_t*) calloc(1, sizeof(example_provider_context_t));

    if (!init_components(provider, app_opts)) {
        log_error("Failed to initialize example provider components");
        return false;
    }

    return true;
}


static void example_provider_handle_command(clx_provider_t* provider,
                                            const command_request_t* request,
                                            command_response_t* response) {
    /// Optionally implement .command callback to
    /// Process the request from agx server (clxcli) and send the response.
    /// Use to change provider configuration, e.g. to enable/disable components
    /// Note that clxcli should be updated with new commands.

    assert(provider != NULL);
    assert(request  != NULL);
    assert(response != NULL);

    /// Get command from request and prepare responce with `cmd_set_string_by_key`

    /// Option 1. Accept command with command_key and boolean command_new_val to set it
    ///           to example_boolean_param if new val differs from the old one.
    bool command_new_val;
    const char *command_key = "example_boolean_param";
    example_provider_context_t* provider_ctx = (example_provider_context_t*)(provider->extra);
    if (cmd_get_bool_by_key(request, command_key, &command_new_val)) {
        if (command_new_val != provider_ctx->example_boolean_param) {
            provider_ctx->example_boolean_param = command_new_val;
            // do enable/disable
            cmd_set_string_by_key(response, command_key, "example_boolean_param has been changed");
        } else {
            cmd_set_string_by_key(response, command_key, "Nothing to do");
        }
    }


    /// Option 2. Read command as a status string.
    ///           This example considers only 2 available commands:
    ///               - enable
    ///               - disable
    const char* new_string_val = NULL;

    if (cmd_get_string_by_key(request, command_key, &new_string_val)) {
        if (!strcmp("enable", new_string_val)) {
            // enable
            log_info("[example_counters] [%s] do enable", __FUNCTION__);
            cmd_set_string_by_key(response, command_key, "accepted enable command");
        } else if (!strcmp("disable", new_string_val)) {
            // disable
            log_info("[example_counters] [%s] do disable", __FUNCTION__);
            cmd_set_string_by_key(response, command_key, "accepted disable command");
        } else {
            log_error("[example_counters] Unknown '%s' for %s", new_string_val, command_key);
            cmd_set_string_by_key(response, new_string_val, "unknown command");
        }
    }
}


static clx_counter_group_t* example_create_counter_group(const clx_component_t* comp) {
    ///
    /// Allocate counter group from the component and set comp info into the group.
    /// Groups are used by clx collector to create counter schema.
    /// Group will be arranged from enabled component counters.
    /// If the example_group_t has custom fields initialize them here.
    ///
    /// Note that another to access provider-specific fields is
    ///       (example_provider_context_t*)(group->provider->extra)
    example_group_t* group = (example_group_t*) calloc(1, sizeof(example_group_t));
    if (group == NULL) {
        log_error("Failed to allocate memory for example counter group");
        return NULL;
    }
    group->base.component_id         = comp->id;
    group->base.granularity          = CLX_GRANULARITY_NODE;
    group->base.num_counter_sets     = 1;

    return (clx_counter_group_t*) group;
}


static bool example_destroy_counter_group(clx_counter_group_t* group) {
    free(group);
    return true;
}



static bool example_add_counter_to_group(clx_counter_group_t* group,
                                       const clx_counter_info_t* counter) {
    ///
    /// This callback does the following:
    ///    -# coppies mapping idx_to_ptr from component to the counter group,
    ///    -# updates number of group counters
    /// Actual counter info is copied outside of the provider.
    /// Note that if clx_config.ini name match if enabled, number of counters in component and group mi
    ///

    if (counter->id >= NUM_EXAMPLE_COUNTERS) {
        log_error("Failed adding counter to example group: wrong id.");
        return false;
    }

    example_provider_context_t* provider_ctx = (example_provider_context_t*)(group->provider->extra);

    provider_ctx->group_idx_to_ptr[group->num_counters] = provider_ctx->comp_idx_to_ptr[counter->id];
    group->num_counters += 1;
    return true;
}


static bool example_start_counting_group(clx_counter_group_t* group) {
    /// Nothing to do here: return true
    return true;
}


static bool example_stop_counting_group(clx_counter_group_t* group, clx_counter_value_t* buffer) {
    // Nothing to do here: read the latest data and return true
    if (buffer != NULL) {
        example_read_counter_group_values(group, buffer);
    }
    return true;
}


void update_example_data(example_provider_context_t* provider_ctx) {
    // update first data
    provider_ctx->generic_data.example_data_1 += 10;
    provider_ctx->generic_data.example_data_1 %= 100;

    // update second data
    provider_ctx->generic_data.example_data_2 += 1;
    provider_ctx->generic_data.example_data_2 %= 100;

    // update string data
    snprintf(provider_ctx->generic_data.example_data_string,
            sizeof(provider_ctx->generic_data.example_data_string),
            "example_string:num_reads=%d", provider_ctx->num_reads);
    provider_ctx->num_reads++;
}


static bool example_read_counter_group_values(clx_counter_group_t* group,
                                              clx_counter_value_t* values) {
    ///
    /// Implement this function to copy counter values from provider_ctx into values buffer.
    /// Iterate through group counters and copy each value based on its type.
    ///
    example_provider_context_t* provider_ctx = (example_provider_context_t*) group->provider->extra;

    // generate new data
    update_example_data(provider_ctx);

    // Read values
    for (uint32_t counter_set = 0; counter_set < group->num_counter_sets; counter_set++) {
        for (uint32_t idx = 0; idx < group->num_counters; idx++) {
            clx_counter_info_t* counter_info = group->counters[counter_set * group->num_counters + idx];
            clx_counter_value_t* cur_val_ptr = (clx_counter_value_t*) ((unsigned char *) values + counter_info->offset);

            switch (counter_info->data_type) {
            case CLX_DATA_TYPE_INT64:
                cur_val_ptr->i64 = *(int64_t*)provider_ctx->group_idx_to_ptr[idx];
                break;
            case CLX_DATA_TYPE_UINT64:
            case CLX_DATA_TYPE_BIT64:
            case CLX_DATA_TYPE_FP64:
                break;
            case CLX_DATA_TYPE_STRING:
                {
                    strncpy((char*)cur_val_ptr, (char*)provider_ctx->group_idx_to_ptr[idx], counter_info->length);
                }
            break;
            default:
                log_error("unknown data_type: %d", counter_info->data_type);
            }
        }
    }

    return true;
}


/**
 * @struct example_provider
 * @brief Clang-style inherited structure from clx_provider_t. Has provider fields, extra
 * @var example_provider::extra
 * Member extra should be used as implementation-specific structure
 * @var example_provider::group
 * Member struct group contains callbacks for counters group.
 * Although most of interaction with counter group is made by clx collector, these callbacks can be reimplemented
 * if they interact with new provider-specific structures.
*/
static clx_provider_t example_provider = {
    .name               = "example",
    .description        = "Counter provider example",
    .version            = 0x010000,
    .num_components     = 0,
    .components         = NULL,                                /// will be allocated in .initialize callback
    .extra              = NULL,                                /// use for provider specific fields
    .initialize         = &example_init_provider,              /// implement (alloc structs)
    .finalize           = &example_finalize_provider,          /// implement (free structs)
    .command            = &example_provider_handle_command,    /// optional

     // group callbacks
    .group.create       = &example_create_counter_group,       /// ignore
    .group.destroy      = &example_destroy_counter_group,      /// ignore
    .group.add_counter  = &example_add_counter_to_group,       /// implement
    .group.start        = &example_start_counting_group,       /// ignore
    .group.stop         = &example_stop_counting_group,        /// ignore
    .group.read         = &example_read_counter_group_values,  /// implement
};


clx_provider_t* construct_counter_provider(const struct clx_options_t* app_opts) {
    log_debug("Construct example counter provider");
    example_init_provider(&example_provider, app_opts);
    return &example_provider;
}
