# Collectx Provider Plugin Tester
Tester allows user to test new plugin without interaction with clx or clx configuration files.
The clx provider tester is based on `clx_api.h` functions.

Clx provider plugin is either event or counter provider compiled as so-library.

To start writing new provider consider examples `example_counter_provider/example_provider.c` and `example_event_provider/example_provider.c`.

Tester is installed under "/opt/mellanox/collectx/bin/clx_plugin_tester".

## Testing
Once Collectx is built, run tester with example event provider (same for the counter provider):
~~~
./clx_plugin_tester -s example_event_provider/libevents_example_provider.so -n 10 -v 7

[2021-04-19 15:05:24.273] [info] will run for 10 iters with sleep interval 1.000000 sec
[2021-04-19 15:05:24.276] [info] source 0: id='jazz19', tag='tag_default'
[2021-04-19 15:05:24.277] [info] source 1: id='example_id', tag='example_tag'
[2021-04-19 15:05:24.277] [info] Starting to progress:
[2021-04-19 15:05:24.277] [info]     iter = 0
[2021-04-19 15:05:24.277] [info]        progressing source 0: id='jazz19', tag='tag_default'
[2021-04-19 15:05:24.277] [info]        progressing source 1: id='example_id', tag='example_tag'
[2021-04-19 15:05:25.277] [info]     iter = 1
[2021-04-19 15:05:25.277] [info]        progressing source 0: id='jazz19', tag='tag_default'
[2021-04-19 15:05:25.277] [info]        progressing source 1: id='example_id', tag='example_tag'
[2021-04-19 15:05:26.277] [info]     iter = 2
[2021-04-19 15:05:26.277] [info]        progressing source 0: id='jazz19', tag='tag_default'
[2021-04-19 15:05:26.277] [info]        progressing source 1: id='example_id', tag='example_tag'
[2021-04-19 15:05:27.277] [info]     iter = 3
[2021-04-19 15:05:27.278] [info]        progressing source 0: id='jazz19', tag='tag_default'
[2021-04-19 15:05:27.278] [info]        progressing source 1: id='example_id', tag='example_tag'
[2021-04-19 15:05:28.278] [info]     iter = 4
[2021-04-19 15:05:28.278] [info]        progressing source 0: id='jazz19', tag='tag_default'
[2021-04-19 15:05:28.278] [info]        progressing source 1: id='example_id', tag='example_tag'
[2021-04-19 15:05:29.278] [info]     iter = 5
[2021-04-19 15:05:29.278] [info]        progressing source 0: id='jazz19', tag='tag_default'
[2021-04-19 15:05:29.278] [info]        progressing source 1: id='example_id', tag='example_tag'
[2021-04-19 15:05:30.278] [info]     iter = 6
[2021-04-19 15:05:30.278] [info]        progressing source 0: id='jazz19', tag='tag_default'
[2021-04-19 15:05:30.278] [info]        progressing source 1: id='example_id', tag='example_tag'
[2021-04-19 15:05:31.278] [info]     iter = 7
[2021-04-19 15:05:31.278] [info]        progressing source 0: id='jazz19', tag='tag_default'
[2021-04-19 15:05:31.278] [info]        progressing source 1: id='example_id', tag='example_tag'
[2021-04-19 15:05:32.278] [info]     iter = 8
[2021-04-19 15:05:32.278] [info]        progressing source 0: id='jazz19', tag='tag_default'
[2021-04-19 15:05:32.278] [info]        progressing source 1: id='example_id', tag='example_tag'
[2021-04-19 15:05:33.278] [info]     iter = 9
[2021-04-19 15:05:33.278] [info]        progressing source 0: id='jazz19', tag='tag_default'
[2021-04-19 15:05:33.278] [info]        progressing source 1: id='example_id', tag='example_tag'
~~~

to get the data from the provider for 10 iterations.
To check usage run:
~~~
user@jazz19:$ ./clx_plugin_tester
~~~

It will create data folder 'tester_data' and write generated data and schemas according to the following tree:
~~~
user@jazz19: $tree tester_data/
tester_data/
├── 2021
│   └── 0420
│       ├── example_id
│       │   ├── example_tag_1618918641409380.bin
│       │   └── example_tag_1618919082653059.bin
│       └── jazz19
│           ├── tag_default_1618918641409085.bin
│           └── tag_default_1618919082652686.bin
└── schema
    ├── schema_c4463f7dab85186fcd5694fec7c135c0.json
    └── schema_ec9d36e6844048461d96c859e176cf32.json

~~~

### Reading data with clxcli.py
To query generated data use clxcli.py
1. `cd Collectx/server/clxcli`
2. create a fieldset file for your event under counterset folder
~~~
   vi counterset/example.fset
~~~
   and add the following line to that file
~~~
   [example_event]
~~~
3. run clxcli:
~~~
    python clxcli.py
~~~
4. set the directory root
~~~
    CollectX:  set_data_root{...}/tester_data

    Setting data root to:  {...}/tester_data
    Setting schema dir to: {...}/tester_data/schema
    Loaded 2 schemas from {...}/tester_data/schema/schema*.json
    Loaded 4 counter sets
    Loaded 5 field sets
    Loaded 4 filters
~~~
5.	query the events according to fset file
~~~
    CollectX:  events example past=1d

    time,source,timestamp,name,source_index,value,
    2021-04-20T13:56:54.301991,example_id,1618916214301991,test event,1,0,
    2021-04-20T13:56:54.302154,example_id,1618916214302154,test event,1,1,
    2021-04-20T13:56:54.302349,example_id,1618916214302349,test event,1,2,
    2021-04-20T13:56:54.302792,example_id,1618916214302792,test event,1,3,
    2021-04-20T13:56:54.303081,example_id,1618916214303081,test event,1,4,
    2021-04-20T13:56:54.303486,example_id,1618916214303486,test event,1,5,
    2021-04-20T13:56:54.303996,example_id,1618916214303996,test event,1,6,
    2021-04-20T13:56:54.304312,example_id,1618916214304312,test event,1,7,
    2021-04-20T13:56:54.304746,example_id,1618916214304746,test event,1,8,
    2021-04-20T13:56:54.305297,example_id,1618916214305297,test event,1,9,
    2021-04-20T13:56:54.301900,jazz19,1618916214301900,even test event,0,0,
    2021-04-20T13:56:54.301900,jazz19,1618916214301900,odd test event,0,1,
    2021-04-20T13:56:54.302064,jazz19,1618916214302064,even test event,0,2,
    2021-04-20T13:56:54.302064,jazz19,1618916214302064,odd test event,0,3,
    2021-04-20T13:56:54.302252,jazz19,1618916214302252,even test event,0,4,
    2021-04-20T13:56:54.302252,jazz19,1618916214302252,odd test event,0,5,
    2021-04-20T13:56:54.302621,jazz19,1618916214302621,even test event,0,6,
    2021-04-20T13:56:54.302621,jazz19,1618916214302621,odd test event,0,7,
    2021-04-20T13:56:54.302915,jazz19,1618916214302915,even test event,0,8,
    2021-04-20T13:56:54.302915,jazz19,1618916214302915,odd test event,0,9,
    2021-04-20T13:56:54.303183,jazz19,1618916214303183,even test event,0,10,
    2021-04-20T13:56:54.303183,jazz19,1618916214303183,odd test event,0,11,
    2021-04-20T13:56:54.303639,jazz19,1618916214303639,even test event,0,12,
    2021-04-20T13:56:54.303639,jazz19,1618916214303639,odd test event,0,13,
    2021-04-20T13:56:54.304121,jazz19,1618916214304121,even test event,0,14,
    2021-04-20T13:56:54.304121,jazz19,1618916214304121,odd test event,0,15,
    2021-04-20T13:56:54.304438,jazz19,1618916214304438,even test event,0,16,
    2021-04-20T13:56:54.304438,jazz19,1618916214304438,odd test event,0,17,
    2021-04-20T13:56:54.304928,jazz19,1618916214304928,even test event,0,18,
    2021-04-20T13:56:54.304928,jazz19,1618916214304928,odd test event,0,19,
~~~

## Events provider
This section contains an overview of steps that are required for implementation of the new provider.
See example_event_provider/example_provider.c with more tips.

To implement new event provider do the following steps:
1. inherit *clx_event_provider_t* structure from Collectx/src/events/event_provider.h:
~~~
    #include "/events/event_provider.h"

    typedef struct example_event_provider_t {
        // put the base provider to inheritate callbacks
        clx_event_provider_t    base;

        // mandatory fields to find the correct schema
        clx_schema_t*     schema;
        uint8_t           schema_index;
        uint8_t           type_index;

        // implementation specific fields
        uint32_t          pending_events;
        example_event_t*  events_cache[NUM_EVENTS];

        // parameters that can be set from clx_config.ini
        int               example_param1;
        int               example_param2;
    } test_event_provider_t;
~~~
2. Create event type/types. It should be consistent with the corresponding schema type/types.
~~~
    typedef struct __attribute__((packed)) example_event_t {
        clx_timestamp_t         timestamp;
        uint16_t                source_index;
        char                    name[16];
        uint64_t                value;
    } example_event_t;
~~~


2. implement the following callback functions. Consider that each provider can have several data sources with their configs/progress function:

    | Function | When it is called | Description |
    | -------- | ----------------- | ----------- |
    | `clx_progress_status_t (* progress)(struct clx_event_provider_t* provider, uint16_t source_index, clx_data_serializer_t* ds)` | Called according to clx sample rate | Main routine for putting collected event data from plugin internal buffer to clx data buffer, by calling  `clx_data_serializer_write_events(ds, ... )`. Return: <ul><li>CLX_PROGRESS_ERROR on any error, </li><li> CLX_PROGRESS_SUCCESS when all events have been successfully written, </li><li> CLX_PROGRESS_MORE_DATA if number of written events is less that expected. clx will call progress function again immediately to read the rest of the events. Consider not to overwrite pending events </li><li> CLX_PROGRESS_FORCE_SYNC to force provider to request urgent data exchange. </ul> |
    |INITIALIZATION CALLBACKS|||
    | `bool (* initialize)(struct clx_event_provider_t* provider, clx_type_system_t* ts)` | Once (on clx start) | Cast `provider` into the provider type. Use type system `ts` to create the provider schema and add types to it. |
    | `clx_source_array_t* (* get_sources)(struct clx_event_provider_t* provider)` | Once (after `initialize` call) | Return array of sources. Source is a pair of index and tag. One provider can have several sources |
    | `bool (* start)(struct clx_event_provider_t* provider)`    | Once (after `initialize` call) | Return true or start new thread which will keep the data updated and ready for progress function calls. |
    | `bool (* get_source_config)(struct clx_event_provider_t* provider, uint16_t source_index, clx_event_provider_config_t* const config)` | Once in the startap | Get configuration of source config. Use the default implementation that can be found in the example events plugin. |
    ||||
    |FINALIZATION CALLBACKS|||
    | `bool (* stop)(struct clx_event_provider_t* provider)`     | once before `finalize` call  | Return true or finalize provider thread which will  data for progress function|
    | `void (* finalize)(struct clx_event_provider_t* provider)` | Once (on clx cleanup) | Cast `provider` into the new provider type and cleanup memory |
    ||||
    |OPTIONAL CALLBACKS|||
    | `void (* get_configuration)(struct clx_event_provider_t* provider, command_response_t* response)` | On user request | Get the provider parameter configuration and construct the response to server |
    | `void (* command)(struct clx_event_provider_t* provider, const command_request_t* request, command_response_t* response)` | On user request| Process the request from agx server and send the response to the agx server. Use to change provider configuration. |

3. Create constructor  that will initialize all the callbacks
~~~
    static example_event_provider_t example_event_provider = {
        .base = {
            .name                  = "example_events",
            .description           = "Event provider example",
            .version               = {{1, 0, 0}},
            .initialize            = &example_event_provider_initialize,
            .start                 = &example_event_provider_start,
            .progress              = &example_event_provider_progress,
            .stop                  = &example_event_provider_stop,
            .finalize              = &example_event_provider_finalize,
            .get_sources           = &example_event_provider_get_sources,
            .get_source_config     = &example_event_provider_get_source_config,

            //  optional callbacks for clx server:
            // .get_configuration  = &example_event_provider_get_configuration;
            // .command            = &example_event_provider_command;
        },
        .pending_events = 0,
        .custom_option_1 = INIT_VAL_1;
        .custom_option_2 = INIT_VAL_2;

    };

    clx_event_provider_t* construct_event_provider(const clx_dict_t* opts) {
        // read parameters from app_opts. app_opts is the parsed clx_config.ini file
        // which can contain provider-specific key=value pairs
        example_event_provider.custom_option_1 = clx_dict_get(app_opts, "example_param_1_key");
        example_event_provider.custom_option_2 = clx_dict_get(app_opts, "example_param_2_key");

        return (clx_event_provider_t*) &test_event_provider;
    }
~~~



## Counters provider
Comparing to events, counters are more efficient in terms of querying collected data from binary files.
Counters are using the following type:
~~~
CLX_DATA_TYPE_UINT64,
CLX_DATA_TYPE_FP64,
CLX_DATA_TYPE_BIT64,
CLX_DATA_TYPE_INT64,
CLX_DATA_TYPE_STRING,
~~~


### General explanation
Counter provider contains one or several components and implements counter group callbacks.

- Components:
    A provider contains all available components. Components may be enabled/disabled via `clx_config.ini`.
    Component contains several counters of type `clx_counter_info_t`, not the data.
    Components include all available counters to collect.
- Counter group:
    Counter group is reassembled from enabled components as follows:
    - Make a list of all the names from enabled components
    - If name filtering is enabled, add counters with matching names to the group.
    - With no filtering enabled, add all counters from the list to the group.
    Counter groups for all counters providers will be converted to a single counters schema.
    This process is handled by clx collector outside of the provider.

Even though the counter group is created outside of the provider, the user can modify group callbacks if the implementation-specific structures should be updated.

### Example
In the example `example_counter_provider/example_provider.c`, the generic data is stored to `example_counters_data_t generic_data`.
Generic data has 3 fields: 2 fields of type `CLX_DATA_TYPE_UINT64` and one field of type `CLX_DATA_TYPE_STRING`.
Example component has 3 fields to fit generic data.
`comp_idx_to_ptr` and `group_idx_to_ptr` contain list of adresses to generic data.
On the call of `group.read()` callback data is reading the generic data from `group_idx_to_ptr` into `clx_counter_value_t *values` array.
Data is being read according to its type. Generic data is being updated on each read call, while in real-world applications
data can be updated by other processes. For instance, sysfs provider `.read` callback reads counter data from files, which are updated elsewhere.


### Implementation:
To implement counters provider:
1. Create a custom provider context:
~~~
    typedef struct example_provider_context_t{
        void*                   comp_idx_to_ptr[NUM_EXAMPLE_COUNTERS];   // for all available counters
        void*                   group_idx_to_ptr[NUM_EXAMPLE_COUNTERS];  // for counters selected to group
        example_counters_data_t generic_data;                            // placeholder for example generic values
        int                     num_reads;                               // number of .read calls
        bool                    example_boolean_param;                   // to demonstrate .command callback
    } example_provider_context_t;
~~~

2. Optionally, wrap group `clx_counter_group_t` to a custom structure and add fields if needed on the group level to access them from the group callbacks:
~~~
    typedef struct example_group_t {
        clx_counter_group_t  base;
    } example_group_t;
~~~


3. Inherit new provider from `clx_provider_t`. Put `example_provider_context_t` under extra field. Setup callbacks. Call `.initialize` callback from `construct_counter_provider()` function which is the constructor for counters provider:
~~~
    static clx_provider_t example_provider = {
        .name               = "example",
        .description        = "Counter provider example",
        .version            = 0x010000,
        .num_components     = 0,
        .components         = NULL,                                /// will be allocated in .initialize callback
        .extra              = NULL,                                /// use for provider specific fields (in .initialize callback)
        .initialize         = &example_init_provider,              /// implement (alloc structs and components)
        .finalize           = &example_finalize_provider,          /// implement (free structs)
        .command            = &example_provider_handle_command,    /// optional

        /// group provider callbacks
        .group.read         = &example_read_counter_group_values,  /// implement according to counters types in group
        .group.add_counter  = &example_add_counter_to_group,       /// implement if provider need to interact with provider.extra fields
        .group.create       = &example_create_counter_group,       /// ignore or re-implement if group was customized
        .group.destroy      = &example_destroy_counter_group,      /// ignore or re-implement if group was customized
        .group.start        = &example_start_counting_group,       /// ignore (returns true)
        .group.stop         = &example_stop_counting_group,        /// ignore (returns true)
    };

    clx_provider_t* construct_counter_provider(const clx_dict_t* opts) {
        example_init_provider(&example_provider, opts);
        return &example_provider;
    }
~~~

4. Implement provider callbacks:

    | Function | When it is called | Description |
    | -------- | ----------------- | ----------- |
    | `bool (*initialize)(struct clx_provider_t*, const struct clx_options_t* app_opts)` | Once (on clx start) | Implement `.initialize` provider callback. Allocate provider-specific structures and create of one or several components. Provider components are accosiated with counter groups. Components can be turned on/off from user defuned options `app_opts`. To check if component is enabled use `clx_comp_is_enabled()`.|
    | `void (*finalize)(struct clx_provider_t*)` | Once (on clx cleanup) | Cleanup memory if needed. |
    ||||
    |OPTIONAL CALLBACKS|||
    | `void (*get_configuration)(struct clx_provider_t* provider, command_response_t* response)` | On user request | Get the provider parameter configuration and construct the response to server. |
    | `void (*command)(struct clx_provider_t* provider, const command_request_t* request, command_response_t* response)` | On user request| Process the request from agx server (clxcli) and send the response to the agx server. Use to change provider configuration. Use to change provider configuration, e.g. to enable/disable components. Note that clxcli should be updated with new commands.|
    | | | |

5. Re-implement group callbacks if needed based on example functions.

Use the `clx_plugin_tester` to run and generate data.
