// SPDX-License-Identifier: GPL-2.0 or BSD-3-Clause

#ifndef __MLXBF_PMC_H__
#define __MLXBF_PMC_H__

#define MLNX_WRITE_REG_32		(0x82000009)
#define MLNX_READ_REG_32		(0x8200000A)
#define MLNX_WRITE_REG_64		(0x8200000B)
#define MLNX_READ_REG_64		(0x8200000C)
#define MLNX_SIP_SVC_UID		(0x8200ff01)
#define MLNX_SIP_SVC_VERSION		(0x8200ff03)

#define SMCCC_INVALID_PARAMETERS	(-2)
#define SMCCC_OUT_OF_RANGE		(-3)
#define SMCCC_ACCESS_VIOLATION		(-4)

#define MLNX_EVENT_SET_BF1	0
#define MLNX_EVENT_SET_BF2	1
#define MLNX_EVENT_SET_BF3	2

#define MLNX_PMC_SVC_REQ_MAJOR 0
#define MLNX_PMC_SVC_MIN_MINOR 3

#define MLXBF_PMC_MAX_BLOCKS		30

/**
 * Structure to hold info for each HW block
 *
 * @mmio_base: The VA at which the PMC block is mapped
 * @blk_size: Size of each mapped region
 * @counters: Number of counters in the block
 * @type: Type of counters in the block
 * @block_dir: Kobjects to create sub-directories
 * @attr_counter: Attributes for "counter" sysfs files
 * @attr_event: Attributes for "event" sysfs files
 * @attr_event_list: Attributes for "event_list" sysfs files
 * @attr_enable: Attributes for "enable" sysfs files
 * @sysfs_event_cnt: Number of sysfs event files in the block
 */
struct mlxbf_pmc_block_info {
	void *mmio_base;
	size_t blk_size;
	size_t counters;
	int type;
	struct kobject *block_dir;
	struct kobj_attribute *attr_counter;
	struct kobj_attribute *attr_event;
	struct kobj_attribute attr_event_list;
	struct kobj_attribute attr_enable;
	int sysfs_event_cnt;
};

/**
 * Structure to hold PMC context info
 *
 * @pdev: The kernel structure representing the device
 * @total_blocks: Total number of blocks
 * @tile_count: Number of tiles in the system
 * @llt_enable: Info on enabled LLTs
 * @mss_enable: Info on enabled MSSs
 * @hwmon_dev: Hwmon device for bfperf
 * @ko: Kobject for bfperf
 * @block_name: Block name
 * @block_name:  Block info
 * @sv_sreg_support: Whether SMCs are used to access performance registers
 * @sreg_tbl_perf: Secure register access table number
 * @event_set: Event set to use
 */
struct mlxbf_pmc_context {
	struct platform_device *pdev;
	uint32_t total_blocks;
	uint32_t tile_count;
	uint8_t llt_enable;
	uint8_t mss_enable;
	struct device *hwmon_dev;
	struct kobject *ko;
	const char *block_name[MLXBF_PMC_MAX_BLOCKS];
	struct mlxbf_pmc_block_info block[MLXBF_PMC_MAX_BLOCKS];
	bool svc_sreg_support;
	uint32_t sreg_tbl_perf;
	unsigned int event_set;
};

#define MLXBF_PERFTYPE_REGISTER	0
#define MLXBF_PERFTYPE_COUNTER	1
#define MLXBF_PERFTYPE_CRSPACE	2

#define MLXBF_PERFCTL		0
#define MLXBF_PERFEVT		1
#define MLXBF_PERFVALEXT	2
#define MLXBF_PERFACC0		4
#define MLXBF_PERFACC1		5
#define MLXBF_PERFMVAL0		6
#define MLXBF_PERFMVAL1		7

#define MLXBF_GEN_PERFMON_CONFIG__WR_R_B	BIT(0)
#define MLXBF_GEN_PERFMON_CONFIG__STROBE	BIT(1)
#define MLXBF_GEN_PERFMON_CONFIG__ADDR		GENMASK_ULL(4, 2)
#define MLXBF_GEN_PERFMON_CONFIG__WDATA		GENMASK_ULL(60, 5)

#define MLXBF_GEN_PERFCTL__FM1			GENMASK_ULL(2, 0)
#define MLXBF_GEN_PERFCTL__MS1			GENMASK_ULL(5, 4)
#define MLXBF_GEN_PERFCTL__ACCM1		GENMASK_ULL(10, 8)
#define MLXBF_GEN_PERFCTL__AD1			BIT(11)
#define MLXBF_GEN_PERFCTL__ETRIG1		GENMASK_ULL(13, 12)
#define MLXBF_GEN_PERFCTL__EB1			BIT(14)
#define MLXBF_GEN_PERFCTL__EN1			BIT(15)
#define MLXBF_GEN_PERFCTL__FM0			GENMASK_ULL(18, 16)
#define MLXBF_GEN_PERFCTL__MS0			GENMASK_ULL(21, 20)
#define MLXBF_GEN_PERFCTL__ACCM0		GENMASK_ULL(26, 24)
#define MLXBF_GEN_PERFCTL__AD0			BIT(27)
#define MLXBF_GEN_PERFCTL__ETRIG0		GENMASK_ULL(29, 28)
#define MLXBF_GEN_PERFCTL__EB0			BIT(30)
#define MLXBF_GEN_PERFCTL__EN0			BIT(31)

#define MLXBF_GEN_PERFEVT__PVALSEL		GENMASK_ULL(19, 16)
#define MLXBF_GEN_PERFEVT__MODSEL		GENMASK_ULL(23, 20)
#define MLXBF_GEN_PERFEVT__EVTSEL		GENMASK_ULL(31, 24)

#define MLXBF_L3C_PERF_CNT_CFG			0x0
#define MLXBF_L3C_PERF_CNT_CFG_1		0x4
#define MLXBF_L3C_PERF_CNT_CFG_2		0x8
#define MLXBF_L3C_PERF_CNT_SEL			0x10
#define MLXBF_L3C_PERF_CNT_SEL_1		0x14
#define MLXBF_L3C_PERF_CNT_LOW			0x40
#define MLXBF_L3C_PERF_CNT_HIGH			0x60

#define MLXBF_L3C_PERF_CNT_CFG__EN		BIT(0)
#define MLXBF_L3C_PERF_CNT_CFG__RST		BIT(1)
#define MLXBF_L3C_PERF_CNT_CFG__SRCID_SEL	GENMASK(14, 8)
#define MLXBF_L3C_PERF_CNT_CFG__SRCID_MASK	GENMASK(22, 16)
#define MLXBF_L3C_PERF_CNT_CFG__PRF_SEL		GENMASK(27, 24)
#define MLXBF_L3C_PERF_CNT_CFG__PRF_MASK	GENMASK(31, 28)

#define MLXBF_L3C_PERF_CNT_CFG_1__SET_SEL	GENMASK(10,0)
#define MLXBF_L3C_PERF_CNT_CFG_1__SET_MASK	GENMASK(22,12)
#define MLXBF_L3C_PERF_CNT_CFG_1__EMEM_USAGE_TH	GENMASK(30, 24)

#define MLXBF_L3C_PERF_CNT_CFG_2__STRM_SEL	GENMASK(7, 0)
#define MLXBF_L3C_PERF_CNT_CFG_2__STRM_MASK	GENMASK(15, 8)

#define MLXBF_L3C_PERF_CNT_SEL__CNT_0		GENMASK(5, 0)
#define MLXBF_L3C_PERF_CNT_SEL__CNT_1		GENMASK(13, 8)
#define MLXBF_L3C_PERF_CNT_SEL__CNT_2		GENMASK(21, 16)
#define MLXBF_L3C_PERF_CNT_SEL__CNT_3		GENMASK(29, 24)

#define MLXBF_L3C_PERF_CNT_SEL_1__CNT_4		GENMASK(5, 0)

#define MLXBF_L3C_PERF_CNT_LOW__VAL		GENMASK(31, 0)
#define MLXBF_L3C_PERF_CNT_HIGH__VAL		GENMASK(24, 0)

#define MLXBF_CRSPACE_PERFMON_REG0		0x0
#define MLXBF_CRSPACE_PERFSEL0			GENMASK(23, 16)
#define MLXBF_CRSPACE_PERFSEL1			GENMASK(7, 0)
#define MLXBF_CRSPACE_PERFMON_CTL(n)		(n * 2)
#define MLXBF_CRSPACE_PERFMON_EN		BIT(30)
#define MLXBF_CRSPACE_PERFMON_CLR		BIT(28)
#define MLXBF_CRSPACE_PERFMON_VAL0(n)		(MLXBF_CRSPACE_PERFMON_CTL(n) + 0xc)

struct mlxbf_pmc_events {
	uint32_t evt_num;
	char *evt_name;
};

struct mlxbf_pmc_events mlxbf_pcie_events[] = {
{0x0,  "IN_P_PKT_CNT"},
{0x10, "IN_NP_PKT_CNT"},
{0x18, "IN_C_PKT_CNT"},
{0x20, "OUT_P_PKT_CNT"},
{0x28, "OUT_NP_PKT_CNT"},
{0x30, "OUT_C_PKT_CNT"},
{0x38, "IN_P_BYTE_CNT"},
{0x40, "IN_NP_BYTE_CNT"},
{0x48, "IN_C_BYTE_CNT"},
{0x50, "OUT_P_BYTE_CNT"},
{0x58, "OUT_NP_BYTE_CNT"},
{0x60, "OUT_C_BYTE_CNT"},
{-1, NULL}
};

struct mlxbf_pmc_events mlxbf_smgen_events[] = {
{0x0,  "AW_REQ"},
{0x1,  "AW_BEATS"},
{0x2,  "AW_TRANS"},
{0x3,  "AW_RESP"},
{0x4,  "AW_STL"},
{0x5,  "AW_LAT"},
{0x6,  "AW_REQ_TBU"},
{0x8,  "AR_REQ"},
{0x9,  "AR_BEATS"},
{0xa,  "AR_TRANS"},
{0xb,  "AR_STL"},
{0xc,  "AR_LAT"},
{0xd,  "AR_REQ_TBU"},
{0xe,  "TBU_MISS"},
{0xf,  "TX_DAT_AF"},
{0x10, "RX_DAT_AF"},
{0x11, "RETRYQ_CRED"},
{-1, NULL}
};

struct mlxbf_pmc_events mlxbf1_trio_events[] = {
{0x00, "DISABLE"},
{0xa0, "TPIO_DATA_BEAT"},
{0xa1, "TDMA_DATA_BEAT"},
{0xa2, "MAP_DATA_BEAT"},
{0xa3, "TXMSG_DATA_BEAT"},
{0xa4, "TPIO_DATA_PACKET"},
{0xa5, "TDMA_DATA_PACKET"},
{0xa6, "MAP_DATA_PACKET"},
{0xa7, "TXMSG_DATA_PACKET"},
{0xa8, "TDMA_RT_AF"},
{0xa9, "TDMA_PBUF_MAC_AF"},
{0xaa, "TRIO_MAP_WRQ_BUF_EMPTY"},
{0xab, "TRIO_MAP_CPL_BUF_EMPTY"},
{0xac, "TRIO_MAP_RDQ0_BUF_EMPTY"},
{0xad, "TRIO_MAP_RDQ1_BUF_EMPTY"},
{0xae, "TRIO_MAP_RDQ2_BUF_EMPTY"},
{0xaf, "TRIO_MAP_RDQ3_BUF_EMPTY"},
{0xb0, "TRIO_MAP_RDQ4_BUF_EMPTY"},
{0xb1, "TRIO_MAP_RDQ5_BUF_EMPTY"},
{0xb2, "TRIO_MAP_RDQ6_BUF_EMPTY"},
{0xb3, "TRIO_MAP_RDQ7_BUF_EMPTY"},
{-1, NULL}
};

struct mlxbf_pmc_events mlxbf2_trio_events[] = {
{0x00, "DISABLE"},
{0xa0, "TPIO_DATA_BEAT"},
{0xa1, "TDMA_DATA_BEAT"},
{0xa2, "MAP_DATA_BEAT"},
{0xa3, "TXMSG_DATA_BEAT"},
{0xa4, "TPIO_DATA_PACKET"},
{0xa5, "TDMA_DATA_PACKET"},
{0xa6, "MAP_DATA_PACKET"},
{0xa7, "TXMSG_DATA_PACKET"},
{0xa8, "TDMA_RT_AF"},
{0xa9, "TDMA_PBUF_MAC_AF"},
{0xaa, "TRIO_MAP_WRQ_BUF_EMPTY"},
{0xab, "TRIO_MAP_CPL_BUF_EMPTY"},
{0xac, "TRIO_MAP_RDQ0_BUF_EMPTY"},
{0xad, "TRIO_MAP_RDQ1_BUF_EMPTY"},
{0xae, "TRIO_MAP_RDQ2_BUF_EMPTY"},
{0xaf, "TRIO_MAP_RDQ3_BUF_EMPTY"},
{0xb0, "TRIO_MAP_RDQ4_BUF_EMPTY"},
{0xb1, "TRIO_MAP_RDQ5_BUF_EMPTY"},
{0xb2, "TRIO_MAP_RDQ6_BUF_EMPTY"},
{0xb3, "TRIO_MAP_RDQ7_BUF_EMPTY"},
{0xb4, "TRIO_RING_TX_FLIT_CH0"},
{0xb5, "TRIO_RING_TX_FLIT_CH1"},
{0xb6, "TRIO_RING_TX_FLIT_CH2"},
{0xb7, "TRIO_RING_TX_FLIT_CH3"},
{0xb8, "TRIO_RING_TX_FLIT_CH4"},
{0xb9, "TRIO_RING_RX_FLIT_CH0"},
{0xba, "TRIO_RING_RX_FLIT_CH1"},
{0xbb, "TRIO_RING_RX_FLIT_CH2"},
{0xbc, "TRIO_RING_RX_FLIT_CH3"},
{-1, NULL}
};

struct mlxbf_pmc_events mlxbf_ecc_events[] = {
{0x00, "DISABLE"},
{0x100, "ECC_SINGLE_ERROR_CNT"},
{0x104, "ECC_DOUBLE_ERROR_CNT"},
{0x114, "SERR_INJ"},
{0x118, "DERR_INJ"},
{0x124, "ECC_SINGLE_ERROR_0"},
{0x164, "ECC_DOUBLE_ERROR_0"},
{0x340, "DRAM_ECC_COUNT"},
{0x344, "DRAM_ECC_INJECT"},
{0x348, "DRAM_ECC_ERROR",},
{-1, NULL}
};

struct mlxbf_pmc_events mlxbf_mss_events[] = {
{0x00, "DISABLE"},
{0xc0, "RXREQ_MSS"},
{0xc1, "RXDAT_MSS"},
{0xc2, "TXRSP_MSS"},
{0xc3, "TXDAT_MSS"},
{-1, NULL}
};

struct mlxbf_pmc_events mlxbf_hnf_events[] = {
{0x00, "DISABLE"},
{0x45, "HNF_REQUESTS"},
{0x46, "HNF_REJECTS"},
{0x47, "ALL_BUSY"},
{0x48, "MAF_BUSY"},
{0x49, "MAF_REQUESTS"},
{0x4a, "RNF_REQUESTS"},
{0x4b, "REQUEST_TYPE"},
{0x4c, "MEMORY_READS"},
{0x4d, "MEMORY_WRITES"},
{0x4e, "VICTIM_WRITE"},
{0x4f, "POC_FULL"},
{0x50, "POC_FAIL"},
{0x51, "POC_SUCCESS"},
{0x52, "POC_WRITES"},
{0x53, "POC_READS"},
{0x54, "FORWARD"},
{0x55, "RXREQ_HNF"},
{0x56, "RXRSP_HNF"},
{0x57, "RXDAT_HNF"},
{0x58, "TXREQ_HNF"},
{0x59, "TXRSP_HNF"},
{0x5a, "TXDAT_HNF"},
{0x5b, "TXSNP_HNF"},
{0x5c, "INDEX_MATCH"},
{0x5d, "A72_ACCESS"},
{0x5e, "IO_ACCESS"},
{0x5f, "TSO_WRITE"},
{0x60, "TSO_CONFLICT"},
{0x61, "DIR_HIT"},
{0x62, "HNF_ACCEPTS"},
{0x63, "REQ_BUF_EMPTY"},
{0x64, "REQ_BUF_IDLE_MAF"},
{0x65, "TSO_NOARB"},
{0x66, "TSO_NOARB_CYCLES"},
{0x67, "MSS_NO_CREDIT"},
{0x68, "TXDAT_NO_LCRD"},
{0x69, "TXSNP_NO_LCRD"},
{0x6a, "TXRSP_NO_LCRD"},
{0x6b, "TXREQ_NO_LCRD"},
{0x6c, "TSO_CL_MATCH"},
{0x6d, "MEMORY_READS_BYPASS"},
{0x6e, "TSO_NOARB_TIMEOUT"},
{0x6f, "ALLOCATE"},
{0x70, "VICTIM"},
{0x71, "A72_WRITE"},
{0x72, "A72_Read"},
{0x73, "IO_WRITE"},
{0x74, "IO_READ"},
{0x75, "TSO_REJECT"},
{0x80, "TXREQ_RN"},
{0x81, "TXRSP_RN"},
{0x82, "TXDAT_RN"},
{0x83, "RXSNP_RN"},
{0x84, "RXRSP_RN"},
{0x85, "RXDAT_RN"},
{-1, NULL}
};

struct mlxbf_pmc_events mlxbf2_hnfnet_events[] = {
{0x00, "DISABLE"},
{0x12, "CDN_REQ"},
{0x13, "DDN_REQ"},
{0x14, "NDN_REQ"},
{0x15, "CDN_DIAG_N_OUT_OF_CRED"},
{0x16, "CDN_DIAG_S_OUT_OF_CRED"},
{0x17, "CDN_DIAG_E_OUT_OF_CRED"},
{0x18, "CDN_DIAG_W_OUT_OF_CRED"},
{0x19, "CDN_DIAG_C_OUT_OF_CRED"},
{0x1a, "CDN_DIAG_N_EGRESS"},
{0x1b, "CDN_DIAG_S_EGRESS"},
{0x1c, "CDN_DIAG_E_EGRESS"},
{0x1d, "CDN_DIAG_W_EGRESS"},
{0x1e, "CDN_DIAG_C_EGRESS"},
{0x1f, "CDN_DIAG_N_INGRESS"},
{0x20, "CDN_DIAG_S_INGRESS"},
{0x21, "CDN_DIAG_E_INGRESS"},
{0x22, "CDN_DIAG_W_INGRESS"},
{0x23, "CDN_DIAG_C_INGRESS"},
{0x24, "CDN_DIAG_CORE_SENT"},
{0x25, "DDN_DIAG_N_OUT_OF_CRED"},
{0x26, "DDN_DIAG_S_OUT_OF_CRED"},
{0x27, "DDN_DIAG_E_OUT_OF_CRED"},
{0x28, "DDN_DIAG_W_OUT_OF_CRED"},
{0x29, "DDN_DIAG_C_OUT_OF_CRED"},
{0x2a, "DDN_DIAG_N_EGRESS"},
{0x2b, "DDN_DIAG_S_EGRESS"},
{0x2c, "DDN_DIAG_E_EGRESS"},
{0x2d, "DDN_DIAG_W_EGRESS"},
{0x2e, "DDN_DIAG_C_EGRESS"},
{0x2f, "DDN_DIAG_N_INGRESS"},
{0x30, "DDN_DIAG_S_INGRESS"},
{0x31, "DDN_DIAG_E_INGRESS"},
{0x32, "DDN_DIAG_W_INGRESS"},
{0x33, "DDN_DIAG_C_INGRESS"},
{0x34, "DDN_DIAG_CORE_SENT"},
{0x35, "NDN_DIAG_N_OUT_OF_CRED"},
{0x36, "NDN_DIAG_S_OUT_OF_CRED"},
{0x37, "NDN_DIAG_E_OUT_OF_CRED"},
{0x38, "NDN_DIAG_W_OUT_OF_CRED"},
{0x39, "NDN_DIAG_C_OUT_OF_CRED"},
{0x3a, "NDN_DIAG_N_EGRESS"},
{0x3b, "NDN_DIAG_S_EGRESS"},
{0x3c, "NDN_DIAG_E_EGRESS"},
{0x3d, "NDN_DIAG_W_EGRESS"},
{0x3e, "NDN_DIAG_C_EGRESS"},
{0x3f, "NDN_DIAG_N_INGRESS"},
{0x40, "NDN_DIAG_S_INGRESS"},
{0x41, "NDN_DIAG_E_INGRESS"},
{0x42, "NDN_DIAG_W_INGRESS"},
{0x43, "NDN_DIAG_C_INGRESS"},
{0x44, "NDN_DIAG_CORE_SENT"},
{-1, NULL}
};

struct mlxbf_pmc_events mlxbf_l3cache_events[] = {
{0x00, "DISABLE"},
{0x01, "CYCLES"},
{0x02, "TOTAL_RD_REQ_IN"},
{0x03, "TOTAL_WR_REQ_IN"},
{0x04, "TOTAL_WR_DBID_ACK"},
{0x05, "TOTAL_WR_DATA_IN"},
{0x06, "TOTAL_WR_COMP"},
{0x07, "TOTAL_RD_DATA_OUT"},
{0x08, "TOTAL_CDN_REQ_IN_BANK0"},
{0x09, "TOTAL_CDN_REQ_IN_BANK1"},
{0x0a, "TOTAL_DDN_REQ_IN_BANK0"},
{0x0b, "TOTAL_DDN_REQ_IN_BANK1"},
{0x0c, "TOTAL_EMEM_RD_RES_IN_BANK0"},
{0x0d, "TOTAL_EMEM_RD_RES_IN_BANK1"},
{0x0e, "TOTAL_CACHE_RD_RES_IN_BANK0"},
{0x0f, "TOTAL_CACHE_RD_RES_IN_BANK1"},
{0x10, "TOTAL_EMEM_RD_REQ_BANK0"},
{0x11, "TOTAL_EMEM_RD_REQ_BANK1"},
{0x12, "TOTAL_EMEM_WR_REQ_BANK0"},
{0x13, "TOTAL_EMEM_WR_REQ_BANK1"},
{0x14, "TOTAL_RD_REQ_OUT"},
{0x15, "TOTAL_WR_REQ_OUT"},
{0x16, "TOTAL_RD_RES_IN"},
{0x17, "HITS_BANK0"},
{0x18, "HITS_BANK1"},
{0x19, "MISSES_BANK0"},
{0x1a, "MISSES_BANK1"},
{0x1b, "ALLOCATIONS_BANK0"},
{0x1c, "ALLOCATIONS_BANK1"},
{0x1d, "EVICTIONS_BANK0"},
{0x1e, "EVICTIONS_BANK1"},
{0x1f, "DBID_REJECT"},
{0x20, "WRDB_REJECT_BANK0"},
{0x21, "WRDB_REJECT_BANK1"},
{0x22, "CMDQ_REJECT_BANK0"},
{0x23, "CMDQ_REJECT_BANK1"},
{0x24, "COB_REJECT_BANK0"},
{0x25, "COB_REJECT_BANK1"},
{0x26, "TRB_REJECT_BANK0"},
{0x27, "TRB_REJECT_BANK1"},
{0x28, "TAG_REJECT_BANK0"},
{0x29, "TAG_REJECT_BANK1"},
{0x2a, "ANY_REJECT_BANK0"},
{0x2b, "ANY_REJECT_BANK1"},
{-1, NULL}
};

struct mlxbf_pmc_events mlxbf3_llt_events[] = {
{0, "HNF0_CYCLES"},
{1, "HNF0_REQS_RECEIVED"},
{2, "HNF0_REQS_PROCESSED"},
{3, "HNF0_DIR_HIT"},
{4, "HNF0_DIR_MISS"},
{5, "HNF0_DIR_RD_ALLOC"},
{6, "HNF0_DIR_WR_ALLOC"},
{7, "HNF0_DIR_VICTIM"},
{8, "HNF0_CL_HAZARD"},
{9, "HNF0_ALL_HAZARD"},
{10, "HNF0_PIPE_STALLS"},
{11, "HNF0_MEM_READS"},
{12, "HNF0_MEM_WRITES"},
{13, "HNF0_MEM_ACCESS"},
{14, "HNF0_DCL_READ"},
{15, "HNF0_DCL_INVAL"},
{16, "HNF0_CHI_RXDAT"},
{17, "HNF0_CHI_RXRSP"},
{18, "HNF0_CHI_TXDAT"},
{19, "HNF0_CHI_TXRSP"},
{20, "HNF0_CHI_TXSNP"},
{21, "HNF0_DCT_SNP"},
{22, "HNF0_SNP_FWD_DATA"},
{23, "HNF0_SNP_FWD_RSP"},
{24, "HNF0_SNP_RSP"},
{25, "HNF0_EXCL_FULL"},
{26, "HNF0_EXCL_WRITE_F"},
{27, "HNF0_EXCL_WRITE_S"},
{28, "HNF0_EXCL_WRITE"},
{29, "HNF0_EXCL_READ"},
{30, "HNF0_REQ_BUF_EMPTY"},
{31, "HNF0_ALL_MAFS_BUSY"},
{32, "HNF0_TXDAT_NO_LCRD"},
{33, "HNF0_TXSNP_NO_LCRD"},
{34, "HNF0_TXRSP_NO_LCRD"},
{35, "HNF0_TXREQ_NO_LCRD"},
{36, "HNF0_WRITE"},
{37, "HNF0_READ"},
{38, "HNF0_ACCESS"},
{39, "HNF0_MAF_N_BUSY"},
{40, "HNF0_MAF_N_REQS"},
{41, "HNF0_SEL_OPCODE"},
{42, "HNF1_CYCLES"},
{43, "HNF1_REQS_RECEIVED"},
{44, "HNF1_REQS_PROCESSED"},
{45, "HNF1_DIR_HIT"},
{46, "HNF1_DIR_MISS"},
{47, "HNF1_DIR_RD_ALLOC"},
{48, "HNF1_DIR_WR_ALLOC"},
{49, "HNF1_DIR_VICTIM"},
{50, "HNF1_CL_HAZARD"},
{51, "HNF1_ALL_HAZARD"},
{52, "HNF1_PIPE_STALLS"},
{53, "HNF1_MEM_READS"},
{54, "HNF1_MEM_WRITES"},
{55, "HNF1_MEM_ACCESS"},
{56, "HNF1_DCL_READ"},
{57, "HNF1_DCL_INVAL"},
{58, "HNF1_CHI_RXDAT"},
{59, "HNF1_CHI_RXRSP"},
{60, "HNF1_CHI_TXDAT"},
{61, "HNF1_CHI_TXRSP"},
{62, "HNF1_CHI_TXSNP"},
{63, "HNF1_DCT_SNP"},
{64, "HNF1_SNP_FWD_DATA"},
{65, "HNF1_SNP_FWD_RSP"},
{66, "HNF1_SNP_RSP"},
{67, "HNF1_EXCL_FULL"},
{68, "HNF1_EXCL_WRITE_F"},
{69, "HNF1_EXCL_WRITE_S"},
{70, "HNF1_EXCL_WRITE"},
{71, "HNF1_EXCL_READ"},
{72, "HNF1_REQ_BUF_EMPTY"},
{73, "HNF1_ALL_MAFS_BUSY"},
{74, "HNF1_TXDAT_NO_LCRD"},
{75, "HNF1_TXSNP_NO_LCRD"},
{76, "HNF1_TXRSP_NO_LCRD"},
{77, "HNF1_TXREQ_NO_LCRD"},
{78, "HNF1_WRITE"},
{79, "HNF1_READ"},
{80, "HNF1_ACCESS"},
{81, "HNF1_MAF_N_BUSY"},
{82, "HNF1_MAF_N_REQS"},
{83, "HNF1_SEL_OPCODE"},
{84, "GDC_BANK0_RD_REQ"},
{85, "GDC_BANK0_WR_REQ"},
{86, "GDC_BANK0_ALLOCATE"},
{87, "GDC_BANK0_HIT"},
{88, "GDC_BANK0_MISS"},
{89, "GDC_BANK0_INVALIDATE"},
{90, "GDC_BANK0_EVICT"},
{91, "GDC_BANK0_RD_RESP"},
{92, "GDC_BANK0_WR_ACK"},
{93, "GDC_BANK0_SNOOP"},
{94, "GDC_BANK0_SNOOP_NORMAL"},
{95, "GDC_BANK0_SNOOP_FWD"},
{96, "GDC_BANK0_SNOOP_STASH"},
{97, "GDC_BANK0_SNOOP_STASH_INDPND_RD"},
{98, "GDC_BANK0_FOLLOWER"},
{99, "GDC_BANK0_FW"},
{100, "GDC_BANK0_HIT_DCL_BOTH"},
{101, "GDC_BANK0_HIT_DCL_PARTIAL"},
{102, "GDC_BANK0_EVICT_DCL"},
{103, "GDC_BANK0_G_RSE_PIPE_CACHE_DATA0"},
{103, "GDC_BANK0_G_RSE_PIPE_CACHE_DATA1"},
{105, "GDC_BANK0_ARB_STRB"},
{106, "GDC_BANK0_ARB_WAIT"},
{107, "GDC_BANK0_GGA_STRB"},
{108, "GDC_BANK0_GGA_WAIT"},
{109, "GDC_BANK0_FW_STRB"},
{110, "GDC_BANK0_FW_WAIT"},
{111, "GDC_BANK0_SNP_STRB"},
{112, "GDC_BANK0_SNP_WAIT"},
{113, "GDC_BANK0_MISS_INARB_STRB"},
{114, "GDC_BANK0_MISS_INARB_WAIT"},
{115, "GDC_BANK0_G_FIFO_FF_GGA_RSP_RD0"},
{116, "GDC_BANK0_G_FIFO_FF_GGA_RSP_RD1"},
{117, "GDC_BANK0_G_FIFO_FF_GGA_RSP_RD2"},
{118, "GDC_BANK0_G_FIFO_FF_GGA_RSP_RD3"},
{119, "GDC_BANK0_G_FIFO_FF_GGA_RSP_WR0"},
{120, "GDC_BANK0_G_FIFO_FF_GGA_RSP_WR1"},
{121, "GDC_BANK0_G_FIFO_FF_GGA_RSP_WR2"},
{122, "GDC_BANK0_G_FIFO_FF_GGA_RSP_WR3"},
{123, "GDC_BANK1_RD_REQ"},
{124, "GDC_BANK1_WR_REQ"},
{125, "GDC_BANK1_ALLOCATE"},
{126, "GDC_BANK1_HIT"},
{127, "GDC_BANK1_MISS"},
{128, "GDC_BANK1_INVALIDATE"},
{129, "GDC_BANK1_EVICT"},
{130, "GDC_BANK1_RD_RESP"},
{131, "GDC_BANK1_WR_ACK"},
{132, "GDC_BANK1_SNOOP"},
{133, "GDC_BANK1_SNOOP_NORMAL"},
{134, "GDC_BANK1_SNOOP_FWD"},
{135, "GDC_BANK1_SNOOP_STASH"},
{136, "GDC_BANK1_SNOOP_STASH_INDPND_RD"},
{137, "GDC_BANK1_FOLLOWER"},
{138, "GDC_BANK1_FW"},
{139, "GDC_BANK1_HIT_DCL_BOTH"},
{140, "GDC_BANK1_HIT_DCL_PARTIAL"},
{141, "GDC_BANK1_EVICT_DCL"},
{142, "GDC_BANK1_G_RSE_PIPE_CACHE_DATA0"},
{143, "GDC_BANK1_G_RSE_PIPE_CACHE_DATA1"},
{144, "GDC_BANK1_ARB_STRB"},
{145, "GDC_BANK1_ARB_WAIT"},
{146, "GDC_BANK1_GGA_STRB"},
{147, "GDC_BANK1_GGA_WAIT"},
{148, "GDC_BANK1_FW_STRB"},
{149, "GDC_BANK1_FW_WAIT"},
{150, "GDC_BANK1_SNP_STRB"},
{151, "GDC_BANK1_SNP_WAIT"},
{152, "GDC_BANK1_MISS_INARB_STRB"},
{153, "GDC_BANK1_MISS_INARB_WAIT"},
{154, "GDC_BANK1_G_FIFO_FF_GGA_RSP_RD0"},
{155, "GDC_BANK1_G_FIFO_FF_GGA_RSP_RD1"},
{156, "GDC_BANK1_G_FIFO_FF_GGA_RSP_RD2"},
{157, "GDC_BANK1_G_FIFO_FF_GGA_RSP_RD3"},
{158, "GDC_BANK1_G_FIFO_FF_GGA_RSP_WR0"},
{159, "GDC_BANK1_G_FIFO_FF_GGA_RSP_WR1"},
{160, "GDC_BANK1_G_FIFO_FF_GGA_RSP_WR2"},
{161, "GDC_BANK1_G_FIFO_FF_GGA_RSP_WR3"},
{162, "HISTOGRAM_HISTOGRAM_BIN0"},
{163, "HISTOGRAM_HISTOGRAM_BIN1"},
{164, "HISTOGRAM_HISTOGRAM_BIN2"},
{165, "HISTOGRAM_HISTOGRAM_BIN3"},
{166, "HISTOGRAM_HISTOGRAM_BIN4"},
{167, "HISTOGRAM_HISTOGRAM_BIN5"},
{168, "HISTOGRAM_HISTOGRAM_BIN6"},
{169, "HISTOGRAM_HISTOGRAM_BIN7"},
{170, "HISTOGRAM_HISTOGRAM_BIN8"},
{171, "HISTOGRAM_HISTOGRAM_BIN9"},
{-1, NULL}
};

struct mlxbf_pmc_events mlxbf3_llt_miss_events[] = {
{0, "GDC_MISS_MACHINE_RD_REQ"},
{1, "GDC_MISS_MACHINE_WR_REQ"},
{2, "GDC_MISS_MACHINE_SNP_REQ"},
{3, "GDC_MISS_MACHINE_EVICT_REQ"},
{4, "GDC_MISS_MACHINE_FW_REQ"},
{5, "GDC_MISS_MACHINE_RD_RESP"},
{6, "GDC_MISS_MACHINE_WR_RESP"},
{7, "GDC_MISS_MACHINE_SNP_STASH_DATAPULL_DROP"},
{8, "GDC_MISS_MACHINE_SNP_STASH_DATAPULL_DROP_TXDAT"},
{9, "GDC_MISS_MACHINE_CHI_TXREQ"},
{10, "GDC_MISS_MACHINE_CHI_RXRSP"},
{11, "GDC_MISS_MACHINE_CHI_TXDAT"},
{12, "GDC_MISS_MACHINE_CHI_RXDAT"},
{13, "GDC_MISS_MACHINE_G_FIFO_FF_EXEC0_0"},
{14, "GDC_MISS_MACHINE_G_FIFO_FF_EXEC0_1 "},
{15, "GDC_MISS_MACHINE_G_FIFO_FF_EXEC0_2"},
{16, "GDC_MISS_MACHINE_G_FIFO_FF_EXEC0_3 "},
{17, "GDC_MISS_MACHINE_G_FIFO_FF_EXEC1_0 "},
{18, "GDC_MISS_MACHINE_G_FIFO_FF_EXEC1_1 "},
{19, "GDC_MISS_MACHINE_G_FIFO_FF_EXEC1_2 "},
{20, "GDC_MISS_MACHINE_G_FIFO_FF_EXEC1_3 "},
{21, "GDC_MISS_MACHINE_G_FIFO_FF_EXEC_DONE0_0"},
{22, "GDC_MISS_MACHINE_G_FIFO_FF_EXEC_DONE0_1"},
{23, "GDC_MISS_MACHINE_G_FIFO_FF_EXEC_DONE0_2"},
{24, "GDC_MISS_MACHINE_G_FIFO_FF_EXEC_DONE0_3"},
{25, "GDC_MISS_MACHINE_G_FIFO_FF_EXEC_DONE1_0 "},
{26, "GDC_MISS_MACHINE_G_FIFO_FF_EXEC_DONE1_1"},
{27, "GDC_MISS_MACHINE_G_FIFO_FF_EXEC_DONE1_2"},
{28, "GDC_MISS_MACHINE_G_FIFO_FF_EXEC_DONE1_3"},
{29, "GDC_MISS_MACHINE_GDC_LINK_LIST_FF_0"},
{30, "GDC_MISS_MACHINE_GDC_LINK_LIST_FF_1"},
{31, "GDC_MISS_MACHINE_GDC_LINK_LIST_FF_2"},
{32, "GDC_MISS_MACHINE_GDC_LINK_LIST_FF_3"},
{33, "GDC_MISS_MACHINE_GDC_LINK_LIST_FF_4"},
{34, "GDC_MISS_MACHINE_GDC_LINK_LIST_FF_5"},
{35, "GDC_MISS_MACHINE_GDC_LINK_LIST_FF_6"},
{36, "GDC_MISS_MACHINE_G_RSE_PIPE_TXREQ_0"},
{37, "GDC_MISS_MACHINE_G_RSE_PIPE_TXREQ_1"},
{38, "GDC_MISS_MACHINE_G_CREDIT_TXREQ_0"},
{39, "GDC_MISS_MACHINE_G_CREDIT_TXREQ_1"},
{40, "GDC_MISS_MACHINE_G_RSE_PIPE_TXDAT_0"},
{41, "GDC_MISS_MACHINE_G_RSE_PIPE_TXDAT_1"},
{42, "GDC_MISS_MACHINE_G_CREDIT_TXDAT_0"},
{43, "GDC_MISS_MACHINE_G_CREDIT_TXDAT_1"},
{44, "GDC_MISS_MACHINE_G_FIFO_FF_COMPACK_0"},
{45, "GDC_MISS_MACHINE_G_FIFO_FF_COMPACK_1"},
{46, "GDC_MISS_MACHINE_G_FIFO_FF_COMPACK_2"},
{47, "GDC_MISS_MACHINE_G_FIFO_FF_COMPACK_3"},
{48, "GDC_MISS_MACHINE_G_RSE_PIPE_TXRSP_0"},
{49, "GDC_MISS_MACHINE_G_RSE_PIPE_TXRSP_1"},
{50, "GDC_MISS_MACHINE_G_CREDIT_TXRSP_0"},
{51, "GDC_MISS_MACHINE_G_CREDIT_TXRSP_1"},
{52, "GDC_MISS_MACHINE_G_RSE_PIPE_INARB_0"},
{53, "GDC_MISS_MACHINE_G_RSE_PIPE_INARB_1"},
{54, "GDC_MISS_MACHINE_G_FIFO_FF_SNOOP_IN_0"},
{55, "GDC_MISS_MACHINE_G_FIFO_FF_SNOOP_IN_1"},
{56, "GDC_MISS_MACHINE_G_FIFO_FF_SNOOP_IN_2"},
{57, "GDC_MISS_MACHINE_G_FIFO_FF_SNOOP_IN_3"},
{58, "GDC_MISS_MACHINE_G_FIFO_FF_TXRSP_SNOOP_DATAPULL_0"},
{59, "GDC_MISS_MACHINE_G_FIFO_FF_TXRSP_SNOOP_DATAPULL_1"},
{60, "GDC_MISS_MACHINE_G_FIFO_FF_TXRSP_SNOOP_DATAPULL_2"},
{61, "GDC_MISS_MACHINE_G_FIFO_FF_TXRSP_SNOOP_DATAPULL_3"},
{62, "GDC_MISS_MACHINE_G_FIFO_FF_TXDAT_SNOOP_DATAPULL_4"},
{63, "GDC_MISS_MACHINE_G_FIFO_FF_TXDAT_SNOOP_DATAPULL_5"},
{64, "GDC_MISS_MACHINE_G_FIFO_FF_TXDAT_SNOOP_DATAPULL_6"},
{65, "GDC_MISS_MACHINE_G_FIFO_FF_TXDAT_SNOOP_DATAPULL_7"},
{66, "HISTOGRAM_HISTOGRAM_BIN0"},
{67, "HISTOGRAM_HISTOGRAM_BIN1"},
{68, "HISTOGRAM_HISTOGRAM_BIN2"},
{69, "HISTOGRAM_HISTOGRAM_BIN3"},
{70, "HISTOGRAM_HISTOGRAM_BIN4"},
{71, "HISTOGRAM_HISTOGRAM_BIN5"},
{72, "HISTOGRAM_HISTOGRAM_BIN6"},
{73, "HISTOGRAM_HISTOGRAM_BIN7"},
{74, "HISTOGRAM_HISTOGRAM_BIN8"},
{75, "HISTOGRAM_HISTOGRAM_BIN9"},
{-1, NULL}
};

struct mlxbf_pmc_events mlxbf3_mss_events[] = {
{0, "SKYLIB_CDN_TX_FLITS"},
{1, "SKYLIB_DDN_TX_FLITS"},
{2, "SKYLIB_NDN_TX_FLITS"},
{3, "SKYLIB_SDN_TX_FLITS"},
{4, "SKYLIB_UDN_TX_FLITS"},
{5, "SKYLIB_CDN_RX_FLITS"},
{6, "SKYLIB_DDN_RX_FLITS"},
{7, "SKYLIB_NDN_RX_FLITS"},
{8, "SKYLIB_SDN_RX_FLITS"},
{9, "SKYLIB_UDN_RX_FLITS"},
{10, "SKYLIB_CDN_TX_STALL"},
{11, "SKYLIB_DDN_TX_STALL"},
{12, "SKYLIB_NDN_TX_STALL"},
{13, "SKYLIB_SDN_TX_STALL"},
{14, "SKYLIB_UDN_TX_STALL"},
{15, "SKYLIB_CDN_RX_STALL"},
{16, "SKYLIB_DDN_RX_STALL"},
{17, "SKYLIB_NDN_RX_STALL"},
{18, "SKYLIB_SDN_RX_STALL"},
{19, "SKYLIB_UDN_RX_STALL"},
{20, "SKYLIB_CHI_REQ0_TX_FLITS"},
{21, "SKYLIB_CHI_DATA0_TX_FLITS"},
{22, "SKYLIB_CHI_RESP0_TX_FLITS"},
{23, "SKYLIB_CHI_SNP0_TX_FLITS"},
{24, "SKYLIB_CHI_REQ1_TX_FLITS"},
{25, "SKYLIB_CHI_DATA1_TX_FLITS"},
{26, "SKYLIB_CHI_RESP1_TX_FLITS"},
{27, "SKYLIB_CHI_SNP1_TX_FLITS"},
{28, "SKYLIB_CHI_REQ2_TX_FLITS"},
{29, "SKYLIB_CHI_DATA2_TX_FLITS"},
{30, "SKYLIB_CHI_RESP2_TX_FLITS"},
{31, "SKYLIB_CHI_SNP2_TX_FLITS"},
{32, "SKYLIB_CHI_REQ3_TX_FLITS"},
{33, "SKYLIB_CHI_DATA3_TX_FLITS"},
{34, "SKYLIB_CHI_RESP3_TX_FLITS"},
{35, "SKYLIB_CHI_SNP3_TX_FLITS"},
{36, "SKYLIB_TLP_REQ_TX_FLITS"},
{37, "SKYLIB_TLP_RESP_TX_FLITS"},
{38, "SKYLIB_TLP_META_TX_FLITS"},
{39, "SKYLIB_AXIS_DATA_TX_FLITS"},
{40, "SKYLIB_AXIS_CRED_TX_FLITS"},
{41, "SKYLIB_APB_TX_FLITS"},
{42, "SKYLIB_VW_TX_FLITS"},
{43, "SKYLIB_GGA_MSN_W_TX_FLITS"},
{44, "SKYLIB_GGA_MSN_N_TX_FLITS"},
{45, "SKYLIB_CR_REQ_TX_FLITS"},
{46, "SKYLIB_CR_RESP_TX_FLITS"},
{47, "SKYLIB_MSN_PRNF_TX_FLITS"},
{48, "SKYLIB_DBG_DATA_TX_FLITS"},
{49, "SKYLIB_DBG_CRED_TX_FLITS"},
{50, "SKYLIB_CHI_REQ0_RX_FLITS"},
{51, "SKYLIB_CHI_DATA0_RX_FLITS"},
{52, "SKYLIB_CHI_RESP0_RX_FLITS"},
{53, "SKYLIB_CHI_SNP0_RX_FLITS"},
{54, "SKYLIB_CHI_REQ1_RX_FLITS"},
{55, "SKYLIB_CHI_DATA1_RX_FLITS"},
{56, "SKYLIB_CHI_RESP1_RX_FLITS"},
{57, "SKYLIB_CHI_SNP1_RX_FLITS"},
{58, "SKYLIB_CHI_REQ2_RX_FLITS"},
{59, "SKYLIB_CHI_DATA2_RX_FLITS"},
{60, "SKYLIB_CHI_RESP2_RX_FLITS"},
{61, "SKYLIB_CHI_SNP2_RX_FLITS"},
{62, "SKYLIB_CHI_REQ3_RX_FLITS"},
{63, "SKYLIB_CHI_DATA3_RX_FLITS"},
{64, "SKYLIB_CHI_RESP3_RX_FLITS"},
{65, "SKYLIB_CHI_SNP3_RX_FLITS"},
{66, "SKYLIB_TLP_REQ_RX_FLITS"},
{67, "SKYLIB_TLP_RESP_RX_FLITS"},
{68, "SKYLIB_TLP_META_RX_FLITS"},
{69, "SKYLIB_AXIS_DATA_RX_FLITS"},
{70, "SKYLIB_AXIS_CRED_RX_FLITS"},
{71, "SKYLIB_APB_RX_FLITS"},
{72, "SKYLIB_VW_RX_FLITS"},
{73, "SKYLIB_GGA_MSN_W_RX_FLITS"},
{74, "SKYLIB_GGA_MSN_N_RX_FLITS"},
{75, "SKYLIB_CR_REQ_RX_FLITS"},
{76, "SKYLIB_CR_RESP_RX_FLITS"},
{77, "SKYLIB_MSN_PRNF_RX_FLITS"},
{78, "SKYLIB_DBG_DATA_RX_FLITS"},
{79, "SKYLIB_DBG_CRED_RX_FLITS"},
{80, "SKYLIB_CHI_REQ0_TX_STALL"},
{81, "SKYLIB_CHI_DATA0_TX_STALL"},
{82, "SKYLIB_CHI_RESP0_TX_STALL"},
{83, "SKYLIB_CHI_SNP0_TX_STALL"},
{84, "SKYLIB_CHI_REQ1_TX_STALL"},
{85, "SKYLIB_CHI_DATA1_TX_STALL"},
{86, "SKYLIB_CHI_RESP1_TX_STALL"},
{87, "SKYLIB_CHI_SNP1_TX_STALL"},
{88, "SKYLIB_CHI_REQ2_TX_STALL"},
{89, "SKYLIB_CHI_DATA2_TX_STALL"},
{90, "SKYLIB_CHI_RESP2_TX_STALL"},
{91, "SKYLIB_CHI_SNP2_TX_STALL"},
{92, "SKYLIB_CHI_REQ3_TX_STALL"},
{93, "SKYLIB_CHI_DATA3_TX_STALL"},
{94, "SKYLIB_CHI_RESP3_TX_STALL"},
{95, "SKYLIB_CHI_SNP3_TX_STALL"},
{96, "SKYLIB_TLP_REQ_TX_STALL"},
{97, "SKYLIB_TLP_RESP_TX_STALL"},
{98, "SKYLIB_TLP_META_TX_STALL"},
{99, "SKYLIB_AXIS_DATA_TX_STALL"},
{100, "SKYLIB_AXIS_CRED_TX_STALL"},
{101, "SKYLIB_APB_TX_STALL"},
{102, "SKYLIB_VW_TX_STALL"},
{103, "SKYLIB_GGA_MSN_W_TX_STALL"},
{104, "SKYLIB_GGA_MSN_N_TX_STALL"},
{105, "SKYLIB_CR_REQ_TX_STALL"},
{106, "SKYLIB_CR_RESP_TX_STALL"},
{107, "SKYLIB_MSN_PRNF_TX_STALL"},
{108, "SKYLIB_DBG_DATA_TX_STALL"},
{109, "SKYLIB_DBG_CRED_TX_STALL"},
{110, "SKYLIB_CHI_REQ0_RX_STALL"},
{111, "SKYLIB_CHI_DATA0_RX_STALL"},
{112, "SKYLIB_CHI_RESP0_RX_STALL"},
{113, "SKYLIB_CHI_SNP0_RX_STALL"},
{114, "SKYLIB_CHI_REQ1_RX_STALL"},
{115, "SKYLIB_CHI_DATA1_RX_STALL"},
{116, "SKYLIB_CHI_RESP1_RX_STALL"},
{117, "SKYLIB_CHI_SNP1_RX_STALL"},
{118, "SKYLIB_CHI_REQ2_RX_STALL"},
{119, "SKYLIB_CHI_DATA2_RX_STALL"},
{120, "SKYLIB_CHI_RESP2_RX_STALL"},
{121, "SKYLIB_CHI_SNP2_RX_STALL"},
{122, "SKYLIB_CHI_REQ3_RX_STALL"},
{123, "SKYLIB_CHI_DATA3_RX_STALL"},
{124, "SKYLIB_CHI_RESP3_RX_STALL"},
{125, "SKYLIB_CHI_SNP3_RX_STALL"},
{126, "SKYLIB_TLP_REQ_RX_STALL"},
{127, "SKYLIB_TLP_RESP_RX_STALL"},
{128, "SKYLIB_TLP_META_RX_STALL"},
{129, "SKYLIB_AXIS_DATA_RX_STALL"},
{130, "SKYLIB_AXIS_CRED_RX_STALL"},
{131, "SKYLIB_APB_RX_STALL"},
{132, "SKYLIB_VW_RX_STALL"},
{133, "SKYLIB_GGA_MSN_W_RX_STALL"},
{134, "SKYLIB_GGA_MSN_N_RX_STALL"},
{135, "SKYLIB_CR_REQ_RX_STALL"},
{136, "SKYLIB_CR_RESP_RX_STALL"},
{137, "SKYLIB_MSN_PRNF_RX_STALL"},
{138, "SKYLIB_DBG_DATA_RX_STALL"},
{139, "SKYLIB_DBG_CRED_RX_STALL"},
{140, "SKYLIB_CDN_LOOPBACK_FLITS"},
{141, "SKYLIB_DDN_LOOPBACK_FLITS"},
{142, "SKYLIB_NDN_LOOPBACK_FLITS"},
{143, "SKYLIB_SDN_LOOPBACK_FLITS"},
{144, "SKYLIB_UDN_LOOPBACK_FLITS"},
{145, "HISTOGRAM_HISTOGRAM_BIN0"},
{146, "HISTOGRAM_HISTOGRAM_BIN1"},
{147, "HISTOGRAM_HISTOGRAM_BIN2"},
{148, "HISTOGRAM_HISTOGRAM_BIN3"},
{149, "HISTOGRAM_HISTOGRAM_BIN4"},
{150, "HISTOGRAM_HISTOGRAM_BIN5"},
{151, "HISTOGRAM_HISTOGRAM_BIN6"},
{152, "HISTOGRAM_HISTOGRAM_BIN7"},
{153, "HISTOGRAM_HISTOGRAM_BIN8"},
{154, "HISTOGRAM_HISTOGRAM_BIN9"},
{-1, NULL}
};

#endif				/* __MLXBF_PMC_H__ */
