// SPDX-License-Identifier: GPL-2.0-only or BSD-3-Clause

/*
 *  Copyright (c) 2020 NVIDIA Corporation.
 */

#include <linux/acpi.h>
#include <linux/bitfield.h>
#include <linux/bitops.h>
#include <linux/device.h>
#include <linux/gpio/driver.h>
#include <linux/io.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/pm.h>
#include <linux/resource.h>
#include <linux/types.h>
#include <linux/version.h>

#include "gpio-mlxbf_regs.h"

/* Number of pins on BlueField */
#define MLXBF_GPIO_NR 54

/* Pad Electrical Controls. */
#define MLXBF_GPIO_PAD_CONTROL_FIRST_WORD	0x0700
#define MLXBF_GPIO_PAD_CONTROL_1_FIRST_WORD	0x0708
#define MLXBF_GPIO_PAD_CONTROL_2_FIRST_WORD	0x0710
#define MLXBF_GPIO_PAD_CONTROL_3_FIRST_WORD	0x0718

/* GPIO pin 7 is reserved for power good indication on BlueField. */
#define MLXBF_GPIO_POWER_GOOD_PIN			7

/* GPIO pins enabled for interrupt */
#define MLXBF_GPIO_INT_ASSERT_ENA_BITS		0
#define MLXBF_GPIO_INT_DEASSERT_ENA_BITS	BIT(MLXBF_GPIO_POWER_GOOD_PIN)

#ifdef CONFIG_PM
struct mlxbf_gpio_context_save_regs {
	u64 scratchpad;
	u64 pad_control[MLXBF_GPIO_NR];
	u64 pin_dir_i;
	u64 pin_dir_o;
};
#endif

/* Device state structure. */
struct mlxbf_gpio_state {
	struct gpio_chip gc;

	/*
	 * Must hold this lock to modify shared data
	 * in kernel versions < 5.2
	 */
	spinlock_t lock;

	/* Memory Address */
	void __iomem *base;

	int hwirq;

#ifdef CONFIG_PM
	struct mlxbf_gpio_context_save_regs csave_regs;
#endif
};

void acpi_gpiochip_request_interrupts(struct gpio_chip *chip);

#if LINUX_VERSION_CODE < KERNEL_VERSION(5, 2, 0)
static int mlxbf_gpio_direction_input(struct gpio_chip *chip,
					unsigned int offset)
{
	struct mlxbf_gpio_state *gs = gpiochip_get_data(chip);
	u64 in;
	u64 out;

	spin_lock(&gs->lock);
	out = readq(gs->base + GPIO_PIN_DIR_O);
	in = readq(gs->base + GPIO_PIN_DIR_I);

	writeq(out & ~BIT(offset), gs->base + GPIO_PIN_DIR_O);
	writeq(in | BIT(offset), gs->base + GPIO_PIN_DIR_I);
	spin_unlock(&gs->lock);

	return 0;
}

static int mlxbf_gpio_direction_output(struct gpio_chip *chip,
		unsigned int offset, int value)
{
	struct mlxbf_gpio_state *gs = gpiochip_get_data(chip);
	u64 in;
	u64 out;

	spin_lock(&gs->lock);
	out = readq(gs->base + GPIO_PIN_DIR_O);
	in = readq(gs->base + GPIO_PIN_DIR_I);

	writeq(out | BIT(offset), gs->base + GPIO_PIN_DIR_O);
	writeq(in & ~BIT(offset), gs->base + GPIO_PIN_DIR_I);
	spin_unlock(&gs->lock);

	return 0;
}

static int mlxbf_gpio_get(struct gpio_chip *chip, unsigned int offset)
{
	u64 value;
	struct mlxbf_gpio_state *gs = gpiochip_get_data(chip);

	spin_lock(&gs->lock);
	value = readq(gs->base + GPIO_PIN_STATE);
	spin_unlock(&gs->lock);

	return (value >> offset) & 1;
}

static void mlxbf_gpio_set(struct gpio_chip *chip, unsigned int offset,
	int value)
{
	u64 data;
	struct mlxbf_gpio_state *gs = gpiochip_get_data(chip);

	spin_lock(&gs->lock);
	data = readq(gs->base + GPIO_PIN_STATE);

	if (value)
		data |= BIT(offset);
	else
		data &= ~BIT(offset);
	writeq(data, gs->base + GPIO_PIN_STATE);
	spin_unlock(&gs->lock);
}
#endif /* LINUX_VERSION_CODE < KERNEL_VERSION(5, 2, 0) */

static int mlxbf_gpio_to_irq(struct gpio_chip *chip, u32 gpio)
{
	struct mlxbf_gpio_state *gs = gpiochip_get_data(chip);

	return gs->hwirq;
}

static void mlxbf_gpio_cfg_irq(struct mlxbf_gpio_state *gs)
{
	u64 intr_cfg;

	intr_cfg = readq(gs->base + GPIO_INT_SETUP);
	intr_cfg |= FIELD_PREP(GPIO_INT_SETUP__GBL_ENA_MASK, 1);
	writeq(intr_cfg, gs->base + GPIO_INT_SETUP);

	intr_cfg = readq(gs->base + GPIO_INT_ENA_ASSERT);
	intr_cfg |= FIELD_PREP(GPIO_INT_ENA_ASSERT__INT_ENA_ASSERT_MASK,
			       MLXBF_GPIO_INT_ASSERT_ENA_BITS);
	writeq(intr_cfg, gs->base + GPIO_INT_ENA_ASSERT);

	intr_cfg = readq(gs->base + GPIO_INT_ENA_DEASSERT);
	intr_cfg |= FIELD_PREP(GPIO_INT_ENA_DEASSERT__INT_ENA_DEASSERT_MASK,
			       MLXBF_GPIO_INT_DEASSERT_ENA_BITS);
	writeq(intr_cfg, gs->base + GPIO_INT_ENA_DEASSERT);
}

static int mlxbf_gpio_probe(struct platform_device *pdev)
{
	struct mlxbf_gpio_state *gs;
	struct device *dev = &pdev->dev;
	struct gpio_chip *gc;
	int ret, irq;
#if LINUX_VERSION_CODE < KERNEL_VERSION(5, 2, 0)
	struct resource *res;
	const char *name = dev_name(dev);
#endif

	gs = devm_kzalloc(&pdev->dev, sizeof(*gs), GFP_KERNEL);
	if (!gs)
		return -ENOMEM;

#if LINUX_VERSION_CODE < KERNEL_VERSION(5, 2, 0)
	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	gs->base = devm_ioremap_resource(&pdev->dev, res);
	if (IS_ERR(gs->base))
		return PTR_ERR(gs->base);

	gc = &gs->gc;

	gc->direction_input = mlxbf_gpio_direction_input;
	gc->direction_output = mlxbf_gpio_direction_output;
	gc->get = mlxbf_gpio_get;
	gc->set = mlxbf_gpio_set;
	gc->label = name;
	gc->parent = dev;
	gc->base = -1;
	spin_lock_init(&gs->lock);
#else
	gs->base = devm_platform_ioremap_resource(pdev, 0);
	if (IS_ERR(gs->base))
		return PTR_ERR(gs->base);

	gc = &gs->gc;

	ret = bgpio_init(gc, dev, 8,
			 gs->base + GPIO_PIN_STATE,
			 NULL,
			 NULL,
			 gs->base + GPIO_PIN_DIR_O,
			 gs->base + GPIO_PIN_DIR_I,
			 0);
	if (ret)
		return -ENODEV;
#endif /* LINUX_VERSION_CODE < KERNEL_VERSION(5, 2, 0) */

	irq = platform_get_irq(pdev, 0);
	if (irq < 0)
		return irq;

	gc->owner = THIS_MODULE;
	gc->ngpio = MLXBF_GPIO_NR;
	gs->hwirq = irq;
	gc->to_irq = mlxbf_gpio_to_irq;

	mlxbf_gpio_cfg_irq(gs);

	ret = devm_gpiochip_add_data(dev, &gs->gc, gs);
	if (ret) {
		dev_err(&pdev->dev, "Failed adding memory mapped gpiochip\n");
		return ret;
	}

	platform_set_drvdata(pdev, gs);
	acpi_gpiochip_request_interrupts(gc);
	dev_info(&pdev->dev, "registered Mellanox BlueField GPIO");

	return 0;
}

#ifdef CONFIG_PM
static int mlxbf_gpio_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct mlxbf_gpio_state *gs = platform_get_drvdata(pdev);

	gs->csave_regs.scratchpad = readq(gs->base + GPIO_SCRATCHPAD);
	gs->csave_regs.pad_control[0] =
		readq(gs->base + MLXBF_GPIO_PAD_CONTROL_FIRST_WORD);
	gs->csave_regs.pad_control[1] =
		readq(gs->base + MLXBF_GPIO_PAD_CONTROL_1_FIRST_WORD);
	gs->csave_regs.pad_control[2] =
		readq(gs->base + MLXBF_GPIO_PAD_CONTROL_2_FIRST_WORD);
	gs->csave_regs.pad_control[3] =
		readq(gs->base + MLXBF_GPIO_PAD_CONTROL_3_FIRST_WORD);
	gs->csave_regs.pin_dir_i = readq(gs->base + GPIO_PIN_DIR_I);
	gs->csave_regs.pin_dir_o = readq(gs->base + GPIO_PIN_DIR_O);

	return 0;
}

static int mlxbf_gpio_resume(struct platform_device *pdev)
{
	struct mlxbf_gpio_state *gs = platform_get_drvdata(pdev);

	writeq(gs->csave_regs.scratchpad, gs->base + GPIO_SCRATCHPAD);
	writeq(gs->csave_regs.pad_control[0],
	       gs->base + MLXBF_GPIO_PAD_CONTROL_FIRST_WORD);
	writeq(gs->csave_regs.pad_control[1],
	       gs->base + MLXBF_GPIO_PAD_CONTROL_1_FIRST_WORD);
	writeq(gs->csave_regs.pad_control[2],
	       gs->base + MLXBF_GPIO_PAD_CONTROL_2_FIRST_WORD);
	writeq(gs->csave_regs.pad_control[3],
	       gs->base + MLXBF_GPIO_PAD_CONTROL_3_FIRST_WORD);
	writeq(gs->csave_regs.pin_dir_i, gs->base + GPIO_PIN_DIR_I);
	writeq(gs->csave_regs.pin_dir_o, gs->base + GPIO_PIN_DIR_O);

	return 0;
}
#endif

static const struct acpi_device_id mlxbf_gpio_acpi_match[] = {
	{ "MLNXBF02", 0 },
	{}
};
MODULE_DEVICE_TABLE(acpi, mlxbf_gpio_acpi_match);

static struct platform_driver mlxbf_gpio_driver = {
	.driver = {
		.name = "mlxbf_gpio",
		.acpi_match_table = ACPI_PTR(mlxbf_gpio_acpi_match),
	},
	.probe    = mlxbf_gpio_probe,
#ifdef CONFIG_PM
	.suspend  = mlxbf_gpio_suspend,
	.resume   = mlxbf_gpio_resume,
#endif
};

module_platform_driver(mlxbf_gpio_driver);

MODULE_DESCRIPTION("Mellanox BlueField GPIO Driver");
MODULE_AUTHOR("Mellanox Technologies");
MODULE_LICENSE("Dual BSD/GPL");
