// SPDX-License-Identifier: GPL-2.0 or BSD-3-Clause
/*
 * Device driver for Mellanox Sella CPLD
 *
 * Copyright (c) 2018, Mellanox Technologies. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */

#include <linux/acpi.h>
#include <linux/bitops.h>
#include <linux/device.h>
#include <linux/errno.h>
#include <linux/hwmon.h>
#include <linux/hwmon-sysfs.h>
#include <linux/i2c.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/of_device.h>
#include <linux/platform_device.h>
#include <linux/regmap.h>
#include <linux/sysfs.h>

#include "cpld_regs.h"

struct mlx_cpld_info {
	struct i2c_client *client;
	struct device *hwmon;
};

struct reg_map {
	char *name;
	uint8_t offset;
	uint8_t mask;
};

struct reg_map power_status[] = {
{"12v_pcie",	CPLD_POWER_STATUS_0,	CPLD_POWER_STATUS_0__12V_PCIE},
{"12v_atx",	CPLD_POWER_STATUS_0,	CPLD_POWER_STATUS_0__12V_ATX},
{"3p3v",	CPLD_POWER_STATUS_0,	CPLD_POWER_STATUS_0__3P3V},
{"5v",		CPLD_POWER_STATUS_0,	CPLD_POWER_STATUS_0__5V},
{"vcore",	CPLD_POWER_STATUS_0,	CPLD_POWER_STATUS_0__VCORE},
{"2p5v_ddr",	CPLD_POWER_STATUS_0,	CPLD_POWER_STATUS_0__2P5V_DDR},
{"1p2v_ddr",	CPLD_POWER_STATUS_0,	CPLD_POWER_STATUS_0__1P2V_DDR},
{"0p6v_ddr0",	CPLD_POWER_STATUS_1,	CPLD_POWER_STATUS_1__0P6V_DDR0},
{"0p6v_ddr1",	CPLD_POWER_STATUS_1,	CPLD_POWER_STATUS_1__0P6V_DDR1},
{"1p2v_serdes",	CPLD_POWER_STATUS_1,	CPLD_POWER_STATUS_1__1P2V_SERDES},
{"1p8v",	CPLD_POWER_STATUS_1,	CPLD_POWER_STATUS_1__1P8V},
{"vrd_fault",	CPLD_POWER_STATUS_1,	CPLD_POWER_STATUS_1__VRD_FAULT},
{NULL, -1, -1}
};

struct mlx_cpld_info *info;
struct i2c_client *cpld_client;

static ssize_t revision_show(struct device *dev,
		struct device_attribute *devattr, char *buf)
{
	unsigned int value;
	int ret;

	ret = i2c_smbus_write_byte_data(info->client, CPLD_REVISION >> 8,
		CPLD_REVISION & CPLD_REVISION_MASK);
	if (ret)
		return -EFAULT;
	value = i2c_smbus_read_byte_data(info->client, CPLD_REVISION);

	return sprintf(buf, "0x%x\n", value);
}

static ssize_t show_power_status(struct device *dev,
		struct device_attribute *devattr, char *buf)
{
	unsigned int value;
	int ret, i = 0, offset = -1, mask = -1;

	while (power_status[i].name != NULL) {
		if (strcmp(devattr->attr.name, power_status[i].name) == 0) {
			offset = power_status[i].offset;
			mask = power_status[i].mask;
			break;
		}
		++i;
	}
	if (offset == -1)
		return -EFAULT;

	ret = i2c_smbus_write_byte_data(info->client, offset >> 8,
		offset & 0xff);
	value = i2c_smbus_read_byte_data(info->client, offset);
	if (value & mask)
		return sprintf(buf, "1\n");
	else
		return sprintf(buf, "0\n");
}

static DEVICE_ATTR_RO(revision);
static SENSOR_DEVICE_ATTR(12v_pcie, 0444, show_power_status, NULL, 0);
static SENSOR_DEVICE_ATTR(12v_atx, 0444, show_power_status, NULL, 0);
static SENSOR_DEVICE_ATTR(3p3v, 0444, show_power_status, NULL, 0);
static SENSOR_DEVICE_ATTR(5v, 0444, show_power_status, NULL, 0);
static SENSOR_DEVICE_ATTR(vcore, 0444, show_power_status, NULL, 0);
static SENSOR_DEVICE_ATTR(2p5v_ddr, 0444, show_power_status, NULL, 0);
static SENSOR_DEVICE_ATTR(1p2v_ddr, 0444, show_power_status, NULL, 0);
static SENSOR_DEVICE_ATTR(0p6v_ddr0, 0444, show_power_status, NULL, 0);
static SENSOR_DEVICE_ATTR(0p6v_ddr1, 0444, show_power_status, NULL, 0);
static SENSOR_DEVICE_ATTR(1p2v_serdes, 0444, show_power_status, NULL, 0);
static SENSOR_DEVICE_ATTR(1p8v, 0444, show_power_status, NULL, 0);
static SENSOR_DEVICE_ATTR(vrd_fault, 0444, show_power_status, NULL, 0);

static struct attribute *mlx_cpld_attrs[] = {
	&dev_attr_revision.attr,
	&sensor_dev_attr_12v_pcie.dev_attr.attr,
	&sensor_dev_attr_12v_atx.dev_attr.attr,
	&sensor_dev_attr_3p3v.dev_attr.attr,
	&sensor_dev_attr_5v.dev_attr.attr,
	&sensor_dev_attr_vcore.dev_attr.attr,
	&sensor_dev_attr_2p5v_ddr.dev_attr.attr,
	&sensor_dev_attr_1p2v_ddr.dev_attr.attr,
	&sensor_dev_attr_0p6v_ddr0.dev_attr.attr,
	&sensor_dev_attr_0p6v_ddr1.dev_attr.attr,
	&sensor_dev_attr_1p2v_serdes.dev_attr.attr,
	&sensor_dev_attr_1p8v.dev_attr.attr,
	&sensor_dev_attr_vrd_fault.dev_attr.attr,
	NULL
};
ATTRIBUTE_GROUPS(mlx_cpld);

static int
mlx_cpld_probe(struct i2c_client *client,
		const struct i2c_device_id *id)
{
	int err;
	info = devm_kzalloc(&client->dev, sizeof(*info), GFP_KERNEL);
	if (!info)
		return -ENOMEM;

	info->client = client;

	info->hwmon = devm_hwmon_device_register_with_groups(&client->dev,
		"mlx_cpld", info, mlx_cpld_groups);
	if (IS_ERR(info->hwmon)) {
		err = PTR_ERR(info->hwmon);
		kfree(info);
		return err;
	}

	i2c_set_clientdata(client, info);

	dev_info(&client->dev, "Probe successful\n");
	return 0;
}

static int
mlx_cpld_remove(struct i2c_client *client)
{

	kfree(info);

	return 0;
}

static const struct i2c_device_id mlx_cpld_id[] = {
	{ "mlx_cpld", 0 },
	{ }
};
MODULE_DEVICE_TABLE(i2c, mlx_cpld_id);

static const struct acpi_device_id mlx_cpld_ids[] = {
	{ "MLNXBF12", 0 },
	{ },
};
MODULE_DEVICE_TABLE(acpi, mlx_cpld_ids);

static struct i2c_driver mlx_cpld_driver = {
	.class		= I2C_CLASS_HWMON,
	.driver		= {
		.owner			= THIS_MODULE,
		.name			= "mlx_cpld",
		.acpi_match_table	= ACPI_PTR(mlx_cpld_ids),
	},
	.probe		= mlx_cpld_probe,
	.remove		= mlx_cpld_remove,
	.id_table	= mlx_cpld_id,
};

static int __init init_cpld(void)
{
	struct i2c_adapter *adapter;
	struct i2c_board_info board_info = {
		.type = "mlx_cpld",
		.addr = 0x40,
	};
	int ret;

	request_module("i2c-mlxbf");

	ret = i2c_add_driver(&mlx_cpld_driver);
	if (ret) {
		pr_err("CPLD Driver add failed: %d\n", ret);
		return ret;
	}

	adapter = i2c_get_adapter(2);
	if (!adapter) {
		pr_err("CPLD Get adapter failed\n");
		goto out_err;
	}

	cpld_client = i2c_new_client_device(adapter, &board_info);
	if (IS_ERR(cpld_client)) {
		pr_err("CPLD Device add failed\n");
		goto out_err;
	}

	return 0;

out_err:
	i2c_del_driver(&mlx_cpld_driver);
	return -ENODEV;
}
module_init(init_cpld);

static void __exit exit_cpld(void)
{
	i2c_unregister_device(info->client);
	i2c_del_driver(&mlx_cpld_driver);
}
module_exit(exit_cpld);

MODULE_AUTHOR("Mellanox Technologies");
MODULE_DESCRIPTION("Mellanox Sella CPLD Driver");
MODULE_LICENSE("Dual BSD/GPL");
