// SPDX-License-Identifier: GPL-2.0-only or BSD-3-Clause
/* Copyright (C) 2021-2023 NVIDIA CORPORATION & AFFILIATES */

#include <linux/bitfield.h>
#include <linux/bitops.h>
#include <linux/device.h>
#include <linux/err.h>
#include <linux/gpio/driver.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/spinlock.h>
#include <linux/types.h>

/* Internal use only */
#include <linux/resource.h>
#include <linux/version.h>

#define DRV_VERSION "2.0"

/*
 * There are 2 YU GPIO blocks:
 * gpio[0]: HOST_GPIO0->HOST_GPIO31
 * gpio[1]: HOST_GPIO32->HOST_GPIO55
 */
#define MLXBF3_GPIO_MAX_PINS_PER_BLOCK 32

/*
 * fw_gpio[x] block registers and their offset
 */
#define MLXBF_GPIO_FW_OUTPUT_ENABLE_SET	  0x00
#define MLXBF_GPIO_FW_DATA_OUT_SET        0x04

#define MLXBF_GPIO_FW_OUTPUT_ENABLE_CLEAR 0x00
#define MLXBF_GPIO_FW_DATA_OUT_CLEAR      0x04

#define MLXBF_GPIO_CAUSE_RISE_EN          0x00
#define MLXBF_GPIO_CAUSE_FALL_EN          0x04
#define MLXBF_GPIO_READ_DATA_IN           0x08

#define MLXBF_GPIO_CAUSE_OR_CAUSE_EVTEN0  0x00
#define MLXBF_GPIO_CAUSE_OR_EVTEN0        0x14
#define MLXBF_GPIO_CAUSE_OR_CLRCAUSE      0x18

struct mlxbf3_gpio_context {
	struct gpio_chip gc;

	/* YU GPIO block address */
	void __iomem *gpio_set_io;
	void __iomem *gpio_clr_io;
	void __iomem *gpio_io;

	/* YU GPIO cause block address */
	void __iomem *gpio_cause_io;

	/* Mask of valid gpios that can be accessed by software */
	unsigned int valid_mask;
};

static void mlxbf3_gpio_irq_enable(struct irq_data *irqd)
{
	struct gpio_chip *gc = irq_data_get_irq_chip_data(irqd);
	struct mlxbf3_gpio_context *gs = gpiochip_get_data(gc);
	irq_hw_number_t offset = irqd_to_hwirq(irqd);
	unsigned long flags;
	u32 val;

#if LINUX_VERSION_CODE > KERNEL_VERSION(4, 20, 0)
	gpiochip_enable_irq(gc, offset);
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 15, 91)
	raw_spin_lock_irqsave(&gs->gc.bgpio_lock, flags);
#else
	spin_lock_irqsave(&gs->gc.bgpio_lock, flags);
#endif

	writel(BIT(offset), gs->gpio_cause_io + MLXBF_GPIO_CAUSE_OR_CLRCAUSE);

	val = readl(gs->gpio_cause_io + MLXBF_GPIO_CAUSE_OR_EVTEN0);
	val |= BIT(offset);
	writel(val, gs->gpio_cause_io + MLXBF_GPIO_CAUSE_OR_EVTEN0);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 15, 91)
	raw_spin_unlock_irqrestore(&gs->gc.bgpio_lock, flags);
#else
	spin_unlock_irqrestore(&gs->gc.bgpio_lock, flags);
#endif

}

static void mlxbf3_gpio_irq_disable(struct irq_data *irqd)
{
	struct gpio_chip *gc = irq_data_get_irq_chip_data(irqd);
	struct mlxbf3_gpio_context *gs = gpiochip_get_data(gc);
	irq_hw_number_t offset = irqd_to_hwirq(irqd);
	unsigned long flags;
	u32 val;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 15, 91)
	raw_spin_lock_irqsave(&gs->gc.bgpio_lock, flags);
#else
	spin_lock_irqsave(&gs->gc.bgpio_lock, flags);
#endif

	val = readl(gs->gpio_cause_io + MLXBF_GPIO_CAUSE_OR_EVTEN0);
	val &= ~BIT(offset);
	writel(val, gs->gpio_cause_io + MLXBF_GPIO_CAUSE_OR_EVTEN0);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 15, 91)
	raw_spin_unlock_irqrestore(&gs->gc.bgpio_lock, flags);
#else
	spin_unlock_irqrestore(&gs->gc.bgpio_lock, flags);
#endif

#if LINUX_VERSION_CODE > KERNEL_VERSION(4, 20, 0)
	gpiochip_disable_irq(gc, offset);
#endif
}

static irqreturn_t mlxbf3_gpio_irq_handler(int irq, void *ptr)
{
	struct mlxbf3_gpio_context *gs = ptr;
	struct gpio_chip *gc = &gs->gc;
	unsigned long pending;
	u32 level;

	pending = readl(gs->gpio_cause_io + MLXBF_GPIO_CAUSE_OR_CAUSE_EVTEN0);
	writel(pending, gs->gpio_cause_io + MLXBF_GPIO_CAUSE_OR_CLRCAUSE);

	for_each_set_bit(level, &pending, gc->ngpio)
		generic_handle_irq(irq_find_mapping(gc->irq.domain, level));

	return IRQ_RETVAL(pending);
}

static int
mlxbf3_gpio_irq_set_type(struct irq_data *irqd, unsigned int type)
{
	struct gpio_chip *gc = irq_data_get_irq_chip_data(irqd);
	struct mlxbf3_gpio_context *gs = gpiochip_get_data(gc);
	irq_hw_number_t offset = irqd_to_hwirq(irqd);
	unsigned long flags;
	u32 val;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 15, 91)
	raw_spin_lock_irqsave(&gs->gc.bgpio_lock, flags);
#else
	spin_lock_irqsave(&gs->gc.bgpio_lock, flags);
#endif

	switch (type & IRQ_TYPE_SENSE_MASK) {
	case IRQ_TYPE_EDGE_BOTH:
		val = readl(gs->gpio_io + MLXBF_GPIO_CAUSE_FALL_EN);
		val |= BIT(offset);
		writel(val, gs->gpio_io + MLXBF_GPIO_CAUSE_FALL_EN);
		val = readl(gs->gpio_io + MLXBF_GPIO_CAUSE_RISE_EN);
		val |= BIT(offset);
		writel(val, gs->gpio_io + MLXBF_GPIO_CAUSE_RISE_EN);
		break;
	case IRQ_TYPE_EDGE_RISING:
		val = readl(gs->gpio_io + MLXBF_GPIO_CAUSE_RISE_EN);
		val |= BIT(offset);
		writel(val, gs->gpio_io + MLXBF_GPIO_CAUSE_RISE_EN);
		break;
	case IRQ_TYPE_EDGE_FALLING:
		val = readl(gs->gpio_io + MLXBF_GPIO_CAUSE_FALL_EN);
		val |= BIT(offset);
		writel(val, gs->gpio_io + MLXBF_GPIO_CAUSE_FALL_EN);
		break;
	default:
#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 15, 91)
		raw_spin_unlock_irqrestore(&gs->gc.bgpio_lock, flags);
#else
		spin_unlock_irqrestore(&gs->gc.bgpio_lock, flags);
#endif
		return -EINVAL;
	}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 15, 91)
	raw_spin_unlock_irqrestore(&gs->gc.bgpio_lock, flags);
#else
	spin_unlock_irqrestore(&gs->gc.bgpio_lock, flags);
#endif

	irq_set_handler_locked(irqd, handle_edge_irq);

	return 0;
}

/* This function needs to be defined for handle_edge_irq() */
static void mlxbf3_gpio_irq_ack(struct irq_data *data)
{
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 20, 0)
static int mlxbf3_gpio_init_valid_mask(struct gpio_chip *gc,
				       unsigned long *valid_mask,
				       unsigned int ngpios)
{
	struct mlxbf3_gpio_context *gs = gpiochip_get_data(gc);

	*valid_mask = gs->valid_mask;

	return 0;
}
#endif

static struct irq_chip gpio_mlxbf3_irqchip = {
	.name = "MLNXBF33",
	.irq_ack = mlxbf3_gpio_irq_ack,
	.irq_set_type = mlxbf3_gpio_irq_set_type,
	.irq_enable = mlxbf3_gpio_irq_enable,
	.irq_disable = mlxbf3_gpio_irq_disable,
};

static int mlxbf3_gpio_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct mlxbf3_gpio_context *gs;
	struct gpio_irq_chip *girq;
	struct gpio_chip *gc;
	int ret, irq;
#if LINUX_VERSION_CODE < KERNEL_VERSION(5, 1, 0)
	struct resource *res;
#endif
#if LINUX_VERSION_CODE < KERNEL_VERSION(4, 20, 0)
	unsigned int ngpio;
#endif

	gs = devm_kzalloc(dev, sizeof(*gs), GFP_KERNEL);
	if (!gs)
		return -ENOMEM;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 1, 0)
	gs->gpio_io = devm_platform_ioremap_resource(pdev, 0);
	if (IS_ERR(gs->gpio_io))
		return PTR_ERR(gs->gpio_io);

	gs->gpio_cause_io = devm_platform_ioremap_resource(pdev, 1);
	if (IS_ERR(gs->gpio_cause_io))
		return PTR_ERR(gs->gpio_cause_io);

	gs->gpio_set_io = devm_platform_ioremap_resource(pdev, 2);
	if (IS_ERR(gs->gpio_set_io))
		return PTR_ERR(gs->gpio_set_io);

	gs->gpio_clr_io = devm_platform_ioremap_resource(pdev, 3);
	if (IS_ERR(gs->gpio_clr_io))
		return PTR_ERR(gs->gpio_clr_io);
#else
	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res)
		return -ENODEV;
	gs->gpio_io = devm_ioremap_resource(dev, res);
	if (IS_ERR(gs->gpio_io))
		return PTR_ERR(gs->gpio_io);

	res = platform_get_resource(pdev, IORESOURCE_MEM, 1);
	if (!res)
		return -ENODEV;
	gs->gpio_cause_io = devm_ioremap_resource(dev, res);
	if (IS_ERR(gs->gpio_cause_io))
		return PTR_ERR(gs->gpio_cause_io);

	res = platform_get_resource(pdev, IORESOURCE_MEM, 2);
	if (!res)
		return -ENODEV;
	gs->gpio_set_io = devm_ioremap_resource(dev, res);
	if (IS_ERR(gs->gpio_set_io))
		return PTR_ERR(gs->gpio_set_io);

	res = platform_get_resource(pdev, IORESOURCE_MEM, 3);
	if (!res)
		return -ENODEV;
	gs->gpio_clr_io = devm_ioremap_resource(dev, res);
	if (IS_ERR(gs->gpio_clr_io))
		return PTR_ERR(gs->gpio_clr_io);
#endif
	gs->valid_mask = 0x0;
	device_property_read_u32(dev, "valid_mask", &gs->valid_mask);

	gc = &gs->gc;

	ret = bgpio_init(gc, dev, 4,
			gs->gpio_io + MLXBF_GPIO_READ_DATA_IN,
			gs->gpio_set_io + MLXBF_GPIO_FW_DATA_OUT_SET,
			gs->gpio_clr_io + MLXBF_GPIO_FW_DATA_OUT_CLEAR,
			gs->gpio_set_io + MLXBF_GPIO_FW_OUTPUT_ENABLE_SET,
			gs->gpio_clr_io + MLXBF_GPIO_FW_OUTPUT_ENABLE_CLEAR, 0);

	gc->request = gpiochip_generic_request;
	gc->free = gpiochip_generic_free;
	gc->owner = THIS_MODULE;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 20, 0)
	gc->init_valid_mask = mlxbf3_gpio_init_valid_mask;
#else
	/* This is needed for gpiochip_init_valid_mask to initialize
	 * valid_mask pointer.
	 */
	gc->need_valid_mask = true;

	/* Set ngpio in cases where the customer hasn't cherry-picked
	 * the fix for setting ngpio properly in bgpio_init
	 */
	if (device_property_read_u32(dev, "ngpios", &ngpio))
		ngpio = MLXBF3_GPIO_MAX_PINS_PER_BLOCK;

	gc->ngpio = ngpio;
#endif

	irq = platform_get_irq(pdev, 0);
	if (irq >= 0) {
		girq = &gs->gc.irq;
		girq->chip = &gpio_mlxbf3_irqchip;
		girq->default_type = IRQ_TYPE_NONE;
		/* This will let us handle the parent IRQ in the driver */
		girq->num_parents = 0;
		girq->parents = NULL;
		girq->parent_handler = NULL;
		girq->handler = handle_bad_irq;

		/*
		 * Directly request the irq here instead of passing
		 * a flow-handler because the irq is shared.
		 */
		ret = devm_request_irq(dev, irq, mlxbf3_gpio_irq_handler,
				       IRQF_SHARED, dev_name(dev), gs);
		if (ret) {
			dev_err(dev, "failed to request IRQ");
			return ret;
		}
	}

	platform_set_drvdata(pdev, gs);

	ret = devm_gpiochip_add_data(dev, &gs->gc, gs);
	if (ret) {
		dev_err(dev, "Failed adding memory mapped gpiochip\n");
		return ret;
	}

#if LINUX_VERSION_CODE < KERNEL_VERSION(4, 20, 0)
	/* Do this after devm_gpiochip_add_data to overwrite the valid_mask */
	*(gs->gc.valid_mask) = gs->valid_mask;
#endif

	return 0;
}

static const struct acpi_device_id mlxbf3_gpio_acpi_match[] = {
	{ "MLNXBF33", 0 },
	{}
};
MODULE_DEVICE_TABLE(acpi, mlxbf3_gpio_acpi_match);

static struct platform_driver mlxbf3_gpio_driver = {
	.driver = {
		.name = "mlxbf3_gpio",
		.acpi_match_table = mlxbf3_gpio_acpi_match,
	},
	.probe    = mlxbf3_gpio_probe,
};
module_platform_driver(mlxbf3_gpio_driver);

MODULE_DESCRIPTION("NVIDIA BlueField-3 GPIO Driver");
MODULE_AUTHOR("Asmaa Mnebhi <asmaa@nvidia.com>");
MODULE_LICENSE("Dual BSD/GPL");
MODULE_VERSION(DRV_VERSION);
