/*
 * SPDX-FileCopyrightText: NVIDIA CORPORATION & AFFILIATES
 * Copyright (c) 2023-2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: LicenseRef-NvidiaProprietary
 *
 * NVIDIA CORPORATION, its affiliates and licensors retain all intellectual
 * property and proprietary rights in and to this material, related
 * documentation and any modifications thereto. Any use, reproduction,
 * disclosure or distribution of this material and related documentation
 * without an express license agreement from NVIDIA CORPORATION or
 * its affiliates is strictly prohibited.
 */

#include <config.h>

#include <doca_bitfield.h>
#include <doca_flow.h>
#include <doca_flow_ct.h>
#include <rte_flow.h>
#include <sys/types.h>

#include "conntrack-offload.h"
#include "conntrack-private.h"
#include "coverage.h"
#include "doca-action-split.h"
#include "doca-mirror.h"
#include "doca-pipe-group.h"
#include "dp-packet.h"
#include "dpif-doca.h"
#include "dpdk-offload-doca.h"
#include "dpdk-offload-provider.h"
#include "id-fpool.h"
#include "id-pool.h"
#include "openflow/openflow.h"
#include "openvswitch/vlog.h"
#include "offload-metadata.h"
#include "openvswitch/list.h"
#include "openvswitch/match.h"
#include "openvswitch/vlog.h"
#include "ovs-doca.h"
#include "netdev-doca.h"
#include "netdev-offload-doca.h"
#include "netdev-vport.h"
#include "refmap.h"
#include "timeval.h"
#include "util.h"

/*
 * DOCA offload implementation for DPDK provider.
 *
 * Connection Tracking
 * ===================
 *
 * The CT offload implementation over basic pipes is designed as such:
 *
 * +---------------------------------------------------------------------------------------------+
 * | Control pipes                                                                               |
 * |                                                                                             |
 * |                 ,-[ CT Zone X ]-----.                                                       |
 * |                 |  ,-[ CT Zone Y ]-----.                                                    |
 * |                 |  |  ,-[ CT Zone Z ]-----.                                                 |
 * |                 |  |  |                   |                                                 |
 * |                 |  |  |  ,---------.      |                                                 |
 * |                 |  |  |  |ct_zone=Z+------+--------------------------------------.          |
 * |                 |  |  |  `---------'hit   |                                      |          |
 * |                 |  |  |                   | +----------------------------+       |          |
 * |                 |  |  |                   | | Basic pipes                |       |          |
 * |                 |  |  |                   | |                            |       |          |
 * |                 |  |  |                   | |  ,-[ CT IPv4 x UDP ]-.     |       |          |
 * |                 |  |  |    ,----------.   | |  |                   |     |       |          |
 * |                 |  |  |    |IPv4 + UDP+---+--->|  ,-[ CT IPv4 x TCP ]-.  |       v          |
 * |  ,-[ Pre-CT ]-. |  |  |    `----------'hit| |  |  |                   |  | ,-[ Post-CT ]--. |
 * |  |            +------>|    ,----------.   | |  |  | ,-------------.   |--->|              | |
 * |  |            | |  |  |    |IPv4 + TCP+---+------>| | CT entries  +---+--->|              | |
 * |  `------------' |  |  |    `----------'hit| |  |  | `-------------'hit|  | `--------------' |
 * |                 |  |  |       ,---------. | |  |  |   ,---------.     |  |                  |
 * |                 |  |  |       |Catch-all| | |  |  |   |Catch-all|     |  |                  |
 * |                 `--|  |       `----+----' | |  `--|   `----+----'     |  |                  |
 * |                    `--|            |      | |     `--------+----------'  |                  |
 * |                       `------------+------' |           |  |             |                  |
 * |                                    |        +-----------|--|-------------+                  |
 * |                                    v                    |  |                                |
 * |                             ,-[ Miss pipe ]-------------v--v-------.                        |
 * |                             |         < Execute miss handling >    |                        |
 * |                             `--------------------------------------'                        |
 * +---------------------------------------------------------------------------------------------+
 *
 * This model is replicated once per eswitch.
 *
 * A megaflow that contains a 'ct()' action is split
 * into its 'pre-CT' and 'post-CT' part. The pre-CT is inserted
 * into the eswitch root pipe, and contains the megaflow original
 * match.
 *
 * On match, to execute CT, the packet is sent to the 'CT-zone' pipes,
 * one pipe per CT zone ID. If the ct_zone value is already set on the packet
 * and the value matches that of the current CT-zone pipe, then CT is known
 * to have already been executed. The packet is thus immediately forwarded to
 * post-CT. Post-CT contains the rest of the original megaflow that was not
 * used in pre-CT.
 *
 * If this ct_zone match fails, then either CT was never executed, or
 * it was executed in a different CT zone. If it matches the currently
 * supported CT (network x protocol) tuple, then its ct_zone is set and
 * it is forwarded to the corresponding CT pipe. If no (network x protocol)
 * tuple matches, then CT is not supported for this flow and the packet
 * goes to software.
 *
 * The CT pipe is a basic pipe with a single action type, which writes to
 *
 *  * The packet registers used for CT metadata.
 *  * The packet 5-tuple.
 *
 * For plain CT, the 5-tuple is overwritten with its own values.
 * For NAT, the translations are written instead where relevant.
 *
 * In both cases, all fields are written anyway.
 * This way, the number of template used by the CT pipe is minimal.
 * During performance tests, no impact was measured due to the
 * superfluous writes.
 *
 * If a CT entry matches the packet, the CT pipe action is executed
 * and the packet is then forwarded to post-CT. Otherwise, the packet
 * goes to the sw-meter pipe to execute miss handling.
 *
 * IPv6 Connection Tracking Implementation
 * =======================================
 *
 * By default IPv6 connection offloading is disabled.
 * Set 'other_config:hw-offload-ct-ipv6-enabled=true' to enable.
 *
 * The diagram below shows how IPv6 connection tracking is implemented in the
 * hardware datapath. IPv6 CT rules are too large to fit into single steering
 * hardware objects (STE) and must be split.
 *
 * Note:
 * This is HW specific.
 * In BF3 and above, jumbo STE is supported and the rules can match in a
 * single STE. However, as OVS is HW agnostic, and to support < BF3 cards,
 * this split is done.
 *
 * The IPv6 tuple is divided into a prefix and a suffix, each inserted into
 * their respective pipe. A packet has to match both prefix a suffix rules
 * for a complete IPv6 5-tuple match and continue into the common post-CT pipe.
 *
 * This implementation complements and integrates with the IPv4 model.
 * The distinction is made during the CT-zone stage, matching on L3 protocol
 * to steer the packet toward the relevant basic pipe. Beside splitting match
 * into two, the same logic applies as for the IPv4 implementation.
 *
 *               +-----------------------------------------------+
 *               |IPv6 CT basic pipes                            |
 *               |                         +-[CT suffix TCP]-+   |
 *               |                         |                 |   |
 *               |                         +-----------------+   |
 *               |                         |prefix_id +      |   |
 *               |                         |IPv6.dst +       +-+ |
 *               |                         |TCP ports        | | |
 *               | +[CT prefix pipe]+      +-----------------+ | |
 *+[CT Zone X]+  | |                |      |                 | | |
 *|           |  | |                | +--->|                 | | |
 *|           |  | |                | |    |                 | | |
 *|           |  | |+-------------+ | |    |  +------------+ | | |
 *|+---------+|  | ||IPv6.src+TCP +-+-+ +--+--+ Miss fwd   | | | | +-[Post CT]+
 *||IPv6+UDP ++--+>|+-------------+ |   |  |  +------------+ | | | |          |
 *|+---------+|  | ||IPv6.src+UDP +-+-+ |  |                 | | | |          |
 *||IPv6+TCP ++--+>|+-------------+ | | |  +-----------------+ | | |          |
 *|+---------+|  | |                | | |                      +-+>|          |
 *|           |  | |                | | |  +-[CT suffix UDP]-+ | | |          |
 *|           |  | |                | +-+->|                 | | | |          |
 *|           |  | |+-------------+ |   |  |                 | | | |          |
 *|           |  | ||  Miss fwd   | |   |  +-----------------+ | | |          |
 *|           |  | |+-------+-----+ |   |  |Prefix_id +      | | | |          |
 *|           |  | |        |       |   |  |IPv6.dst +       +-+ | |          |
 *+-----------+  | |        |       |   |  |UDP ports        |   | |          |
 *               | +--------+-------+   |  +-----------------+   | +----------+
 *               |          |           |  |  +------------+ |   |
 *               |          |           | ++--+ Miss fwd   | |   |
 *               |          |           | ||  +------------+ |   |
 *               |          |           | ||                 |   |
 *               |          |           | |+-----------------+   |
 *               +----------+-----------+-+----------------------+
 *                          |           | |
 *                          |    +------v-v---------+
 *                          +--->|     Miss Pipe    |
 *                               +------------------+
 *
 * Meter action post processing
 * ============================
 *
 * +-[pre-CT]------+
 * |               |
 * | +-[CT-zones]--+--+
 * | |                |
 * | | +-[post-CT]----+-+         +-[normal-tables]-+        +-[POST-METER]-------+
 * | | |                |         |                 |        |                    |
 * | | |                |         |                 |        |  +--------------+  |
 * | | |                |         |       ...       |    +---+->|action=meter2 +--+--+
 * | | |                |         |                 |    |   |  +--------------+  |  |
 * | | |                |         | +------------+  |    |   |        ...         |  |
 * | | |                +-------->| |action=meter+--+----+   |  +--------------+  |  |
 * | | |                |         | +------------+  |        |  |action=meterN |<-+--+
 * +-+ |                |         |                 |        |  +-------+------+  |
 *   | |                |         |       ...       |        |          |         |      +------+
 *   +-+                |         |                 |        |          +---------+----->| port |
 *     |                |         |                 |        |                    |      +------+
 *     +----------------+         +-----------------+        +--------------------+
 *
 * Single meter action forwards to the POSTMETER_TABLE where matching on
 * red/green color is done and green packets get forwarded to the real
 * destination of the original flow.
 *
 * If there are more then one meter action, then there is a loop over
 * POSTMETER_TABLE to perform second meter action, then third meter action and
 * so on. Each iteration drops red packets and forwards green packets to the
 * next meter in POSTMETER_TABLE. When last meter action is reached, the next
 * match in POSTMETER_TABLE forwards green packets to the real destination.
 *
 * Multiple split flow
 * ===================
 *
 * If an original flow is too big to fit in a single STE, split the matches of
 * such flow to up to 10 different flows, in which each split flow takes some of
 * the original flow's matches. each match is mapped to an id "prefix_id" which
 * is then matched in the following split flow.
 *
 * Notes:
 * If an original flow has a tunnel header match, the split is done forcefully
 * over the outer header first then the inner.
 *
 * +-[split_depth(0)]---+    +-[split_depth(1)]---+       +-[split_depth(n)]---+
 * |                    |    |                    |       |                    |
 * |match:              |    |match:              |       |match:              |
 * | M(0)               |    | M(1)               |       | M(n)               |
 * |                    |    | prefix_id(1)       |       | prefix_id(n-1)     |
 * |                    |    |                    |       |                    |
 * |actions:            +--->|actions:            | . . . |actions:            |
 * | set prefix_id(1)   |    | set prefix_id(2)   |       | orig_actions       |
 * | jump split_depth(1)|    | jump split_depth(2)|       |                    |
 * |                    |    |                    |       |                    |
 * +--------------------+    +--------------------+       +--------------------+
 *
 * The diagrams were initially drawn with https://asciiflow.com/ and edited in VIM.
 * The resulting extended ASCII chars should however be avoided.
 *
 * Miss handling
 * =============
 *
 * The hardware datapath is magnitudes faster than its software
 * counterpart and is capable of flooding it in some occasion.
 * The exception path sending packets from hardware to software
 * datapath implements a flood protection in the form of a per-port
 * software rate-limit.
 *
 * It is implemented on each eswitch as follows:
 *
 *     .-[ Control ]---.     .-[ Basic ]-----.
 *     |               |     |               |
 *     `-------+-------'     `-------+-------'
 *             |miss                 |miss
 *             `----------+----------'
 *                        |
 *   .-[ pre-miss ]-------V------------------------.
 *   |    .--------------------------------------. |
 *   |    | match eth_type 0x8809 | fwd to kernel| |
 *   |    | match eth_type 0x88cc | fwd to kernel| |
 *   |    | match eth_type 0x888e | fwd to kernel| |
 *   |    `--------------------------------------' |
 *   |      .-------------.                        |
 *   |  miss|  Catch-all  +--,                     |
 *   |      `-------------'  |                     |
 *   `-----------------------|---------------------'
 *   .-[ port-meter ]--------v----.  .-[ post-port-meter ]------------------.
 *   |     Match      Action      |  |                                      |
 *   |    .---------------------. |  |    .-------------------------------. |
 *   |    |Port 0   | Meter     +-------->| doca-color | PORT_METER_COLOR | |
 *   |    `---------------------' |  |    `-------------------------------' |
 *   |    .---------------------. |  |    .-------------------------------. |
 *   |    |Port N   | Meter     +-------->| doca-color | PORT_METER_COLOR | |
 *   |    `---------------------' |  |    `-------------------------------' |
 *   `----------------------------'  `--------------------v-----------------'
 *                           +-------- miss --------------+
 *   .-[ core-meter ]--------v----.  .-[ post-core-meter ]---------------.
 *   |     Match      Action      |  |                                   |
 *   |  .-------------------.     |  |    .--------------------.         |
 *   |  | Catch-all | Meter +------------>| RED + RED | drop   |         |
 *   |  `-----------+-------'     |  |    `--------------------'         |
 *   `----------------------------'  `--------------------v-----------------'
 *                           +-------- miss --------------+
 *   .-[ sw-meter ]----------v----.  .-[ post-sw-meter ]-----------------.
 *   |     Match      Action      |  |                                   |
 *   |    .---------------------. |  |    .----------------------------. |
 *   |    |Port 0   | Meter     +-------->| Port 0 + RED | count, drop | |
 *   |    `---------------------' |  |    `----------------------------' |
 *   |    .---------------------. |  |    .----------------------------. |
 *   |    |Port N   | Meter     +-------->| Port N + RED | count, drop | |
 *   |    `---------------------' |  |    `----------------------------' |
 *   |    .------------------.    |  |    .-------------.                |
 *   |miss|    Catch-all     |    |  |miss|  Catch-all  +--,             |
 *   |    `------------+-----'    |  |    `-------------'  |             |
 *   `-----------------|----------'  `---------------------|-------------'
 *   .-[ meta-push ]---v-----------------------------------v-------------.
 *   |                  .------------------------------.                 |
 *   |                  |    Catch-all  | push vxlan   +--,              |
 *   |                  `------------------------------'  |              |
 *   `----------------------------------------------------|--------------'
 *   .-[ meta-copy ]--------------------------------------v----.
 *   |                  .------------------------------.       |
 *   |                  |    Catch-all  | copy tags    +-------+--,
 *   |                  `------------------------------'       |  |
 *   `---------------------------------------------------------'  |
 *   .-[ tag0 ]------------------------------------------------+  |
 *   |                                                         +<-+
 *   |                  .------------------------------.       |
 *   |                  |    Catch-all  | zero tag0    +--,    |
 *   |                  `------------------------------'  |    |
 *   `----------------------------------------------------|----'
 *   .-[ miss ]-----------------------------------.       |
 *   |     < RSS pipe >                           |<------+
 *   `--------------------------------------------'
 *
 * All pipe types that are expected to miss (control and basic) are configured
 * with a miss rule forwarding to the kernel pipe which will match on LACP,
 * LLDP or 802.1X ether types, and all other packets will be
 * forwarded to the sw-meter pipe.
 *
 * The sw-meter pipe is itself configured with a miss directive steering
 * toward the meta-push pipe. If a software meter is configured on a port,
 * a metering rule is inserted in the sw-meter pipe matching that specific
 * port. This rule avoids the miss config and apply the requested rate
 * limiting, before forwarding the packet to the post-sw-meter pipe.
 *
 * There, if the color RED is set on the packet, it matches a rule
 * that counts and drop that packet. Otherwise, if the packet was marked
 * GREEN, no rule matches and the catch-all miss directive applies,
 * forwarding the packet back to the meta-push pipe.
 *
 * The meta-push pipe itself is configured with a catch all rule to push a VXLAN
 * header to the packet which then is forwarded toward the meta-copy pipe.
 *
 * The meta-copy pipe contains a catch all rule which copies tag registers that are
 * not propagated to SW otherwise into the VXLAN header.
 *
 * The zero-tag0 pipe zeroes TAG0, as a doca has internal use of it for
 * send-to-kernel.
 *
 * The RSS pipe does RSS according to packet types moving the packet to the RX
 * queues.
 *
 * Sample handling
 * ===============
 *
 * Packet sampling is not active by default and doesn't have any overhead. Only
 * when "sflow" option is added to a bridge the sampling pipeline becomes
 * active. It is implemented on each eswitch as follows:
 *
 * +---------------------------+
 * |                           |
 * |         root pipe         |                    Normal
 * |                           +---------------->  datapath
 * |                           |                  processing
 * |                           |
 * |            ...            |   set NONSAMPLE_FLOW_MARK
 * |                           |<----------------------------+
 * |                           |                             | 99%
 * | +-----------------------+ |                 +-----------+-----------+
 * | |                       | |                 |                       |
 * | | prio0: FLOW_INFO == 0 +-+---------------->|      sample pipe      |
 * | |                       | |                 |                       |
 * | +-----------------------+ |                 +-----------+-----------+
 * |                           |                             |
 * +---------------------------+     set NONSAMPLE_FLOW_MARK | 1%
 *                      ^                                    v
 *                      |                           +-----------------+
 *                      |                original   |                 |
 *                      +---------------------------+   mirror pipe   |
 *                                                  |                 |
 *                                                  +--------+--------+
 *                                                           |
 *                                                           | clone
 *                                                           v
 *               +---------------+                  +------------------+
 *               |               |                  |                  |
 *               | misspath pipe |<-----------------+ post-mirror pipe |
 *               |               |       set        |                  |
 *               +---------------+ SAMPLE_FLOW_MARK +------------------+
 *
 * The sample, mirror and post-mirror pipes are created at eswitch
 * initialization. They're not active however, until the first rule that has
 * "sample" action is offloaded. When it happens, a prio0 entry is added to root
 * pipe that forwards all packets w/o FLOW_INFO metadata to the sample pipe.
 * Then the packets are forwarded back to the root pipe and they continue normal
 * processing - they miss prio0 entry because they have FLOW_INFO set.
 *
 * Some percentage of packets is mirrored to the slow path where they will be
 * marked with a "sample" flag in the HW recovery handler. Such packets will
 * continue datapath processing in software and will be dropped eventually when
 * "sample" action is reached.
 *
 * The rest of the "normal" offloads in the root pipe have their original
 * priority lowered to make prio0 reserved for this special rule that does not
 * exist by default. This is why when sampling is disabled it doesn't have any
 * overhead.
 */

COVERAGE_DEFINE_ERR(doca_invalid_meta_header);
COVERAGE_DEFINE(doca_queue_empty);
COVERAGE_DEFINE_WARN(doca_queue_none_processed);
COVERAGE_DEFINE_WARN(doca_resize_block);

#define BE16_MAX ((OVS_FORCE doca_be16_t) 0xffff)
#define BE32_MAX ((OVS_FORCE doca_be32_t) 0xffffffff)

#define MAX_OFFLOAD_QUEUE_NB MAX_OFFLOAD_THREAD_NB
#define MAX_GENEVE_OPT 1

#define MIN_SHARED_MTR_FLOW_ID 1

#define META_PUSH_VXLAN_UDP_PORT 4789
#define META_PUSH_VNI 0xdead

/* The maximum length of the pipe name as defined in doca_flow */
#define MAX_PIPE_NAME_LEN 128

/* Calculate the bit offset of a field from the end of its containing type. */
#define REV_BIT_OFFSETOF(STRUCT, FIELD) \
    (8 * (sizeof(STRUCT) - offsetof(STRUCT, FIELD) - MEMBER_SIZEOF(STRUCT, FIELD)))

VLOG_DEFINE_THIS_MODULE(dpdk_offload_doca);
static struct vlog_rate_limit rl = VLOG_RATE_LIMIT_INIT(600, 600);

static int
dpdk_offload_doca_destroy__(struct netdev *netdev,
                            struct dpdk_offload_handle *doh,
                            struct rte_flow_error *error);

#define SPLIT_FIELD(field, type, proto_type, proto_field) \
                    {type, proto_type, \
                     offsetof(struct doca_flow_header_format, field), \
                     MEMBER_SIZEOF(struct doca_flow_header_format, field), \
                     offsetof(struct doca_flow_header_format, proto_field), \
                     MEMBER_SIZEOF(struct doca_flow_header_format, proto_field)}

enum split_field_type {
    FIELD_TYPE_INVALID,
    FIELD_TYPE_NONE,
    FIELD_TYPE_SRC,
    FIELD_TYPE_DST,
};

struct split_field {
    int type;
    int proto_type;
    size_t offset;
    size_t size;
    size_t proto_offset;
    size_t proto_size;
};

enum split_l2_field_names {
    FIELD_ETH_MAC_SRC,
    FIELD_ETH_MAC_DST,
    FIELD_ETH_TYPE,
    FIELD_ETH_VLAN_TCI,
    NUM_L2_FIELDS,
};

enum split_l3_field_names {
    FIELD_L3_TYPE,
    FIELD_IP4_SRC,
    FIELD_IP4_DST,
    FIELD_IP4_VER_IHL,
    FIELD_IP4_DSCP_ECN,
    FIELD_IP4_NXT_PROTO,
    FIELD_IP4_TTL,
    FIELD_IP6_SRC,
    FIELD_IP6_DST,
    FIELD_IP6_DSCP_ECN,
    FIELD_IP6_NXT_PROTO,
    FIELD_IP6_HOP_LIMIT,
    NUM_L3_FIELDS,
};

enum split_l4_field_names {
    FIELD_L4_TYPE,
    FIELD_UDP_SRC,
    FIELD_UDP_DSR,
    FIELD_TCP_SRC,
    FIELD_TCP_DST,
    FIELD_TCP_FLAGS,
    FIELD_ICMP_TYPE,
    FIELD_ICMP_CODE,
    NUM_L4_FIELDS,
};

enum split_field_layer {
    L2_HEADERS,
    L3_HEADERS,
    L4_HEADERS,
};

static struct split_field split_fields[][NUM_L3_FIELDS] = {
    [L2_HEADERS] = {
        SPLIT_FIELD(eth.src_mac, FIELD_TYPE_SRC, 0, l2_valid_headers),
        SPLIT_FIELD(eth.dst_mac, FIELD_TYPE_DST, 0, l2_valid_headers),
        SPLIT_FIELD(eth.type, FIELD_TYPE_NONE, 0, l2_valid_headers),
        SPLIT_FIELD(eth_vlan[0].tci, FIELD_TYPE_NONE, 0, l2_valid_headers),
    },
    [L3_HEADERS] = {
        SPLIT_FIELD(ip4.dst_ip, FIELD_TYPE_DST, DOCA_FLOW_L3_TYPE_IP4, l3_type),
        SPLIT_FIELD(ip4.src_ip, FIELD_TYPE_SRC, DOCA_FLOW_L3_TYPE_IP4, l3_type),
        SPLIT_FIELD(ip4.version_ihl, FIELD_TYPE_NONE, DOCA_FLOW_L3_TYPE_IP4, l3_type),
        SPLIT_FIELD(ip4.dscp_ecn, FIELD_TYPE_NONE, DOCA_FLOW_L3_TYPE_IP4, l3_type),
        SPLIT_FIELD(ip4.next_proto, FIELD_TYPE_NONE, DOCA_FLOW_L3_TYPE_IP4, l3_type),
        SPLIT_FIELD(ip4.ttl, FIELD_TYPE_NONE, DOCA_FLOW_L3_TYPE_IP4, l3_type),
        SPLIT_FIELD(ip6.dst_ip, FIELD_TYPE_DST, DOCA_FLOW_L3_TYPE_IP6, l3_type),
        SPLIT_FIELD(ip6.src_ip, FIELD_TYPE_SRC, DOCA_FLOW_L3_TYPE_IP6, l3_type),
        SPLIT_FIELD(ip6.traffic_class, FIELD_TYPE_NONE, DOCA_FLOW_L3_TYPE_IP6, l3_type),
        SPLIT_FIELD(ip6.next_proto, FIELD_TYPE_NONE, DOCA_FLOW_L3_TYPE_IP6, l3_type),
        SPLIT_FIELD(ip6.hop_limit, FIELD_TYPE_NONE, DOCA_FLOW_L3_TYPE_IP6, l3_type),
    },
    [L4_HEADERS] = {
        SPLIT_FIELD(udp.l4_port.src_port, FIELD_TYPE_SRC,
                    DOCA_FLOW_L4_TYPE_EXT_UDP, l4_type_ext),
        SPLIT_FIELD(udp.l4_port.dst_port, FIELD_TYPE_DST,
                    DOCA_FLOW_L4_TYPE_EXT_UDP, l4_type_ext),
        SPLIT_FIELD(tcp.l4_port.src_port, FIELD_TYPE_SRC,
                    DOCA_FLOW_L4_TYPE_EXT_TCP, l4_type_ext),
        SPLIT_FIELD(tcp.l4_port.dst_port, FIELD_TYPE_DST,
                    DOCA_FLOW_L4_TYPE_EXT_TCP, l4_type_ext),
        SPLIT_FIELD(tcp.flags, FIELD_TYPE_NONE, DOCA_FLOW_L4_TYPE_EXT_TCP, l4_type_ext),
        SPLIT_FIELD(icmp.type, FIELD_TYPE_NONE, DOCA_FLOW_L4_TYPE_EXT_ICMP, l4_type_ext),
        SPLIT_FIELD(icmp.code, FIELD_TYPE_NONE, DOCA_FLOW_L4_TYPE_EXT_ICMP, l4_type_ext),
        SPLIT_FIELD(icmp.type, FIELD_TYPE_NONE, DOCA_FLOW_L4_TYPE_EXT_ICMP6, l4_type_ext),
        SPLIT_FIELD(icmp.code, FIELD_TYPE_NONE, DOCA_FLOW_L4_TYPE_EXT_ICMP6, l4_type_ext),
    },
};

enum ct_fwd_type {
    CT_FWD_POST,
    CT_FWD_MISS,
    NUM_CT_FWD,
};

enum ovs_port_meter_color {
    OVS_PORT_METER_COLOR_RED,
    OVS_PORT_METER_COLOR_YELLOW,
    OVS_PORT_METER_COLOR_GREEN,
    NUM_OVS_PORT_METER_COLOR,
};

enum ct_nw_type {
    CT_NW_IP4, /* CT on IPv4 networks. */
    CT_NW_IP6, /* CT on IPv6 networks. */
    NUM_CT_NW,
};

enum ct_tp_type {
    CT_TP_UDP, /* CT on UDP datagrams. */
    CT_TP_TCP, /* CT on TCP streams. */
    NUM_CT_TP,
};

/* DOCA-CT templates are defined as:
 * [0]: Plain CT (IPv4 or IPv6)
 * [1]: CT-NAT IPv4
 * [2]: CT-NAT IPv6
 *
 * If Plain-CT, action template [0] is always used.
 * If CT-NAT, action template [1] + [NW] is used.
 */
#define CT_ACTIONS_IDX(NW, NAT) (!!(NAT) * (!!(NAT) + (NW)))
#define NUM_CT_ACTIONS (1 + NUM_CT_NW)

enum hash_pipe_type {
    HASH_TYPE_IPV4_UDP,
    HASH_TYPE_IPV4_TCP,
    HASH_TYPE_IPV4_L3,
    HASH_TYPE_IPV6_UDP_SUF,
    HASH_TYPE_IPV6_TCP_SUF,
    HASH_TYPE_IPV6_L3_SUF,
    HASH_TYPE_IPV6_UDP_PRE,
    HASH_TYPE_IPV6_TCP_PRE,
    HASH_TYPE_IPV6_L3_PRE,
    HASH_TYPE_IPV6_L3_JUMBO,
    HASH_TYPE_IPV6_UDP_JUMBO,
    HASH_TYPE_IPV6_TCP_JUMBO,
    NUM_HASH_PIPE_TYPE,
};
/* The SUF types must be in lower positions than the PRE ones, as this is the
 * order of initialization, and release in the opposite order.
 */
BUILD_ASSERT_DECL(HASH_TYPE_IPV6_UDP_SUF < HASH_TYPE_IPV6_UDP_PRE);
BUILD_ASSERT_DECL(HASH_TYPE_IPV6_TCP_SUF < HASH_TYPE_IPV6_TCP_PRE);
BUILD_ASSERT_DECL(HASH_TYPE_IPV6_L3_SUF < HASH_TYPE_IPV6_L3_PRE);

#define HASH_PIPE_TYPE_IS_JUMBO(type)        \
    (((type) == HASH_TYPE_IPV6_L3_JUMBO)  || \
     ((type) == HASH_TYPE_IPV6_UDP_JUMBO) || \
     ((type) == HASH_TYPE_IPV6_TCP_JUMBO))

struct doca_basic_pipe_ctx {
    struct doca_flow_pipe *pipe;
    struct doca_pipe_group_ctx *fwd_group_ctx;
    struct doca_pipe_group_ctx *miss_group_ctx;
};

/* +--------+   +-------------+
 * |IPv4-UDP|-->|HASH-IPv4-UDP|
 * |        |   +-------------+
 * |        |   +-------------+
 * |IPv4-TCP|-->|HASH-IPv4-TCP|
 * +---+----+   +-------------+
 *     | miss   +-------------+
 *     +------->|HASH-IPv4-L3 |
 *              +-------------+
 * +--------+   +-----------------+   +-----------------+
 * |IPv6-UDP|-->|HASH-IPv6-UDP_PRE|-->|HASH-IPv6-UDP_SUF|
 * |        |   +-----------------+   +-----------------+
 * |        |   +-----------------+   +-----------------+
 * |IPv6-TCP|-->|HASH-IPv6-TCP_PRE|-->|HASH-IPv6-TCP_SUF|
 * +---+----+   +-----------------+   +-----------------+
 *     | miss   +-----------------+   +-----------------+
 *     +------->|HASH-IPv6-L3_PRE |-->|HASH-IPv6-L3_SUF |
 *              +-----------------+   +-----------------+
 * OVS always matches on ether type.
 * We only need to know TCP/UDP, or miss to simple L3.
 * With IPv6, full IP match cannot fit in a single STE, thus a split is done.
 * In PRE, the SRC IP/port are hashed. In SUF, the hash result from the PRE
 * and the DST IP/port are hashed, to create the final hash.
 */
enum hash_tp_type {
    HASH_TP_UDP,
    HASH_TP_TCP,
    NUM_HASH_TP,
};

enum hash_nw_type {
    HASH_NW_IP4, /* CT on IPv4 networks. */
    HASH_NW_IP6, /* CT on IPv6 networks. */
    NUM_HASH_NW,
};

struct doca_meta_transfer_ctx {
    struct doca_basic_pipe_ctx meta_push_pipe_ctx;
    struct doca_flow_pipe_entry *meta_push_hdr;
    struct doca_basic_pipe_ctx meta_copy_pipe_ctx;
    struct doca_flow_pipe_entry *meta_copy_to_hdr;
    struct doca_basic_pipe_ctx meta_tag0_pipe_ctx;
    struct doca_flow_pipe_entry *meta_tag0_hdr;
};

struct doca_shared_encaps_ctx {
    struct refmap *rfm;
    struct atomic_count count;
};

struct gpr_meter_id {
    bool valid;
    uint32_t id;
};

struct doca_shared_meters_ctx {
    struct refmap *rfm;
    struct gpr_meter_id gpr_meter_ids[OVS_DOCA_MAX_GPR_METERS_PER_ESW];
};

struct doca_hash_pipe_ctx {
    struct {
        struct doca_flow_pipe *pipe;
        struct doca_flow_pipe_entry *entry;
    } hashes[NUM_HASH_PIPE_TYPE];
    struct {
        struct doca_flow_pipe *pipe;
        struct doca_flow_pipe_entry *tcpudp[NUM_HASH_TP];
    } classifier[NUM_HASH_NW];
    struct doca_pipe_group_ctx *post_hash_group_ctx;
};

struct gnv_opt_parser {
    struct doca_flow_parser *parser;
};

struct ct_zone_ctx {
    OVSRCU_TYPE(struct doca_flow_pipe *) plain_pipe;
    OVSRCU_TYPE(struct doca_flow_pipe *) nat_pipe;
    struct dpdk_offload_handle entries[2][CT_ZONE_FLOWS_NUM];
};

enum pre_miss_types {
    SEND_TO_KERNEL_LACP,
    SEND_TO_KERNEL_LLDP,
    SEND_TO_KEREL_802_1X,
    NUM_SEND_TO_KERNEL,
};

static uint16_t pre_miss_mapping[NUM_SEND_TO_KERNEL] = {
    [SEND_TO_KERNEL_LACP] = ETH_TYPE_LACP,
    [SEND_TO_KERNEL_LLDP] = ETH_TYPE_LLDP,
    [SEND_TO_KEREL_802_1X] = ETH_TYPE_802_1X,
};

#define OVS_DOCA_MAX_PRE_MISS_RULES 16

struct doca_pre_miss_rule {
    uint16_t eth_type;
    struct doca_flow_pipe_entry *entry;
};

struct doca_pre_miss_ctx {
    struct doca_basic_pipe_ctx pre_miss_pipe_ctx;
    struct doca_pre_miss_rule pre_miss_rules[OVS_DOCA_MAX_PRE_MISS_RULES];
};

struct doca_sample_ctx {
    struct doca_pipe_group_mask_entry root_sample_entry_mctx;
    struct ovs_refcount root_sample_entry_refcount;
    struct doca_pipe_group_ctx *sample_pipe_ctx;
    struct doca_pipe_group_mask_entry sample_pipe_entry_mctx;
    struct doca_pipe_group_mask_entry sample_pipe_miss_entry_mctx;
    struct doca_basic_pipe_ctx mirror_pipe_ctx;
    struct doca_flow_pipe_entry *mirror_pipe_entry;
    struct doca_pipe_group_ctx *postmirror_pipe_ctx;
    struct doca_pipe_group_mask_entry postmirror_pipe_entry_mctx;
    struct doca_pipe_group_mask_entry postmirror_drop[NUM_SEND_TO_KERNEL];
    uint32_t ratio;
};

struct doca_port_meter_ctx {
    struct doca_flow_pipe *pipe;
    struct doca_flow_pipe_entry *entry[OVS_DOCA_MAX_PORT_METERS_PER_ESW];
};

struct doca_post_port_meter_ctx {
    struct doca_flow_pipe *pipe;
    struct doca_flow_pipe_entry *entry[NUM_OVS_PORT_METER_COLOR];
};

struct doca_fixed_gpr_ctx {
    struct doca_flow_pipe *pipe;
    struct doca_flow_pipe_entry *entry;
};

struct doca_offload_esw_ctx {
    struct doca_flow_port *esw_port;
    struct netdev *esw_netdev;
    struct doca_basic_pipe_ctx post_sw_meter_pipe_ctx;
    struct doca_basic_pipe_ctx sw_meter_pipe_ctx;
    struct doca_fixed_gpr_ctx core_meter_pipe_ctx;
    struct doca_port_meter_ctx port_meter_pipe_ctx;
    struct doca_post_port_meter_ctx post_port_meter_pipe_ctx;
    struct doca_fixed_gpr_ctx post_core_meter_pipe_ctx;
    struct doca_meta_transfer_ctx meta_transfer_ctx;
    struct doca_pre_miss_ctx pre_miss_ctx;
    struct doca_pipe_group_ctx *root_pipe_group_ctx;
    struct gnv_opt_parser gnv_opt_parser;
    struct {
        struct doca_flow_pipe *pipe;
        struct doca_pipe_group_ctx *post_ct_ctx;
        uint32_t fwd_handles[NUM_CT_FWD];
    } ct;
    struct doca_basic_pipe_ctx ct_pipes[NUM_CT_NW][NUM_CT_TP];
    struct ovs_mutex ct_zones_mutex;
    struct ct_zone_ctx ct_zones[NUM_CT_ZONES];
    struct doca_basic_pipe_ctx ct_ip6_prefix;
    struct id_fpool *shared_mtr_flow_id_pool;
    uint32_t esw_id;
    struct ovs_refcount pipe_resizing;
    struct ovs_list resized_pipe_lists[MAX_OFFLOAD_QUEUE_NB];
    struct doca_pipe_group_ctx *post_meter_pipe_group_ctx;
    struct doca_hash_pipe_ctx *hash_pipe_ctx;
    struct doca_sample_ctx sample_ctx;
    struct ovs_doca_offload_queue *offload_queues;
    struct ovs_list destroy_pipe_lists[MAX_OFFLOAD_QUEUE_NB];
    struct too_big_map too_big_map;
    struct doca_mirrors *doca_mirrors;
    struct doca_mirror_ctx *sflow_mirror_ctx;
    struct doca_shared_encaps_ctx shared_encaps_ctx;
    struct doca_shared_meters_ctx shared_meters_ctx;
};

static bool
doca_flow_ct_shared_actions_create(struct doca_offload_esw_ctx *esw,
                                   unsigned int tid,
                                   struct shared_ct_actions_key *key,
                                   uint32_t *id, struct rte_flow_error *error);

static void
doca_flow_ct_shared_actions_destroy(struct doca_offload_esw_ctx *esw,
                                    uint32_t id);

static int
doca_offload_basic_pipe_create(struct netdev *netdev,
                               struct ovs_doca_flow_match *match,
                               struct ovs_doca_flow_match *match_mask,
                               struct doca_flow_monitor *monitor,
                               struct ovs_doca_flow_actions *actions,
                               struct ovs_doca_flow_actions *actions_mask,
                               struct doca_flow_action_descs *descs,
                               struct doca_flow_fwd *fwd,
                               struct doca_flow_fwd *fwd_miss,
                               uint32_t nr_entries,
                               uint64_t queues_bitmap,
                               const char *pipe_type,
                               struct doca_flow_pipe **pipe)
{
    struct doca_flow_actions *actions_arr[1], *actions_masks_arr[1];
    struct doca_flow_action_descs *descs_arr[1];
    char pipe_name[MAX_PIPE_NAME_LEN];
    struct doca_flow_port *doca_port;
    struct doca_flow_pipe_cfg *cfg;
    uint32_t esw_id;
    int ret;

    ovs_assert(*pipe == NULL);

    doca_port = netdev_doca_port_get(netdev);
    esw_id = netdev_doca_get_esw_mgr_port_id(netdev);

    snprintf(pipe_name, sizeof pipe_name, "%s_OVS_%s_PIPE_%"PRIu32, netdev_get_name(netdev),
             pipe_type, esw_id);

    ret = doca_flow_pipe_cfg_create(&cfg, doca_flow_port_switch_get(doca_port));
    if (ret) {
        VLOG_ERR("%s: Could not create doca_flow_pipe_cfg for %s",
                 netdev_get_name(netdev), pipe_name);
        return ret;
    }
    actions_arr[0] = &actions->d;
    actions_masks_arr[0] = &actions_mask->d;
    descs_arr[0] = descs;

    if (doca_flow_pipe_cfg_set_name(cfg, pipe_name) ||
        doca_flow_pipe_cfg_set_type(cfg, DOCA_FLOW_PIPE_BASIC) ||
        doca_flow_pipe_cfg_set_nr_entries(cfg, nr_entries) ||
        doca_pipe_cfg_allow_queues(cfg, queues_bitmap) ||
        (match && doca_flow_pipe_cfg_set_match(cfg, &match->d,
                                               match_mask ? &match_mask->d : &match->d)) ||
        (monitor && doca_flow_pipe_cfg_set_monitor(cfg, monitor)) ||
        (actions && doca_flow_pipe_cfg_set_actions(cfg, actions_arr,
                                                   actions_mask ? actions_masks_arr : actions_arr,
                                                   descs ? descs_arr : NULL, 1))) {
        VLOG_ERR("%s: Could not set doca_flow_pipe_cfg for %s",
                 netdev_get_name(netdev), pipe_name);
        ret = -1;
        goto error;
    }

    ret = doca_flow_pipe_create(cfg, fwd, fwd_miss, pipe);
    if (ret) {
        VLOG_ERR("%s: Failed to create basic pipe '%s': %d (%s)",
                 netdev_get_name(netdev), pipe_name, ret,
                 doca_error_get_descr(ret));
    }

error:
    doca_flow_pipe_cfg_destroy(cfg);
    return ret;
}

struct doca_flow_pipe *
doca_offload_get_misspath_pipe(struct doca_offload_esw_ctx *esw_ctx)
{
    /* Get the start of the miss-path handling. */
    return esw_ctx->pre_miss_ctx.pre_miss_pipe_ctx.pipe;
}

void
doca_offload_esw_ctx_pipe_resizing_ref(struct doca_offload_esw_ctx *esw_ctx)
{
    ovs_refcount_ref(&esw_ctx->pipe_resizing);
}

void
doca_offload_esw_ctx_pipe_resizing_unref(struct doca_offload_esw_ctx *esw_ctx)
{
    ovs_refcount_unref(&esw_ctx->pipe_resizing);
}

void
doca_offload_esw_ctx_resized_list_add(struct doca_offload_esw_ctx *esw_ctx,
                                      struct ovs_list *resized_list_node)
{
    unsigned int tid = netdev_offload_thread_id();

    ovs_list_push_back(&esw_ctx->resized_pipe_lists[tid], resized_list_node);
}

struct meter_info {
    struct ovs_list list_node;
    uint32_t id;
    uint32_t of_id;
    uint32_t red_id;
    uint32_t green_id;
    uint32_t flow_id;
};

struct doca_act_vars {
    uint32_t flow_id;
    uint32_t mtr_of_id;
    uint32_t mtr_flow_id;
    uint32_t mtr_red_cnt_id;
    uint32_t mtr_green_cnt_id;
    struct ovs_list next_meters;
    const struct hash_data *hash_data;
    uint32_t sample_ratio;
    enum doca_flow_tun_type shared_encap_tun_type;
    enum doca_flow_l3_type shared_encap_l3_type;
    ovs_be16 l3_type_hint;
    uint8_t l4_type_hint;
    bool physdev;
    uint32_t port_id_match;
};

struct doca_meter_ctx {
    struct ovs_list list_node;
    uint32_t of_meter_id;
    uint32_t post_meter_flow_id;
    struct doca_pipe_group_mask_entry post_meter_red_entry_mctx;
    struct doca_pipe_group_mask_entry post_meter_green_entry_mctx;
};

static int
destroy_dpdk_offload_handle__(struct doca_offload_esw_ctx *esw,
                              struct dpdk_offload_handle *doh,
                              unsigned int queue_id,
                              struct rte_flow_error *error);
static struct id_pool *esw_id_pool;

static int
shared_mtr_flow_id_alloc(struct doca_offload_esw_ctx *esw_ctx,
                         struct rte_flow_error *error);

static void
shared_mtr_flow_id_free(struct doca_offload_esw_ctx *esw_ctx, uint32_t id);

static int
doca_ct_zone_init(struct netdev *netdev, struct doca_offload_esw_ctx *esw_ctx,
                  uint32_t zone_group);

static inline enum ct_nw_type
l3_to_nw_type(enum doca_flow_l3_meta l3_type)
{
    switch (l3_type) {
    case DOCA_FLOW_L3_META_IPV4: return CT_NW_IP4;
    case DOCA_FLOW_L3_META_IPV6: return CT_NW_IP6;
    case DOCA_FLOW_L3_META_NONE: return NUM_CT_NW;
    };
    return NUM_CT_NW;
}

static inline enum ct_tp_type
l4_to_tp_type(enum doca_flow_l4_meta l4_type)
{
    switch (l4_type) {
    case DOCA_FLOW_L4_META_TCP: return CT_TP_TCP;
    case DOCA_FLOW_L4_META_UDP: return CT_TP_UDP;
    case DOCA_FLOW_L4_META_ICMP:
    case DOCA_FLOW_L4_META_ESP:
    case DOCA_FLOW_L4_META_NONE: return NUM_CT_TP;
    }
    return NUM_CT_TP;
}

#define MIN_SPLIT_PREFIX_ID 1
#define MAX_SPLIT_PREFIX_ID (reg_fields[REG_FIELD_SCRATCH].mask - 1)
#define NUM_SPLIT_PREFIX_ID (MAX_SPLIT_PREFIX_ID - MIN_SPLIT_PREFIX_ID + 1)
static struct refmap *split_prefix_rfm;
static struct id_fpool *split_prefix_id_pool = NULL;

OVS_ASSERT_PACKED(struct doca_split_prefix_key,
    struct ovs_doca_flow_match spec;
    struct doca_pipe_group_mask_ctx *prefix_mask_ctx;
    struct doca_flow_pipe *prefix_pipe;
);

struct doca_split_prefix_ctx {
    struct netdev *netdev;
    struct dpdk_offload_handle doh;
    uint32_t id;
    unsigned int queue_id;
    struct ovs_mutex handle_lock;
};

struct doca_split_prefix_arg {
    struct netdev *netdev;
    struct ovs_doca_flow_match *spec;
    bool prefix_is_group;
    struct doca_flow_pipe *prefix_pipe;
    struct doca_flow_pipe *suffix_pipe;
};

static doca_error_t
doca_offload_add_generic(unsigned int qid,
                         uint32_t hash_index,
                         uint32_t ctl_pipe_priority,
                         struct doca_flow_pipe *pipe,
                         enum doca_flow_pipe_type pipe_type,
                         const struct ovs_doca_flow_match *match,
                         const struct ovs_doca_flow_match *match_mask,
                         const struct ovs_doca_flow_actions *actions,
                         const struct ovs_doca_flow_actions *actions_mask,
                         const struct doca_flow_monitor *monitor,
                         const struct doca_flow_fwd *fwd,
                         uint32_t flags,
                         struct doca_offload_esw_ctx *esw,
                         struct doca_flow_pipe_entry **pentry)
    OVS_NO_THREAD_SAFETY_ANALYSIS
{
    struct ovs_doca_offload_queue *queues = esw ? esw->offload_queues : NULL;
    doca_error_t err;

    if (qid == AUX_QUEUE) {
        ovs_doca_mgmt_queue_lock();
    }

    switch (pipe_type) {
    case DOCA_FLOW_PIPE_BASIC:
        err = doca_flow_pipe_basic_add_entry(qid, pipe, &match->d, 0 ,&actions->d, monitor, fwd,
                                             flags, queues, pentry);
        break;
    case DOCA_FLOW_PIPE_CONTROL:
        err = doca_flow_pipe_control_add_entry(qid, pipe, &match->d,
                                               &match_mask->d, NULL, &actions->d,
                                               &actions_mask->d, NULL, monitor, ctl_pipe_priority,
                                               fwd, queues, pentry);
        break;
    case DOCA_FLOW_PIPE_HASH:
        err = doca_flow_pipe_hash_add_entry(qid, pipe, hash_index, 0, &actions->d, monitor, fwd,
                                            flags, queues, pentry);
        break;
    case DOCA_FLOW_PIPE_LPM:
    case DOCA_FLOW_PIPE_CT:
    case DOCA_FLOW_PIPE_ACL:
    case DOCA_FLOW_PIPE_ORDERED_LIST:
        OVS_NOT_REACHED();
    }

    if (qid == AUX_QUEUE) {
        ovs_doca_mgmt_queue_unlock();
    }
    if (err == DOCA_SUCCESS) {
        if (queues) {
            queues[qid].n_waiting_entries++;
        } else {
            VLOG_DBG("added entry %p without an eswitch handle", *pentry);
        }
    }
    return err;
}

doca_error_t
doca_offload_add_entry_hash(struct netdev *netdev,
                            unsigned int qid,
                            struct doca_flow_pipe *pipe,
                            uint32_t index,
                            const struct ovs_doca_flow_actions *actions,
                            const struct doca_flow_monitor *monitor,
                            const struct doca_flow_fwd *fwd,
                            uint32_t flags,
                            struct doca_flow_pipe_entry **pentry)
    OVS_EXCLUDED(mgmt_queue_lock)
{
    struct doca_offload_esw_ctx *esw = doca_offload_esw_ctx_get(netdev);
    doca_error_t err;

    err = doca_offload_add_generic(qid, index, 0, pipe, DOCA_FLOW_PIPE_HASH,
                                   NULL, NULL, actions, NULL, monitor, fwd, flags, esw, pentry);

    if (err) {
        VLOG_WARN_RL(&rl, "%s: Failed to create hash pipe entry. Error: %d (%s)",
                     netdev_get_name(netdev), err, doca_error_get_descr(err));
        return err;
    }

    if (DOCA_FLOW_FLAGS_IS_SET(flags, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT)) {
        err = doca_offload_complete_queue_esw(esw, qid, true);
        if (err != DOCA_SUCCESS) {
            return err;
        }
    }

    dpdk_offload_counter_inc(netdev);

    return err;
}

static doca_error_t
doca_offload_add_entry_control(unsigned int qid,
                               uint32_t priority,
                               struct doca_flow_pipe *pipe,
                               const struct ovs_doca_flow_match *match,
                               const struct ovs_doca_flow_match *match_mask,
                               const struct ovs_doca_flow_actions *actions,
                               const struct ovs_doca_flow_actions *actions_mask,
                               const struct doca_flow_monitor *monitor,
                               const struct doca_flow_fwd *fwd,
                               struct doca_offload_esw_ctx *esw,
                               struct doca_flow_pipe_entry **pentry)
    OVS_EXCLUDED(mgmt_queue_lock)
{
    return doca_offload_add_generic(qid, 0, priority, pipe, DOCA_FLOW_PIPE_CONTROL,
                                    match, match_mask, actions, actions_mask,
                                    monitor, fwd, 0, esw, pentry);
}

doca_error_t
doca_offload_add_entry(struct netdev *netdev,
                       unsigned int qid,
                       struct doca_flow_pipe *pipe,
                       const struct ovs_doca_flow_match *match,
                       const struct ovs_doca_flow_actions *actions,
                       const struct doca_flow_monitor *monitor,
                       const struct doca_flow_fwd *fwd,
                       uint32_t flags,
                       struct doca_flow_pipe_entry **pentry)
    OVS_EXCLUDED(mgmt_queue_lock)
{
    struct doca_offload_esw_ctx *esw = doca_offload_esw_ctx_get(netdev);
    doca_error_t err;

    err = doca_offload_add_generic(qid, 0, 0, pipe, DOCA_FLOW_PIPE_BASIC, match, NULL, actions,
                                   NULL, monitor, fwd, flags, esw, pentry);

    if (err) {
        VLOG_WARN_RL(&rl, "%s: Failed to create basic pipe entry. Error: %d (%s)",
                     netdev_get_name(netdev), err, doca_error_get_descr(err));
        return err;
    }

    if (DOCA_FLOW_FLAGS_IS_SET(flags, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT)) {
        err = doca_offload_complete_queue_esw(esw, qid, true);
        if (err != DOCA_SUCCESS) {
            return err;
        }
    }

    dpdk_offload_counter_inc(netdev);

    return err;
}

doca_error_t
doca_offload_remove_entry(struct doca_offload_esw_ctx *esw,
                          unsigned int qid, uint32_t flags,
                          struct doca_flow_pipe_entry **entry)
    OVS_EXCLUDED(mgmt_queue_lock)
{
    doca_error_t err;

    if (!*entry) {
        return DOCA_SUCCESS;
    }

    if (qid == AUX_QUEUE) {
        ovs_doca_mgmt_queue_lock();
        err = doca_flow_pipe_remove_entry(qid, flags, *entry);
        ovs_doca_mgmt_queue_unlock();
    } else {
        err = doca_flow_pipe_remove_entry(qid, flags, *entry);
    }

    if (err == DOCA_SUCCESS) {
        esw->offload_queues[qid].n_waiting_entries++;
        if (qid == AUX_QUEUE) {
            /* Ignore potential errors here, as even if the queue completion
             * failed, the entry removal would still be issued. The caller
             * requires knowing so. */
            doca_offload_complete_queue_esw(esw, qid, true);
        }
        *entry = NULL;
    } else {
        VLOG_ERR("Failed to remove entry %p qid=%d. %d (%s)", *entry, qid, err,
                 doca_error_get_descr(err));
    }

    return err;
}

static void
doca_hash_pipe_ctx_uninit(struct doca_offload_esw_ctx *esw)
{
    struct doca_hash_pipe_ctx *ctx = esw->hash_pipe_ctx;
    unsigned int queue_id = netdev_offload_thread_id();
    enum hash_nw_type nw_type;
    int i;

    if (ctx == NULL) {
        return;
    }

    /* Destroy classifiers. */
    for (nw_type = 0; nw_type < NUM_HASH_NW; nw_type++) {
        for (i = 0; i < NUM_HASH_TP; i++) {
            if (ctx->classifier[nw_type].tcpudp[i]) {
                doca_offload_remove_entry(esw, queue_id, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                          &ctx->classifier[nw_type].tcpudp[i]);
                dpdk_offload_counter_dec(esw->esw_netdev);
            }
        }
        if (ctx->classifier[nw_type].pipe) {
            doca_pipe_group_pipe_destroy(esw, ctx->classifier[nw_type].pipe);
        }
    }

    /* Remove entries first then complete the queue,
     * before starting pipe destroy. */

    for (i = 0; i < NUM_HASH_PIPE_TYPE; i++) {
        if (ctx->hashes[i].entry) {
            doca_offload_remove_entry(esw, queue_id, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                      &ctx->hashes[i].entry);
            dpdk_offload_counter_dec(esw->esw_netdev);
        }
    }

    doca_offload_complete_queue_esw(esw, queue_id, true);

    /* Release hash pipes in the reverse order of their dependency.
     * Ordering is described and enforced at the `hash_pipe_type` definition.
     */
    for (i = NUM_HASH_PIPE_TYPE - 1; i >= 0; i--) {
        if (ctx->hashes[i].pipe) {
            doca_pipe_group_pipe_destroy(esw, ctx->hashes[i].pipe);
        }
    }

    doca_pipe_group_ctx_unref(ctx->post_hash_group_ctx);

    free(ctx);
    esw->hash_pipe_ctx = NULL;
}

static void
doca_basic_pipe_ctx_uninit(struct doca_offload_esw_ctx *esw,
                           struct doca_basic_pipe_ctx *ctx)
{
    doca_pipe_group_pipe_destroy(esw, ctx->pipe);
    ctx->pipe = NULL;
    doca_pipe_group_ctx_unref(ctx->miss_group_ctx);
    ctx->miss_group_ctx = NULL;
    doca_pipe_group_ctx_unref(ctx->fwd_group_ctx);
    ctx->fwd_group_ctx = NULL;
}

static struct reg_field reg_fields[] = {
    [REG_FIELD_FLOW_INFO] = {
        .type = REG_TYPE_META,
        .index = 0,
        .offset = 12,
        .mask = 0x000FFFFF,
    },
    [REG_FIELD_DP_HASH] = {
        .type = REG_TYPE_META,
        .index = 0,
        .offset = 4,
        .mask = 0x000000FF,
    },
    [REG_FIELD_CT_ZONE] = {
        .type = REG_TYPE_TAG,
        .index = 0,
        .offset = 0,
        .mask = 0x0000FFFF,
    },
    [REG_FIELD_TUN_INFO] = {
        .type = REG_TYPE_TAG,
        .index = 0,
        .offset = 16,
        .mask = 0x0000FFFF,
    },
    [REG_FIELD_TAG0] = {
        .type = REG_TYPE_TAG,
        .index = 0,
        .offset = 0,
        .mask = 0xFFFFFFFF,
    },
    /* Scratch info:
     * - prefix id for split flows.
     * - dp-hash ipv6 split hash result.
     * - dp-hash seed.
     */
    [REG_FIELD_SCRATCH] = {
        .type = REG_TYPE_TAG,
        .index = 1,
        .offset = 0,
        .mask = 0xFFFFFFFF,
    },
    [REG_FIELD_CT_MARK] = {
        .type = REG_TYPE_TAG,
        .index = 2,
        .offset = 0,
        .mask = 0xFFFFFFFF,
    },
    [REG_FIELD_RECIRC] = {
        .type = REG_TYPE_TAG,
        .index = 3,
        .offset = 0,
        .mask = MATCH_RECIRC_ID_MASK,
    },
    [REG_FIELD_CT_STATE] = {
        .type = REG_TYPE_TAG,
        .index = 3,
        .offset = 24,
        .mask = 0x000000FF,
    },
    [REG_FIELD_CT_LABEL_ID] = {
        .type = REG_TYPE_TAG,
        .index = 4,
        .offset = 0,
        .mask = 0xFFFFFFFF,
    },
};


/* Register usage:
 * +------+--------+--------+--------+--------+--------+--------+--------+--------+
 * |      |  31-28 |  27-24 |  23-20 |  19-16 |  15-12 |  11-8  |   7-4  |   3-0  |
 * |      +-----------------+-----------------+-----------------+-----------------+
 * |      |      31-24      |      23-16      |      15-8       |       7-0       |
 * |      +-----------------+-----------------+-----------------+-----------------+
 * |      |                                 31-0                                  |
 * +------+-----------------+--------------------------+-----------------+--------+
 * | meta |               FLOW_INFO                    |     dp-hash     |  FREE  |
 * |      | - Pre/post CT: flow-id. In case            |                 |        |
 * |      |   of CT miss, it has a                     |                 |        |
 * |      |   flow_miss_ctx to recover.                |                 |        |
 * |      | - Post-hash: flow-id.                      |                 |        |
 * |      | - Meters: meter-flow-id. As a              |                 |        |
 * |      |   temp id in the post-meter loop.          |                 |        |
 * |      |   Last iteration restores to the           |                 |        |
 * |      |   flow-id.                                 |                 |        |
 * +------+-----------------+-----------------+--------+-----------------+--------+
 * | tag0 |   TUNNEL_INFO / PORT_METER_COLOR  |               CT_ZONE             |
 * |      | - recirc_id(0) matches on the     |                                   |
 * |      |   packet's fields, and decap. The |                                   |
 * |      |   tunnel match is mapped.         |                                   |
 * |      |   Following recircs map on this   |                                   |
 * |      |   ID.                             |                                   |
 * |      | - PORT_METER_COLOR when gpr is    |                                   |
 * |      |   enabled.                        |                                   |
 * |      |                                                                       |
 * |      |        ---- TAG0 should be cleared before leaving FDB----             |
 * +------+-----------------+-----------------+-----------------+-----------------+
 * | tag1 |                                SCRATCH                                |
 * |      | - Prefix id for split flows.                                          |
 * |      | - dp-hash ipv6 split hash result.                                     |
 * |      | - dp-hash seed.                                                       |
 * |      |     ---- TAG1 is mapped to regc_5 which is used also by doca. ----    |
 * |      |     ---- Can be used only temorarily (SCRATCH). ----                  |
 * +------+-----------------+-----------------+-----------------+-----------------+
 * | tag2 |                               CT_MARK                                 |
 * +------+-----------------+-----------------+-----------------+-----------------+
 * | tag3 |     CT_STATE    |                      recirc_id                      |
 * +------+-----------------+-----------------+-----------------+-----------------+
 * | tag4 |                      CT_LABEL. 32 low bits or mapped.                 |
 * +------+-----------------+-----------------+-----------------+-----------------+
 */

static struct reg_field *
dpdk_offload_doca_get_reg_fields(void)
{
    return reg_fields;
}

void
doca_set_reg_val(struct doca_flow_meta *dmeta, uint8_t reg_field_id,
                 uint32_t val)
{
    struct reg_field *reg_field = &reg_fields[reg_field_id];

    if (reg_field->type == REG_TYPE_TAG) {
        dmeta->u32[reg_field->index] |= DOCA_HTOBE32(reg_field_spec(reg_field_id, val));
    } else {
        dmeta->pkt_meta |= DOCA_HTOBE32(reg_field_spec(reg_field_id, val));
    }
}

static void
doca_set_reg_mask(struct doca_flow_meta *dmeta, uint8_t reg_field_id)
{
    doca_set_reg_val(dmeta, reg_field_id, UINT32_MAX);
}

static void
doca_set_reg_val_host_order(struct doca_flow_meta *dmeta, uint8_t reg_field_id, uint32_t val)
{
    struct reg_field *reg_field = &reg_fields[reg_field_id];

    if (reg_field->type == REG_TYPE_TAG) {
        dmeta->u32[reg_field->index] |= reg_field_spec(reg_field_id, val);
    } else {
        dmeta->pkt_meta |= reg_field_spec(reg_field_id, val);
    }
}

static void
doca_set_reg_mask_host_order(struct doca_flow_meta *dmeta, uint8_t reg_field_id)
{
    doca_set_reg_val_host_order(dmeta, reg_field_id, UINT32_MAX);
}

void
doca_offload_set_reg_template(struct doca_flow_meta *dmeta, enum dpdk_reg_id id)
{
    struct reg_field *reg_field = &reg_fields[id];
    uint32_t *u32;

    /* This function does not respect the logical layout of the reg-fields.
     * It sets the whole register used by 'id' to 0xff, as is currently
     * required by DOCA-flow for pipe templates definitions.
     */
    u32 = (reg_field->type == REG_TYPE_TAG) ? &dmeta->u32[reg_field->index]
                                            : &dmeta->pkt_meta;
    memset(u32, 0xff, sizeof *u32);
}

void
doca_offload_set_reg_mask(struct doca_flow_meta *dmeta, enum dpdk_reg_id id)
{
    doca_set_reg_val(dmeta, id, UINT32_MAX);
}

void
doca_set_reg_val_mask(struct doca_flow_meta *dmeta,
                      struct doca_flow_meta *dmeta_mask,
                      uint8_t reg_field_id,
                      uint32_t val)
{
    doca_set_reg_val(dmeta, reg_field_id, val);
    doca_set_reg_mask(dmeta_mask, reg_field_id);
}

static uint32_t
doca_get_reg_bit_offset(uint8_t reg_field_id)
{
    struct reg_field *reg_field = &reg_fields[reg_field_id];
    uint32_t offset;

    offset = reg_field->offset;

    if (reg_field->type == REG_TYPE_TAG) {
        offset += offsetof(struct doca_flow_meta, u32[reg_field->index]) * 8;
    } else {
        offset += offsetof(struct doca_flow_meta, pkt_meta) * 8;
    }

    return offset;
}

static uint32_t
doca_get_reg_width(uint8_t reg_field_id)
{
    struct reg_field *reg_field = &reg_fields[reg_field_id];

    return ffsl(~((uint64_t) reg_field->mask)) - 1;
}

static void
doca_spec_mask(void *dspec__, void *dmask__, size_t size)
{
    char *dspec = dspec__;
    char *dmask = dmask__;
    int i;

    if (!dspec || !dmask) {
        return;
    }

    for (i = 0; i < size; i++) {
        dspec[i] &= dmask[i];
    }
}

static void
doca_copy_mask_field__(void *dspec, void *dmask,
                       const void *spec, const void *mask,
                       size_t size)
{
    memcpy(dspec, spec, size);
    memcpy(dmask, mask, size);
    doca_spec_mask(dspec, dmask, size);
}

static void
doca_translate_gre_key_item(const struct rte_flow_item *item,
                            struct doca_flow_match *doca_spec,
                            struct doca_flow_match *doca_mask)
{
    const rte_be32_t *key_spec, *key_mask;

    key_spec = item->spec;
    key_mask = item->mask;

    if (item->spec) {
        doca_spec->tun.gre_key = (OVS_FORCE doca_be32_t) *key_spec;
    }
    if (item->mask) {
        doca_mask->tun.gre_key = (OVS_FORCE doca_be32_t) *key_mask;
        if (doca_mask->tun.gre_key) {
            doca_mask->tun.type = DOCA_FLOW_TUN_GRE;
            doca_spec->tun.type = DOCA_FLOW_TUN_GRE;
        }
    }

    doca_spec_mask(&doca_spec->tun.gre_key, &doca_mask->tun.gre_key,
                   sizeof doca_spec->tun.gre_key);
}

static void
doca_translate_gre_item(const struct rte_flow_item *item,
                        struct doca_flow_match *doca_spec,
                        struct doca_flow_match *doca_mask)
{
    const struct rte_gre_hdr *greh_spec, *greh_mask;

    greh_spec = (struct rte_gre_hdr *) item->spec;
    greh_mask = (struct rte_gre_hdr *) item->mask;

    if (item->spec) {
        doca_spec->tun.key_present = greh_spec->k;
    }
    if (item->mask) {
        doca_mask->tun.key_present = greh_mask->k;
        if (doca_mask->tun.key_present) {
            doca_spec->tun.type = DOCA_FLOW_TUN_GRE;
            doca_mask->tun.type = DOCA_FLOW_TUN_GRE;
        }
    }

    doca_spec_mask(&doca_spec->tun.key_present, &doca_mask->tun.key_present,
                   sizeof doca_spec->tun.key_present);
}

static void
doca_translate_geneve_item(const struct rte_flow_item *item,
                           struct doca_flow_match *doca_spec,
                           struct doca_flow_match *doca_mask)
{
    const struct rte_flow_item_geneve *gnv_spec = item->spec;
    const struct rte_flow_item_geneve *gnv_mask = item->mask;

    if (!item->spec || !item->mask) {
        return;
    }

    doca_spec->tun.geneve.vni =
        (OVS_FORCE doca_be32_t) get_unaligned_be32(ALIGNED_CAST(ovs_be32 *, gnv_spec->vni));

    doca_mask->tun.geneve.vni =
        (OVS_FORCE doca_be32_t) get_unaligned_be32(ALIGNED_CAST(ovs_be32 *, gnv_mask->vni));
    if (doca_mask->tun.geneve.vni) {
        doca_spec->tun.type = DOCA_FLOW_TUN_GENEVE;
        doca_mask->tun.type = DOCA_FLOW_TUN_GENEVE;
    }

    doca_spec_mask(&doca_spec->tun.geneve.vni, &doca_mask->tun.geneve.vni,
                   sizeof doca_spec->tun.geneve.vni);
}

static int
doca_init_geneve_opt_parser(struct netdev *netdev,
                            const struct rte_flow_item *item)
{
    struct doca_flow_parser_geneve_opt_cfg opt_cfg[MAX_GENEVE_OPT];
    const struct rte_flow_item_geneve_opt *geneve_opt_spec;
    struct doca_offload_esw_ctx *esw_ctx;
    int ret;

    esw_ctx = doca_offload_esw_ctx_get(netdev);
    if (!esw_ctx) {
        VLOG_ERR("%s: Failed to create geneve_opt parser - esw_ctx is NULL",
                 netdev_get_name(netdev));
        return -1;
    }

    if (esw_ctx->gnv_opt_parser.parser) {
        return 0;
    }
    geneve_opt_spec = item->spec;

    memset(&opt_cfg[0], 0, sizeof(opt_cfg[0]));
    opt_cfg[0].match_on_class_mode =
        DOCA_FLOW_PARSER_GENEVE_OPT_MODE_FIXED;
    opt_cfg[0].option_len = geneve_opt_spec->option_len;
    opt_cfg[0].option_class = (OVS_FORCE doca_be16_t) geneve_opt_spec->option_class;
    opt_cfg[0].option_type = geneve_opt_spec->option_type;
    BUILD_ASSERT_DECL(MEMBER_SIZEOF(struct doca_flow_parser_geneve_opt_cfg,
                                    data_mask[0]) ==
                      MEMBER_SIZEOF(struct rte_flow_item_geneve_opt, data[0]));
    memset(&opt_cfg[0].data_mask[0], UINT32_MAX,
           sizeof(opt_cfg[0].data_mask[0]) * geneve_opt_spec->option_len);

    ret = doca_flow_parser_geneve_opt_create(esw_ctx->esw_port, opt_cfg,
                                             MAX_GENEVE_OPT,
                                             &esw_ctx->gnv_opt_parser.parser);
    if (ret) {
        VLOG_DBG_RL(&rl, "%s: Create geneve_opt parser failed - doca call failure "
                    "rc %d, (%s)",netdev_get_name(netdev), ret,
                    doca_error_get_descr(ret));
        return -1;
    }
    return 0;
}

static void
doca_translate_geneve_opt_item(const struct rte_flow_item *item,
                               struct doca_flow_match *doca_spec,
                               struct doca_flow_match *doca_mask)
{
    union doca_flow_geneve_option *doca_opt_spec, *doca_opt_mask;
    const struct rte_flow_item_geneve_opt *geneve_opt_spec;
    const struct rte_flow_item_geneve_opt *geneve_opt_mask;

    geneve_opt_spec = item->spec;
    geneve_opt_mask = item->mask;
    doca_opt_spec = &doca_spec->tun.geneve_options[0];
    doca_opt_mask = &doca_mask->tun.geneve_options[0];

    doca_opt_spec->length = geneve_opt_spec->option_len;
    doca_opt_spec->class_id = (OVS_FORCE doca_be16_t) geneve_opt_spec->option_class;
    doca_opt_spec->type = geneve_opt_spec->option_type;
    doca_opt_mask->length = geneve_opt_mask->option_len;
    doca_opt_mask->class_id = (OVS_FORCE doca_be16_t) geneve_opt_mask->option_class;
    doca_opt_mask->type = geneve_opt_mask->option_type;
    doca_spec_mask(&doca_opt_spec->length, &doca_opt_mask->length,
                   sizeof doca_opt_spec->length);
    doca_spec_mask(&doca_opt_spec->class_id, &doca_opt_mask->class_id,
                   sizeof doca_opt_spec->class_id);
    doca_spec_mask(&doca_opt_spec->type, &doca_opt_mask->type,
                   sizeof doca_opt_spec->type);

    doca_opt_mask->length = doca_opt_spec->length;
    doca_opt_mask->type =  doca_opt_spec->type;

    /* doca_flow represents the geneve option header as an array of a union of
     * 32 bits, the array's first element is the type/class/len and this
     * option's data starts from the next element in the array up to option_len
     */
    doca_opt_spec++;
    doca_opt_mask++;
    BUILD_ASSERT_DECL(sizeof(doca_opt_spec->data) ==
                      sizeof(geneve_opt_spec->data[0]));
    memcpy(&doca_opt_spec->data, &geneve_opt_spec->data[0],
           sizeof(doca_opt_spec->data) * geneve_opt_spec->option_len);
    memcpy(&doca_opt_mask->data, &geneve_opt_mask->data[0],
           sizeof(doca_opt_mask->data) * geneve_opt_spec->option_len);
    doca_spec_mask(&doca_opt_spec->data, &doca_opt_mask->data,
                   sizeof(doca_opt_spec->data) * geneve_opt_spec->option_len);
}

static void
doca_translate_vxlan_item(const struct rte_flow_item *item,
                          struct doca_flow_match *doca_spec,
                          struct doca_flow_match *doca_mask)
{
    const struct vxlan_data *vxlan_spec = item->spec;
    const struct vxlan_data *vxlan_mask = item->mask;
    ovs_be32 spec_vni, mask_vni;

    if (vxlan_spec) {
        spec_vni = get_unaligned_be32(ALIGNED_CAST(ovs_be32 *,
                    vxlan_spec->conf.vni));
        doca_spec->tun.vxlan_tun_id = DOCA_HTOBE32(ntohl((OVS_FORCE doca_be32_t) spec_vni) >> 8);
        if (vxlan_spec->gbp_id) {
            doca_spec->tun.vxlan_type = DOCA_FLOW_TUN_EXT_VXLAN_GBP;
            doca_spec->tun.vxlan_gbp_group_policy_id = htons(vxlan_spec->gbp_id);
        }
    }

    if (vxlan_mask) {
        mask_vni = get_unaligned_be32(ALIGNED_CAST(ovs_be32 *,
                    vxlan_mask->conf.vni));
        doca_mask->tun.vxlan_tun_id = DOCA_HTOBE32(ntohl((OVS_FORCE doca_be32_t) mask_vni) >> 8);
        if (vxlan_spec && vxlan_spec->gbp_id) {
            doca_mask->tun.vxlan_type = DOCA_FLOW_TUN_EXT_VXLAN_GBP;
            doca_mask->tun.vxlan_gbp_group_policy_id = htons(vxlan_spec->gbp_id);
            memset(&doca_mask->tun.vxlan_gbp_group_policy_id, 0xFF,
                   sizeof doca_mask->tun.vxlan_gbp_group_policy_id);
        }
        if (mask_vni) {
            doca_spec->tun.type = DOCA_FLOW_TUN_VXLAN;
            doca_mask->tun.type = DOCA_FLOW_TUN_VXLAN;
        }
    }

    doca_spec_mask(&doca_spec->tun.vxlan_tun_id, &doca_mask->tun.vxlan_tun_id,
                   sizeof doca_spec->tun.vxlan_tun_id);
}

static void
doca_offload_rte_flow_error_set(struct rte_flow_error *error,
                                int code,
                                enum rte_flow_error_type type,
                                const char *message)
{
    static char err_buf[256];

    snprintf(err_buf, sizeof(err_buf), "%s translation %s: %s",
             (type == RTE_FLOW_ERROR_TYPE_ACTION) ? "action" : "match",
             (code == EOPNOTSUPP) ? "unsupported" : "error",
             message);

    rte_flow_error_set(error, code, type, NULL, err_buf);
}

static void
doca_port_id_match(struct doca_flow_match *doca_spec,
                   struct doca_flow_match *doca_mask,
                   struct doca_act_vars *dact_vars)
{
    if (!dact_vars->physdev) {
        return;
    }

    doca_spec->parser_meta.port_id = dact_vars->port_id_match;
    memset(&doca_mask->parser_meta.port_id, 0xFF, sizeof doca_mask->parser_meta.port_id);
}

static int
doca_translate_items(struct netdev *netdev,
                     const struct rte_flow_attr *attr,
                     const struct rte_flow_item *items,
                     struct ovs_doca_flow_match *odoca_spec,
                     struct ovs_doca_flow_match *odoca_mask,
                     struct doca_act_vars *dact_vars,
                     struct rte_flow_error *error)
{
    struct doca_flow_header_format *doca_hdr_spec, *doca_hdr_mask;
    struct doca_flow_match *doca_spec = &odoca_spec->d;
    struct doca_flow_match *doca_mask = &odoca_mask->d;

#define doca_copy_mask_field(dfield, field)                   \
    if (spec && mask) {                                       \
        doca_copy_mask_field__(&doca_hdr_spec->dfield,        \
                               &doca_hdr_mask->dfield,        \
                               &spec->field,                  \
                               &mask->field,                  \
                               sizeof doca_hdr_spec->dfield); \
    }

    /* Start by filling out outer header match and
     * switch to inner in case we encounter a tnl proto.
     */
    doca_hdr_spec = &doca_spec->outer;
    doca_hdr_mask = &doca_mask->outer;

    for (; items->type != RTE_FLOW_ITEM_TYPE_END; items++) {
        int item_type = items->type;

        if (item_type == RTE_FLOW_ITEM_TYPE_PORT_ID) {
            const struct port_id_data *spec = items->spec;

            /* Only recirc_id 0 (group_id == 0) may hold flows
             * from different source ports since it's the root table.
             * For every other recirc_id we have a table per port and
             * therefore we can skip matching on port id for those
             * tables.
             */
            dact_vars->physdev = spec->physdev;
            dact_vars->port_id_match = spec->conf.id;
            if (attr->group > 0) {
                continue;
            }

            doca_port_id_match(doca_spec, doca_mask, dact_vars);
        } else if (item_type == RTE_FLOW_ITEM_TYPE_ETH) {
            const struct rte_flow_item_eth *spec = items->spec;
            const struct rte_flow_item_eth *mask = items->mask;

            if (!spec || !mask) {
                continue;
            }

            doca_copy_mask_field(eth.src_mac, src);
            doca_copy_mask_field(eth.dst_mac, dst);
            doca_copy_mask_field(eth.type, type);

            if (mask->has_vlan) {
                if (doca_hdr_spec == &doca_spec->outer) {
                    doca_spec->parser_meta.outer_l2_type =
                        spec->has_vlan
                        ? DOCA_FLOW_L2_META_SINGLE_VLAN
                        : DOCA_FLOW_L2_META_NO_VLAN;
                    doca_mask->parser_meta.outer_l2_type = UINT32_MAX;
                } else {
                    doca_spec->parser_meta.inner_l2_type =
                        spec->has_vlan
                        ? DOCA_FLOW_L2_META_SINGLE_VLAN
                        : DOCA_FLOW_L2_META_NO_VLAN;
                    doca_mask->parser_meta.inner_l2_type = UINT32_MAX;
                }
            }
            for (int i = 0 ; i < NUM_SEND_TO_KERNEL; i++) {
                if (doca_hdr_spec->eth.type == htons(pre_miss_mapping[i])) {
                    VLOG_DBG_RL(&rl, "Offloading 0x%04x ether-type is not supported",
                                pre_miss_mapping[i]);
                    doca_offload_rte_flow_error_set(error, EOPNOTSUPP, RTE_FLOW_ERROR_TYPE_ITEM,
                                                    "unsupported ether-type");
                    return -1;
                }
            }
        } else if (item_type == RTE_FLOW_ITEM_TYPE_VLAN) {
            const struct rte_flow_item_vlan *spec = items->spec;
            const struct rte_flow_item_vlan *mask = items->mask;

            /* HW supports match on one Ethertype, the Ethertype following the
             * last VLAN tag of the packet (see PRM). DOCA API has only that one.
             * Add a match on it as part of the doca eth header.
             */
            doca_copy_mask_field(eth.type, inner_type);
            doca_copy_mask_field(eth_vlan[0].tci, tci);
            doca_hdr_spec->l2_valid_headers = DOCA_FLOW_L2_VALID_HEADER_VLAN_0;
            doca_hdr_mask->l2_valid_headers = DOCA_FLOW_L2_VALID_HEADER_VLAN_0;
            if (doca_hdr_spec == &doca_spec->outer) {
                doca_spec->parser_meta.outer_l2_type = DOCA_FLOW_L2_META_SINGLE_VLAN;
                doca_mask->parser_meta.outer_l2_type = DOCA_FLOW_L2_META_SINGLE_VLAN;
            } else {
                doca_spec->parser_meta.inner_l2_type = DOCA_FLOW_L2_META_SINGLE_VLAN;
                doca_mask->parser_meta.inner_l2_type = DOCA_FLOW_L2_META_SINGLE_VLAN;
            }
        /* L3 */
        } else if (item_type == RTE_FLOW_ITEM_TYPE_IPV4) {
            const struct rte_flow_item_ipv4 *spec = items->spec;
            const struct rte_flow_item_ipv4 *mask = items->mask;

            doca_copy_mask_field(ip4.next_proto, hdr.next_proto_id);
            doca_copy_mask_field(ip4.src_ip, hdr.src_addr);
            doca_copy_mask_field(ip4.dst_ip, hdr.dst_addr);
            doca_copy_mask_field(ip4.ttl, hdr.time_to_live);
            doca_copy_mask_field(ip4.dscp_ecn, hdr.type_of_service);

            if (!is_all_zeros(&doca_hdr_mask->ip4, sizeof doca_hdr_mask->ip4)) {
                doca_hdr_spec->l3_type = DOCA_FLOW_L3_TYPE_IP4;
                doca_hdr_mask->l3_type = DOCA_FLOW_L3_TYPE_IP4;
            }
            /* DOCA API does not provide distinguishment between first/later
             * frags. Reject both.
             */
            if (items->last || (spec && spec->hdr.fragment_offset)) {
                doca_offload_rte_flow_error_set(error, EOPNOTSUPP, RTE_FLOW_ERROR_TYPE_ITEM,
                                                "unsupported IP fragments offload");
                return -1;
            }

            if (doca_hdr_spec == &doca_spec->outer) {
                doca_mask->parser_meta.outer_ip_fragmented = UINT8_MAX;
                doca_spec->parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV4;
                doca_mask->parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV4;
            } else {
                doca_mask->parser_meta.inner_ip_fragmented = UINT8_MAX;
                doca_mask->parser_meta.outer_ip_fragmented = 0;
                doca_spec->parser_meta.inner_l3_type = DOCA_FLOW_L3_META_IPV4;
                doca_mask->parser_meta.inner_l3_type = DOCA_FLOW_L3_META_IPV4;
            }
        } else if (item_type == RTE_FLOW_ITEM_TYPE_IPV6) {
            const struct rte_flow_item_ipv6 *spec = items->spec;
            const struct rte_flow_item_ipv6 *mask = items->mask;
            uint32_t vtc_spec, vtc_mask;

            doca_copy_mask_field(ip6.src_ip, hdr.src_addr.a);
            doca_copy_mask_field(ip6.dst_ip, hdr.dst_addr.a);
            doca_copy_mask_field(ip6.next_proto, hdr.proto);
            vtc_spec = ntohl(spec->hdr.vtc_flow) >> RTE_IPV6_HDR_TC_SHIFT;
            vtc_mask = ntohl(mask->hdr.vtc_flow) >> RTE_IPV6_HDR_TC_SHIFT;
            doca_hdr_spec->ip6.traffic_class = vtc_spec & vtc_mask;
            doca_hdr_mask->ip6.traffic_class = vtc_mask;
            doca_copy_mask_field(ip6.hop_limit, hdr.hop_limits);

            if (!is_all_zeros(&doca_hdr_mask->ip6, sizeof doca_hdr_mask->ip6)) {
                doca_hdr_spec->l3_type = DOCA_FLOW_L3_TYPE_IP6;
                doca_hdr_mask->l3_type = DOCA_FLOW_L3_TYPE_IP6;
            }
            if (doca_hdr_spec == &doca_spec->outer) {
                doca_mask->parser_meta.outer_ip_fragmented = UINT8_MAX;
                doca_spec->parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV6;
                doca_mask->parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV6;
            } else {
                doca_mask->parser_meta.inner_ip_fragmented = UINT8_MAX;
                doca_mask->parser_meta.outer_ip_fragmented = 0;
                doca_spec->parser_meta.inner_l3_type = DOCA_FLOW_L3_META_IPV6;
                doca_mask->parser_meta.inner_l3_type = DOCA_FLOW_L3_META_IPV6;
            }
        /* L4 */
        } else if (item_type == RTE_FLOW_ITEM_TYPE_UDP) {
            const struct rte_flow_item_udp *spec = items->spec;
            const struct rte_flow_item_udp *mask = items->mask;

            if (doca_hdr_spec == &doca_spec->outer) {
                doca_spec->parser_meta.outer_l4_type = DOCA_FLOW_L4_META_UDP;
                doca_mask->parser_meta.outer_l4_type = DOCA_FLOW_L4_META_UDP;
            } else {
                doca_spec->parser_meta.inner_l4_type = DOCA_FLOW_L4_META_UDP;
                doca_mask->parser_meta.inner_l4_type = DOCA_FLOW_L4_META_UDP;
            }

            doca_copy_mask_field(udp.l4_port.src_port, hdr.src_port);
            doca_copy_mask_field(udp.l4_port.dst_port, hdr.dst_port);
            if (!is_all_zeros(&doca_hdr_mask->udp, sizeof doca_hdr_mask->udp)) {
                doca_hdr_spec->l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_UDP;
                doca_hdr_mask->l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_UDP;
            }
        } else if (item_type ==  RTE_FLOW_ITEM_TYPE_TCP) {
            const struct rte_flow_item_tcp *spec = items->spec;
            const struct rte_flow_item_tcp *mask = items->mask;

            if (doca_hdr_spec == &doca_spec->outer) {
                doca_spec->parser_meta.outer_l4_type = DOCA_FLOW_L4_META_TCP;
                doca_mask->parser_meta.outer_l4_type = DOCA_FLOW_L4_META_TCP;
            } else {
                doca_spec->parser_meta.inner_l4_type = DOCA_FLOW_L4_META_TCP;
                doca_mask->parser_meta.inner_l4_type = DOCA_FLOW_L4_META_TCP;
            }

            doca_copy_mask_field(tcp.l4_port.src_port, hdr.src_port);
            doca_copy_mask_field(tcp.l4_port.dst_port, hdr.dst_port);
            doca_copy_mask_field(tcp.flags, hdr.tcp_flags);
            if (!is_all_zeros(&doca_hdr_mask->tcp, sizeof doca_hdr_mask->tcp)) {
                doca_hdr_spec->l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_TCP;
                doca_hdr_mask->l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_TCP;
            }
        } else if (item_type == RTE_FLOW_ITEM_TYPE_VXLAN) {
            doca_translate_vxlan_item(items, doca_spec, doca_mask);

            doca_hdr_spec = &doca_spec->inner;
            doca_hdr_mask = &doca_mask->inner;
        } else if (item_type == RTE_FLOW_ITEM_TYPE_GRE) {
            doca_translate_gre_item(items, doca_spec, doca_mask);

            doca_hdr_spec = &doca_spec->inner;
            doca_hdr_mask = &doca_mask->inner;
        } else if (item_type == RTE_FLOW_ITEM_TYPE_GRE_KEY) {
            doca_translate_gre_key_item(items, doca_spec, doca_mask);

            doca_hdr_spec = &doca_spec->inner;
            doca_hdr_mask = &doca_mask->inner;
        } else if (item_type == RTE_FLOW_ITEM_TYPE_GENEVE) {
            doca_translate_geneve_item(items, doca_spec, doca_mask);

            doca_hdr_spec = &doca_spec->inner;
            doca_hdr_mask = &doca_mask->inner;
        } else if (item_type == RTE_FLOW_ITEM_TYPE_GENEVE_OPT) {
            if (doca_init_geneve_opt_parser(netdev, items)) {
                doca_offload_rte_flow_error_set(error, EINVAL, RTE_FLOW_ERROR_TYPE_ITEM,
                                                "unable to create geneve_opt parser");
                return -1;
            }
            doca_translate_geneve_opt_item(items, doca_spec, doca_mask);
        } else if (item_type == RTE_FLOW_ITEM_TYPE_ICMP) {
            const struct rte_flow_item_icmp *spec = items->spec;
            const struct rte_flow_item_icmp *mask = items->mask;

            if (doca_hdr_spec == &doca_spec->outer) {
                doca_spec->parser_meta.outer_l4_type = DOCA_FLOW_L4_META_ICMP;
                doca_mask->parser_meta.outer_l4_type = DOCA_FLOW_L4_META_ICMP;
            } else {
                doca_spec->parser_meta.inner_l4_type = DOCA_FLOW_L4_META_ICMP;
                doca_mask->parser_meta.inner_l4_type = DOCA_FLOW_L4_META_ICMP;
            }

            doca_copy_mask_field(icmp.type, hdr.icmp_type);
            doca_copy_mask_field(icmp.code, hdr.icmp_code);
            if (!is_all_zeros(&doca_hdr_mask->icmp, sizeof doca_hdr_mask->icmp)) {
                doca_hdr_spec->l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_ICMP;
                doca_hdr_mask->l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_ICMP;
            }
        } else if (item_type == RTE_FLOW_ITEM_TYPE_ICMP6) {
            const struct rte_flow_item_icmp6 *spec = items->spec;
            const struct rte_flow_item_icmp6 *mask = items->mask;

            if (doca_hdr_spec == &doca_spec->outer) {
                doca_spec->parser_meta.outer_l4_type = DOCA_FLOW_L4_META_ICMP;
                doca_mask->parser_meta.outer_l4_type = DOCA_FLOW_L4_META_ICMP;
            } else {
                doca_spec->parser_meta.inner_l4_type = DOCA_FLOW_L4_META_ICMP;
                doca_mask->parser_meta.inner_l4_type = DOCA_FLOW_L4_META_ICMP;
            }

            doca_copy_mask_field(icmp.code, code);
            doca_copy_mask_field(icmp.type, type);
            if (!is_all_zeros(&doca_hdr_mask->icmp, sizeof doca_hdr_mask->icmp)) {
                doca_hdr_spec->l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_ICMP6;
                doca_hdr_mask->l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_ICMP6;
            }
        } else if (item_type == RTE_FLOW_ITEM_TYPE_TAG) {
            const struct rte_flow_item_tag *spec = items->spec;
            const struct rte_flow_item_tag *mask = items->mask;

            if (spec && mask) {
                doca_spec->meta.u32[spec->index] |= DOCA_HTOBE32(spec->data & mask->data);
                doca_mask->meta.u32[spec->index] |= DOCA_HTOBE32(mask->data);
            }
        } else if (item_type == OVS_RTE_FLOW_ITEM_TYPE(FLOW_INFO)) {
            const struct rte_flow_item_mark *spec = items->spec;

            if (!spec) {
                doca_offload_rte_flow_error_set(error, EOPNOTSUPP, RTE_FLOW_ERROR_TYPE_ITEM,
                                                "No flow-info spec");
                return -1;
            }

            doca_set_reg_val_mask(&doca_spec->meta, &doca_mask->meta, REG_FIELD_FLOW_INFO,
                                  spec->id);
        } else if (item_type == OVS_RTE_FLOW_ITEM_TYPE(HASH)) {
            struct reg_field *reg_field = &reg_fields[REG_FIELD_DP_HASH];
            const struct rte_flow_item_mark *hash_spec = items->spec;
            const struct rte_flow_item_mark *hash_mask = items->mask;

            if (!hash_spec || !hash_mask || hash_mask->id & ~reg_field->mask) {
                /* Can't support larger mask. */
                doca_offload_rte_flow_error_set(error, EOPNOTSUPP, RTE_FLOW_ERROR_TYPE_ITEM,
                                                "unsupported mask");
                return -1;
            }

            doca_set_reg_val(&doca_spec->meta, REG_FIELD_DP_HASH, hash_spec->id & hash_mask->id);
            doca_set_reg_val(&doca_mask->meta, REG_FIELD_DP_HASH, hash_mask->id);
        } else if (item_type == OVS_RTE_FLOW_ITEM_TYPE(NV_MP_PID)) {
            struct nv_mp *nv_mp_spec = (struct nv_mp *) &odoca_spec->nv_mp;
            struct nv_mp *nv_mp_mask = (struct nv_mp *) &odoca_mask->nv_mp;
            const struct rte_flow_item_meta *spec = items->spec;
            const struct rte_flow_item_meta *mask = items->mask;

            if (!spec || !mask) {
                continue;
            }

            nv_mp_spec->pid = spec->data;
            nv_mp_mask->pid = mask->data;
        } else if (item_type == OVS_RTE_FLOW_ITEM_TYPE(NV_MP_PREFERRED)) {
            struct nv_mp *nv_mp_spec = (struct nv_mp *) &odoca_spec->nv_mp;
            struct nv_mp *nv_mp_mask = (struct nv_mp *) &odoca_mask->nv_mp;
            const struct rte_flow_item_meta *spec = items->spec;
            const struct rte_flow_item_meta *mask = items->mask;

            if (!spec || !mask) {
                continue;
            }

            nv_mp_spec->preferred = spec->data;
            nv_mp_mask->preferred = mask->data;
        } else if (item_type == OVS_RTE_FLOW_ITEM_TYPE(NV_MP_STRICT)) {
            struct nv_mp *nv_mp_spec = (struct nv_mp *) &odoca_spec->nv_mp;
            struct nv_mp *nv_mp_mask = (struct nv_mp *) &odoca_mask->nv_mp;
            const struct rte_flow_item_meta *spec = items->spec;
            const struct rte_flow_item_meta *mask = items->mask;

            if (!spec || !mask) {
                continue;
            }

            nv_mp_spec->strict = spec->data;
            nv_mp_mask->strict = mask->data;
        } else if (item_type == OVS_RTE_FLOW_ITEM_TYPE(L3_TYPE_HINT)) {
            const struct rte_flow_item_meta *spec = items->spec;

            ovs_assert(spec);

            dact_vars->l3_type_hint = spec->data;
        } else if (item_type == OVS_RTE_FLOW_ITEM_TYPE(L4_TYPE_HINT)) {
            const struct rte_flow_item_meta *spec = items->spec;

            ovs_assert(spec);

            dact_vars->l4_type_hint = spec->data;
        } else {
            doca_offload_rte_flow_error_set(error, EOPNOTSUPP, RTE_FLOW_ERROR_TYPE_ITEM,
                                            "unsupported match during doca translation");
            return -1;
        }
    }

    return 0;
}

OVS_ASSERT_PACKED(struct doca_shared_encap_key,
    struct doca_flow_resource_encap_cfg encap_cfg;
    struct doca_flow_port *port;
);

struct doca_shared_encap_ctx {
    uint32_t id;
    struct doca_flow_port *port;
};

struct doca_shared_encap_arg {
    struct doca_flow_resource_encap_cfg encap_cfg;
    struct doca_flow_port *port;
    struct atomic_count *count;
};

static int
doca_shared_encap_ctx_init(void *ctx_, void *arg_)
{
    struct doca_flow_shared_resource_cfg shared_cfg;
    struct doca_shared_encap_ctx *ctx = ctx_;
    struct doca_shared_encap_arg *arg = arg_;
    unsigned int max_shared_encaps;
    uint32_t id;
    int ret;

    max_shared_encaps = ovs_doca_get_max_shared_encaps();
    if (!max_shared_encaps) {
        VLOG_DBG("Shared encaps are disabled");
        return -1;
    }
    if (atomic_count_get(arg->count) == max_shared_encaps) {
        VLOG_DBG("Shared encaps pool is exhausted");
        return -1;
    }

    memset(&shared_cfg, 0, sizeof shared_cfg);
    memcpy(&shared_cfg.encap_cfg, &arg->encap_cfg, sizeof shared_cfg.encap_cfg);

    ret = doca_flow_port_shared_resource_get(arg->port, DOCA_FLOW_SHARED_RESOURCE_ENCAP, &id);
    if (ret) {
        VLOG_WARN("Failed to get encap ID: Error %d (%s)", ret, doca_error_get_descr(ret));
        return -1;
    }
    ctx->id = id;
    ret = doca_flow_port_shared_resource_set_cfg(arg->port, DOCA_FLOW_SHARED_RESOURCE_ENCAP, id,
                                                 &shared_cfg);
    if (ret) {
        VLOG_WARN("Failed to configure encap context: Error %d (%s)", ret,
                  doca_error_get_descr(ret));
        goto err;
    }
    ctx->port = arg->port;
    atomic_count_inc(arg->count);
    return 0;

err:
    doca_flow_port_shared_resource_put(arg->port, DOCA_FLOW_SHARED_RESOURCE_ENCAP, id);
    return -1;
}

static void
doca_shared_encap_ctx_uninit(void *ctx_)
{
    struct doca_shared_encap_ctx *ctx = ctx_;

    if (!ctx || !ctx->port) {
        return;
    }

    doca_flow_port_shared_resource_put(ctx->port, DOCA_FLOW_SHARED_RESOURCE_ENCAP, ctx->id);
}

static struct ds *
dump_shared_encap_id(struct ds *s, void *key_, void *ctx_, void *arg OVS_UNUSED)
{
    struct doca_shared_encap_key *key = key_;
    struct doca_shared_encap_ctx *ctx = ctx_;

    ds_put_format(s, "shared_encap_ctx=%p, port=%p", ctx, key->port);
    return s;
}

static void
doca_shared_encaps_ctx_uninit_cb(void *ctx_, void *key_ OVS_UNUSED, void *arg_)
{
    struct doca_shared_encap_ctx *ctx = ctx_;
    struct refmap *rfm = arg_;

    refmap_unref(rfm, ctx);
}

static void
doca_shared_encaps_ctx_uninit(struct doca_shared_encaps_ctx *ctx)
{
    refmap_for_each(ctx->rfm, doca_shared_encaps_ctx_uninit_cb, ctx->rfm);
    refmap_destroy(ctx->rfm);
}

static void
doca_shared_encaps_ctx_init(struct netdev *netdev, struct doca_shared_encaps_ctx *ctx)
{
    char map_name[50];

    snprintf(map_name, sizeof map_name, "shared-encap-id-%s", netdev_get_name(netdev));
    ctx->rfm = refmap_create(map_name,
                             sizeof(struct doca_shared_encap_key),
                             sizeof(struct doca_shared_encap_ctx),
                             doca_shared_encap_ctx_init,
                             doca_shared_encap_ctx_uninit,
                             dump_shared_encap_id, false);
    atomic_count_init(&ctx->count, 0);
}

static struct doca_shared_encap_ctx *
doca_shared_encap_ctx_ref(struct doca_shared_encap_key *key,
                          struct doca_shared_encap_arg *args,
                          uint32_t *shared_encap_id,
                          struct doca_shared_encaps_ctx *shared_encaps_ctx)
{
    struct refmap *doca_shared_encap_rfm = shared_encaps_ctx->rfm;
    struct doca_shared_encap_ctx *ctx;

    ctx = refmap_ref(doca_shared_encap_rfm, key, args);
    if (ctx) {
        *shared_encap_id = ctx->id;
    }
    return ctx;
}

static int
doca_translate_geneve_encap(const struct genevehdr *geneve,
                            struct doca_flow_encap_action *encap,
                            struct doca_flow_encap_action *encap_mask)
{
    encap->tun.type = DOCA_FLOW_TUN_GENEVE;
    encap_mask->tun.type = encap->tun.type;
    encap->tun.geneve.ver_opt_len = geneve->opt_len;
    encap->tun.geneve.ver_opt_len |= geneve->ver << 6;
    encap_mask->tun.geneve.ver_opt_len = encap->tun.geneve.ver_opt_len;
    encap->tun.geneve.o_c = geneve->critical << 6;
    encap->tun.geneve.o_c |= geneve->oam << 7;
    encap_mask->tun.geneve.o_c = encap->tun.geneve.o_c;
    encap->tun.geneve.next_proto = (OVS_FORCE doca_be16_t) geneve->proto_type;
    encap_mask->tun.geneve.next_proto = encap->tun.geneve.next_proto;
    encap->tun.geneve.vni = (OVS_FORCE doca_be32_t) get_16aligned_be32(&geneve->vni);
    encap_mask->tun.geneve.vni = UINT32_MAX;

    if (geneve->options[0].length) {
        encap->tun.geneve_options[0].class_id =
            (OVS_FORCE doca_be16_t) geneve->options[0].opt_class;
        encap_mask->tun.geneve_options[0].class_id = 0xFFFF;
        encap->tun.geneve_options[0].type = geneve->options[0].type;
        encap_mask->tun.geneve_options[0].type = 0xFF;
        encap->tun.geneve_options[0].length = geneve->options[0].length;
        encap_mask->tun.geneve_options[0].length = 0xFF;

        /* doca_flow represents the geneve option header as an array of a union
         * of 32 bits, the array's first element is the type/class/len and this
         * option's data starts from the next element in the array up to option_len
         */
        BUILD_ASSERT_DECL(sizeof(encap->tun.geneve_options[1].data) ==
                          sizeof(geneve->options[1]));
        memcpy(&encap->tun.geneve_options[1].data, &geneve->options[1],
               sizeof(encap->tun.geneve_options[1].data) * geneve->options[0].length);
        memset(&encap_mask->tun.geneve_options[1].data, 0xFF,
               sizeof(encap->tun.geneve_options[1].data) * geneve->options[0].length);
    }

    return 0;
}

static int
doca_translate_gre_encap(const struct gre_base_hdr *gre,
                         struct doca_flow_encap_action *encap,
                         struct doca_flow_encap_action *encap_mask)
{
    encap->tun.type = DOCA_FLOW_TUN_GRE;
    encap_mask->tun.type = encap->tun.type;
    encap->tun.protocol = (OVS_FORCE doca_be16_t) gre->protocol;
    encap_mask->tun.protocol = 0xFFFF;
    encap->tun.key_present = !!(gre->flags & htons(GRE_KEY));
    encap_mask->tun.key_present = encap->tun.key_present;

    if (encap->tun.key_present) {
        const void *gre_key = gre + 1;

        memcpy(&encap->tun.gre_key, gre_key, sizeof encap->tun.gre_key);
        encap_mask->tun.gre_key = UINT32_MAX;
    }

    return 0;
}

static int
doca_translate_vxlan_encap(const struct vxlanhdr *vxlan,
                           struct doca_flow_encap_action *encap,
                           struct doca_flow_encap_action *encap_mask)
{
    ovs_be32 vx_flags = get_16aligned_be32(&vxlan->vx_flags);
    ovs_be32 vx_vni = get_16aligned_be32(&vxlan->vx_vni);

    encap->tun.type = DOCA_FLOW_TUN_VXLAN;
    encap_mask->tun.type = encap->tun.type;
    if (vx_flags & htonl(VXLAN_HF_GBP)) {
        /* DOCA doesn't exposes flags in the API, so currently it's not
         * possible to offload non-default flags setting.
         */
        if (vxlan->vx_gbp.flags) {
            return -1;
        }
        encap->tun.vxlan_type = DOCA_FLOW_TUN_EXT_VXLAN_GBP;
        encap_mask->tun.vxlan_type = encap->tun.vxlan_type;
        encap->tun.vxlan_gbp_group_policy_id = vxlan->vx_gbp.policy_id;
        memset(&encap_mask->tun.vxlan_gbp_group_policy_id, 0xFF,
               sizeof encap_mask->tun.vxlan_gbp_group_policy_id);
    }

    encap->tun.vxlan_tun_id = DOCA_HTOBE32(ntohl(vx_vni) >> 8);
    /* Explicitly set the vxlan_tun_id mask as UINT32_MAX to indicate for doca-flow
     * that this field is changeable and should not be ignored.
     */
    encap_mask->tun.vxlan_tun_id = UINT32_MAX;

    return 0;
}

static int
doca_get_shared_encap_id(struct doca_flow_resource_encap_cfg *encap_cfg,
                         struct doca_flow_port *port,
                         uint32_t *shared_encap_id,
                         struct doca_shared_encaps_ctx *shared_encaps_ctx)
{
    struct doca_shared_encap_key shared_encap_key;
    struct doca_shared_encap_ctx *encap_ctx;
    struct doca_shared_encap_arg args;

    memset(&shared_encap_key, 0, sizeof shared_encap_key);
    memcpy(&shared_encap_key.encap_cfg, encap_cfg, sizeof shared_encap_key.encap_cfg);
    memcpy(&args.encap_cfg, encap_cfg, sizeof args.encap_cfg);
    shared_encap_key.port = port;
    args.port = shared_encap_key.port;
    args.count = &shared_encaps_ctx->count;
    encap_ctx = doca_shared_encap_ctx_ref(&shared_encap_key, &args, shared_encap_id,
                                          shared_encaps_ctx);

    if (!encap_ctx) {
        return -1;
    }

    return 0;
}

static int
doca_translate_raw_encap(const struct raw_encap_data *data,
                         struct doca_offload_esw_ctx *esw_ctx,
                         struct ovs_doca_flow_actions *odacts,
                         struct ovs_doca_flow_actions *odacts_masks,
                         struct doca_act_vars *dact_vars,
                         struct rte_flow_error *error)
{
    struct doca_flow_resource_encap_cfg encap_cfg_mask = { .is_l2 = true, };
    struct doca_flow_resource_encap_cfg encap_cfg = { .is_l2 = true, };
    struct doca_flow_actions *dacts, *dacts_masks;
    struct doca_flow_header_format *outer_mask;
    struct doca_flow_encap_action *encap_mask;
    struct doca_flow_header_format *outer;
    struct doca_flow_encap_action *encap;
    struct ovs_16aligned_ip6_hdr *ip6;
    struct vlan_header *vlan;
    uint32_t shared_encap_id;
    struct udp_header *udp;
    struct eth_header *eth;
    struct ip_header *ip;
    ovs_be16 proto;
    void *l4;

    dacts = &odacts->d;
    dacts_masks = &odacts_masks->d;
    encap = &encap_cfg.encap;
    encap_mask = &encap_cfg_mask.encap;
    outer_mask = &encap_mask->outer;
    outer = &encap->outer;

    /* L2 */
    eth = find_raw_encap_spec(data, RTE_FLOW_ITEM_TYPE_ETH);
    if (!eth) {
        doca_offload_rte_flow_error_set(error, EINVAL, RTE_FLOW_ERROR_TYPE_ACTION,
                                        "unable to translate encap - cannot find L2");
        return -1;
    }

    memcpy(&outer->eth.src_mac, &eth->eth_src, sizeof outer->eth.src_mac);
    memset(&outer_mask->eth.src_mac, 0xFF, sizeof outer_mask->eth.src_mac);
    memcpy(&outer->eth.dst_mac, &eth->eth_dst, sizeof outer->eth.dst_mac);
    memset(&outer_mask->eth.dst_mac, 0xFF, sizeof outer_mask->eth.dst_mac);

    outer->eth.type = (OVS_FORCE doca_be16_t) eth->eth_type;
    outer_mask->eth.type = outer->eth.type;

    proto = eth->eth_type;
    if (proto == htons(ETH_TYPE_VLAN_8021Q)) {
        vlan = ALIGNED_CAST(struct vlan_header *, (uint8_t *) (eth + 1));

        outer->eth_vlan[0].tci = (OVS_FORCE doca_be16_t) vlan->vlan_tci;
        outer_mask->eth_vlan[0].tci = outer->eth_vlan[0].tci;
        outer->l2_valid_headers = DOCA_FLOW_L2_VALID_HEADER_VLAN_0;
        outer_mask->l2_valid_headers = outer->l2_valid_headers;

        proto = vlan->vlan_next_type;
    }

    /* L3 */
    if (proto == htons(ETH_TYPE_IP)) {
        ip = find_raw_encap_spec(data, RTE_FLOW_ITEM_TYPE_IPV4);
        if (!ip) {
            doca_offload_rte_flow_error_set(error, EINVAL, RTE_FLOW_ERROR_TYPE_ACTION,
                                            "unable to translate encap - cannot find IPv4");
            return -1;
        }

        outer->l3_type = DOCA_FLOW_L3_TYPE_IP4;
        outer_mask->l3_type = outer->l3_type;
        outer->ip4.src_ip = (OVS_FORCE doca_be32_t) get_16aligned_be32(&ip->ip_src);
        outer_mask->ip4.src_ip = 0xFFFFFFFF;
        outer->ip4.dst_ip = (OVS_FORCE doca_be32_t) get_16aligned_be32(&ip->ip_dst);
        outer_mask->ip4.dst_ip = 0xFFFFFFFF;
        outer->ip4.ttl = ip->ip_ttl;
        outer_mask->ip4.ttl = 0xFF;
        outer->ip4.dscp_ecn = ip->ip_tos;
        outer_mask->ip4.dscp_ecn = 0xFF;
        l4 = ip + 1;
    } else if (proto == htons(ETH_TYPE_IPV6)) {
        ip6 = find_raw_encap_spec(data, RTE_FLOW_ITEM_TYPE_IPV6);
        if (!ip6) {
            doca_offload_rte_flow_error_set(error, EINVAL, RTE_FLOW_ERROR_TYPE_ACTION,
                                            "unable to translate encap - cannot find IPv6");
            return -1;
        }

        outer->l3_type = DOCA_FLOW_L3_TYPE_IP6;
        outer_mask->l3_type = outer->l3_type;
        memcpy(&outer->ip6.src_ip, &ip6->ip6_src, sizeof ip6->ip6_src);
        memset(&outer_mask->ip6.src_ip, 0xFF, sizeof outer_mask->ip6.src_ip);
        memcpy(&outer->ip6.dst_ip, &ip6->ip6_dst, sizeof ip6->ip6_dst);
        memset(&outer_mask->ip6.dst_ip, 0xFF, sizeof outer_mask->ip6.dst_ip);
        outer->ip6.hop_limit = ip6->ip6_hlim;
        outer_mask->ip6.hop_limit = 0xFF;
        outer->ip6.traffic_class = ntohl(get_16aligned_be32(&ip6->ip6_flow)) >> 20;
        outer_mask->ip6.traffic_class = 0xFF;
        l4 = ip6 + 1;
    } else {
        doca_offload_rte_flow_error_set(error, EINVAL, RTE_FLOW_ERROR_TYPE_ACTION,
                                        "unable to translate encap - cannot find IPv4/IPv6");
        return -1;
    }

    /* Tunnel */
    if (data->tnl_type == OVS_VPORT_TYPE_GRE
        || data->tnl_type == OVS_VPORT_TYPE_IP6GRE) {
        if (doca_translate_gre_encap(l4, encap, encap_mask)) {
            doca_offload_rte_flow_error_set(error, EINVAL, RTE_FLOW_ERROR_TYPE_ACTION,
                                            "unable to translate encap - gre");
            return -1;
        }
    } else {
        udp = l4;
        /* L4 */
        outer->l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_UDP;
        outer_mask->l4_type_ext = outer->l4_type_ext;
        outer->udp.l4_port.src_port = (OVS_FORCE doca_be16_t) udp->udp_src;
        outer_mask->udp.l4_port.src_port = 0xFFFF;
        outer->udp.l4_port.dst_port = (OVS_FORCE doca_be16_t) udp->udp_dst;
        outer_mask->udp.l4_port.dst_port = outer->udp.l4_port.dst_port;

        if (data->tnl_type == OVS_VPORT_TYPE_GENEVE) {
            if (doca_translate_geneve_encap((void *) (udp + 1), encap, encap_mask)) {
                doca_offload_rte_flow_error_set(error, EINVAL, RTE_FLOW_ERROR_TYPE_ACTION,
                                                "unable to translate encap - geneve");
                return -1;
            }
        }

        if (data->tnl_type == OVS_VPORT_TYPE_VXLAN) {
            if (doca_translate_vxlan_encap((void *) (udp + 1), encap, encap_mask)) {
                doca_offload_rte_flow_error_set(error, EINVAL, RTE_FLOW_ERROR_TYPE_ACTION,
                                                "unable to translate encap - vxlan");
                return -1;
            }
        }
    }

    odacts->encap_type = data->tnl_type;
    odacts->encap_size = data->conf.size;

    if (!doca_get_shared_encap_id(&encap_cfg, esw_ctx->esw_port,
                                  &shared_encap_id,
                                  &esw_ctx->shared_encaps_ctx)) {
        if (dact_vars) {
            dact_vars->shared_encap_tun_type = encap_cfg.encap.tun.type;
            dact_vars->shared_encap_l3_type = encap_cfg.encap.outer.l3_type;
        }
        dacts->shared_encap_id = shared_encap_id;
        memset(&dacts_masks->shared_encap_id, 0xFF, sizeof dacts_masks->shared_encap_id);
        dacts->encap_type = DOCA_FLOW_RESOURCE_TYPE_SHARED;
        dacts_masks->encap_type = dacts->encap_type;
    } else {
        memcpy(&dacts->encap_cfg, &encap_cfg, sizeof dacts->encap_cfg);
        memcpy(&dacts_masks->encap_cfg, &encap_cfg_mask, sizeof dacts_masks->encap_cfg);
        dacts->encap_type = DOCA_FLOW_RESOURCE_TYPE_NON_SHARED;
        dacts_masks->encap_type = dacts->encap_type;
    }

    return 0;
}

OVS_ASSERT_PACKED(struct doca_shared_meter_key,
    uint32_t of_meter_id;
    uint8_t pad[4];
);

struct doca_shared_meter_ctx {
    uint32_t id;
    uint32_t red_id;
    uint32_t green_id;
    struct doca_flow_port *port;
    struct doca_flow_resource_meter_cfg cfg;
};

struct doca_shared_meter_arg {
    struct doca_flow_resource_meter_cfg meter_cfg;
    struct doca_flow_port *port;
};

static int
doca_shared_meter_ctx_init(void *ctx_, void *arg_)
{
    struct doca_flow_shared_resource_cfg shared_cfg;
    struct doca_shared_meter_ctx *ctx = ctx_;
    struct doca_shared_meter_arg *arg = arg_;
    uint32_t red_id, green_id, meter_id;
    int ret;

    memset(&shared_cfg, 0, sizeof shared_cfg);
    ret = doca_flow_port_shared_resource_get(arg->port, DOCA_FLOW_SHARED_RESOURCE_COUNTER, &red_id);
    if (ret) {
        VLOG_WARN("Failed to get DOCA red meter counter ID: %d (%s)", ret,
                  doca_error_get_descr(ret));
        return -1;
    }
    ret = doca_flow_port_shared_resource_set_cfg(arg->port, DOCA_FLOW_SHARED_RESOURCE_COUNTER,
                                                 red_id, &shared_cfg);
    if (ret) {
        VLOG_WARN("Failed to configure DOCA red meter counter ID%u: %d (%s)", red_id, ret,
                  doca_error_get_descr(ret));
        goto err_red;
    }
    ctx->red_id = red_id;

    ret = doca_flow_port_shared_resource_get(arg->port, DOCA_FLOW_SHARED_RESOURCE_COUNTER,
                                             &green_id);
    if (ret) {
        VLOG_WARN("Failed to get DOCA green meter counter ID: %d (%s)", ret,
                  doca_error_get_descr(ret));
        goto err_red;
    }
    ret = doca_flow_port_shared_resource_set_cfg(arg->port, DOCA_FLOW_SHARED_RESOURCE_COUNTER,
                                                 green_id, &shared_cfg);
    if (ret) {
        VLOG_WARN("Failed to configure DOCA green meter counter ID%u: %d (%s)", green_id, ret,
                  doca_error_get_descr(ret));
        goto err_green;
    }
    ctx->green_id = green_id;

    memcpy(&shared_cfg.meter_cfg, &arg->meter_cfg, sizeof shared_cfg.meter_cfg);
    ret = doca_flow_port_shared_resource_get(arg->port, DOCA_FLOW_SHARED_RESOURCE_METER, &meter_id);
    if (ret) {
        VLOG_WARN("Failed to get DOCA meter ID: %d (%s)", ret, doca_error_get_descr(ret));
        goto err_green;
    }
    ret = doca_flow_port_shared_resource_set_cfg(arg->port, DOCA_FLOW_SHARED_RESOURCE_METER,
                                                 meter_id, &shared_cfg);
    if (ret) {
        VLOG_WARN("Failed to configure DOCA meter ID%u: %d (%s)", meter_id, ret,
                  doca_error_get_descr(ret));
        goto err_meter;
    }
    ctx->id = meter_id;
    ctx->port = arg->port;
    memcpy(&ctx->cfg, &arg->meter_cfg, sizeof ctx->cfg);
    return 0;

err_meter:
    doca_flow_port_shared_resource_put(ctx->port, DOCA_FLOW_SHARED_RESOURCE_COUNTER, meter_id);
err_green:
    doca_flow_port_shared_resource_put(ctx->port, DOCA_FLOW_SHARED_RESOURCE_COUNTER, green_id);
err_red:
    doca_flow_port_shared_resource_put(ctx->port, DOCA_FLOW_SHARED_RESOURCE_COUNTER, red_id);
    return -1;
}

static void
doca_shared_meter_ctx_uninit(void *ctx_)
{
    struct doca_shared_meter_ctx *ctx = ctx_;

    if (!ctx || !ctx->port) {
        return;
    }

    doca_flow_port_shared_resource_put(ctx->port, DOCA_FLOW_SHARED_RESOURCE_METER, ctx->id);
    doca_flow_port_shared_resource_put(ctx->port, DOCA_FLOW_SHARED_RESOURCE_COUNTER, ctx->red_id);
    doca_flow_port_shared_resource_put(ctx->port, DOCA_FLOW_SHARED_RESOURCE_COUNTER, ctx->green_id);
}

static struct ds *
dump_shared_meter_id(struct ds *s, void *key_, void *ctx_, void *arg OVS_UNUSED)
{
    struct doca_shared_meter_key *key = key_;
    struct doca_shared_meter_ctx *ctx = ctx_;

    ds_put_format(s, "shared_meter_ctx=%p, meter_id=%u, doca_meter_id=%u, red_cntr_id=%u,"
                  " green_cntr_id=%u", ctx, key->of_meter_id, ctx->id, ctx->red_id, ctx->green_id);
    return s;
}

static void
doca_shared_meters_ctx_uninit_cb(void *ctx_, void *key_ OVS_UNUSED, void *arg_)
{
    struct doca_shared_meter_ctx *ctx = ctx_;
    struct refmap *rfm = arg_;

    refmap_unref(rfm, ctx);
}

static void
doca_shared_meters_ctx_uninit(struct doca_shared_meters_ctx *ctx)
{
    refmap_for_each(ctx->rfm, doca_shared_meters_ctx_uninit_cb, ctx->rfm);
    refmap_destroy(ctx->rfm);
}

static void
doca_shared_meters_ctx_init(struct netdev *netdev, struct doca_shared_meters_ctx *ctx)
{
    char map_name[50];

    snprintf(map_name, sizeof map_name, "shared-meter-id-%s", netdev_get_name(netdev));
    ctx->rfm = refmap_create(map_name,
                             sizeof(struct doca_shared_meter_key),
                             sizeof(struct doca_shared_meter_ctx),
                             doca_shared_meter_ctx_init,
                             doca_shared_meter_ctx_uninit,
                             dump_shared_meter_id, false);
}

static struct doca_shared_meter_ctx *
doca_shared_meter_ctx_ref(struct doca_shared_meter_key *key,
                          struct doca_shared_meter_arg *args,
                          struct doca_shared_meters_ctx *shared_meters_ctx)
{
    struct refmap *doca_shared_meter_rfm = shared_meters_ctx->rfm;
    struct doca_shared_meter_ctx *ctx;

    /* If meter is already present in the refmap and the requested configuration differs from the
     * last saved one it is a modify meter operation and it must not increase reference count for
     * the meter.
     */
    ctx = refmap_find(doca_shared_meter_rfm, key);
    if (ctx && memcmp(&ctx->cfg, &args->meter_cfg, sizeof ctx->cfg)) {
        struct doca_flow_shared_resource_cfg shared_cfg;
        int ret;

        memcpy(&shared_cfg.meter_cfg, &args->meter_cfg, sizeof shared_cfg.meter_cfg);
        ret = doca_flow_port_shared_resource_set_cfg(args->port, DOCA_FLOW_SHARED_RESOURCE_METER,
                                                     ctx->id, &shared_cfg);
        if (!ret) {
            memcpy(&ctx->cfg, &args->meter_cfg, sizeof ctx->cfg);
        } else {
            VLOG_ERR("Failed to update meter ID%u configuration: %d (%s)", ctx->id, ret,
                     doca_error_get_descr(ret));
        }
    } else {
        ctx = refmap_ref(doca_shared_meter_rfm, key, args);
    }

    return ctx;
}

static struct doca_shared_meter_ctx *
doca_get_shared_meter(uint32_t of_meter_id,
                      struct doca_flow_resource_meter_cfg *meter_cfg,
                      struct doca_flow_port *port,
                      struct doca_shared_meters_ctx *shared_meters_ctx)
{
    struct doca_shared_meter_key shared_meter_key = {
        .of_meter_id = of_meter_id
    };
    struct doca_shared_meter_ctx *meter_ctx;
    struct doca_shared_meter_arg args = {
        .port = port
    };

    memcpy(&args.meter_cfg, meter_cfg, sizeof args.meter_cfg);
    meter_ctx = doca_shared_meter_ctx_ref(&shared_meter_key, &args, shared_meters_ctx);

    return meter_ctx;
}

static void
doca_put_shared_meter(uint32_t of_meter_id, struct doca_shared_meters_ctx *shared_meters_ctx)
{
    struct doca_shared_meter_ctx *meter_ctx;
    struct doca_shared_meter_key key = {
        .of_meter_id = of_meter_id
    };

    meter_ctx = refmap_find(shared_meters_ctx->rfm, &key);
    if (meter_ctx) {
        refmap_unref(shared_meters_ctx->rfm, meter_ctx);
    }
}

static int
doca_hash_pipe_init(struct netdev *netdev,
                    unsigned int queue_id,
                    struct doca_hash_pipe_ctx *hash_pipe_ctx,
                    enum hash_pipe_type type)
{
    struct doca_flow_pipe *next_pipe = doca_pipe_group_get_pipe(hash_pipe_ctx->post_hash_group_ctx);
    struct ovs_doca_flow_match hash_matches[NUM_HASH_PIPE_TYPE] = {
        [HASH_TYPE_IPV4_UDP] = { .d = {
            .outer.l3_type = DOCA_FLOW_L3_TYPE_IP4,
            .outer.ip4.src_ip = BE32_MAX,
            .outer.ip4.dst_ip = BE32_MAX,
            .outer.l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_UDP,
            .outer.udp.l4_port.src_port = BE16_MAX,
            .outer.udp.l4_port.dst_port = BE16_MAX,
            .parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV4,
            .parser_meta.outer_l4_type = DOCA_FLOW_L4_META_UDP,
        }, },
        [HASH_TYPE_IPV4_TCP] = { .d = {
            .outer.l3_type = DOCA_FLOW_L3_TYPE_IP4,
            .outer.ip4.src_ip = BE32_MAX,
            .outer.ip4.dst_ip = BE32_MAX,
            .outer.l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_TCP,
            .outer.tcp.l4_port.src_port = BE16_MAX,
            .outer.tcp.l4_port.dst_port = BE16_MAX,
            .parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV4,
            .parser_meta.outer_l4_type = DOCA_FLOW_L4_META_TCP,
        }, },
        [HASH_TYPE_IPV4_L3] = { .d = {
            .outer.l3_type = DOCA_FLOW_L3_TYPE_IP4,
            .outer.ip4.src_ip = BE32_MAX,
            .outer.ip4.dst_ip = BE32_MAX,
            .parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV4,
        }, },
        [HASH_TYPE_IPV6_UDP_SUF] = { .d = {
            .outer.l3_type = DOCA_FLOW_L3_TYPE_IP6,
            .outer.ip6.dst_ip[0] = BE32_MAX,
            .outer.ip6.dst_ip[1] = BE32_MAX,
            .outer.ip6.dst_ip[2] = BE32_MAX,
            .outer.ip6.dst_ip[3] = BE32_MAX,
            .outer.l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_UDP,
            .outer.udp.l4_port.dst_port = BE16_MAX,
            .parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV6,
            .parser_meta.outer_l4_type = DOCA_FLOW_L4_META_UDP,
        }, },
        [HASH_TYPE_IPV6_TCP_SUF] = { .d = {
            .outer.l3_type = DOCA_FLOW_L3_TYPE_IP6,
            .outer.ip6.dst_ip[0] = BE32_MAX,
            .outer.ip6.dst_ip[1] = BE32_MAX,
            .outer.ip6.dst_ip[2] = BE32_MAX,
            .outer.ip6.dst_ip[3] = BE32_MAX,
            .outer.l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_TCP,
            .outer.tcp.l4_port.dst_port = BE16_MAX,
            .parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV6,
            .parser_meta.outer_l4_type = DOCA_FLOW_L4_META_TCP,
        }, },
        [HASH_TYPE_IPV6_L3_SUF] = { .d = {
            .outer.l3_type = DOCA_FLOW_L3_TYPE_IP6,
            .outer.ip6.dst_ip[0] = BE32_MAX,
            .outer.ip6.dst_ip[1] = BE32_MAX,
            .outer.ip6.dst_ip[2] = BE32_MAX,
            .outer.ip6.dst_ip[3] = BE32_MAX,
            .parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV6,
        }, },
        [HASH_TYPE_IPV6_UDP_PRE] = { .d = {
            .outer.l3_type = DOCA_FLOW_L3_TYPE_IP6,
            .outer.ip6.src_ip[0] = BE32_MAX,
            .outer.ip6.src_ip[1] = BE32_MAX,
            .outer.ip6.src_ip[2] = BE32_MAX,
            .outer.ip6.src_ip[3] = BE32_MAX,
            .outer.l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_UDP,
            .outer.udp.l4_port.src_port = BE16_MAX,
            .parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV6,
            .parser_meta.outer_l4_type = DOCA_FLOW_L4_META_UDP,
        }, },
        [HASH_TYPE_IPV6_TCP_PRE] = { .d = {
            .outer.l3_type = DOCA_FLOW_L3_TYPE_IP6,
            .outer.ip6.src_ip[0] = BE32_MAX,
            .outer.ip6.src_ip[1] = BE32_MAX,
            .outer.ip6.src_ip[2] = BE32_MAX,
            .outer.ip6.src_ip[3] = BE32_MAX,
            .outer.l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_TCP,
            .outer.tcp.l4_port.src_port = BE16_MAX,
            .parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV6,
            .parser_meta.outer_l4_type = DOCA_FLOW_L4_META_TCP,
        }, },
        [HASH_TYPE_IPV6_L3_PRE] = { .d = {
            .outer.l3_type = DOCA_FLOW_L3_TYPE_IP6,
            .outer.ip6.src_ip[0] = BE32_MAX,
            .outer.ip6.src_ip[1] = BE32_MAX,
            .outer.ip6.src_ip[2] = BE32_MAX,
            .outer.ip6.src_ip[3] = BE32_MAX,
            .parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV6,
        }, },
        [HASH_TYPE_IPV6_L3_JUMBO] = { .d = {
            .outer.l3_type = DOCA_FLOW_L3_TYPE_IP6,
            .outer.ip6.src_ip[0] = BE32_MAX,
            .outer.ip6.src_ip[1] = BE32_MAX,
            .outer.ip6.src_ip[2] = BE32_MAX,
            .outer.ip6.src_ip[3] = BE32_MAX,
            .outer.ip6.dst_ip[0] = BE32_MAX,
            .outer.ip6.dst_ip[1] = BE32_MAX,
            .outer.ip6.dst_ip[2] = BE32_MAX,
            .outer.ip6.dst_ip[3] = BE32_MAX,
            .parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV6,
        }, },
        [HASH_TYPE_IPV6_UDP_JUMBO] = { .d = {
            .outer.l3_type = DOCA_FLOW_L3_TYPE_IP6,
            .outer.ip6.src_ip[0] = BE32_MAX,
            .outer.ip6.src_ip[1] = BE32_MAX,
            .outer.ip6.src_ip[2] = BE32_MAX,
            .outer.ip6.src_ip[3] = BE32_MAX,
            .outer.ip6.dst_ip[0] = BE32_MAX,
            .outer.ip6.dst_ip[1] = BE32_MAX,
            .outer.ip6.dst_ip[2] = BE32_MAX,
            .outer.ip6.dst_ip[3] = BE32_MAX,
            .outer.l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_UDP,
            .outer.udp.l4_port.src_port = BE16_MAX,
            .outer.udp.l4_port.dst_port = BE16_MAX,
            .parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV6,
            .parser_meta.outer_l4_type = DOCA_FLOW_L4_META_UDP,
        }, },
        [HASH_TYPE_IPV6_TCP_JUMBO] = { .d = {
            .outer.l3_type = DOCA_FLOW_L3_TYPE_IP6,
            .outer.ip6.src_ip[0] = BE32_MAX,
            .outer.ip6.src_ip[1] = BE32_MAX,
            .outer.ip6.src_ip[2] = BE32_MAX,
            .outer.ip6.src_ip[3] = BE32_MAX,
            .outer.ip6.dst_ip[0] = BE32_MAX,
            .outer.ip6.dst_ip[1] = BE32_MAX,
            .outer.ip6.dst_ip[2] = BE32_MAX,
            .outer.ip6.dst_ip[3] = BE32_MAX,
            .outer.l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_TCP,
            .outer.tcp.l4_port.src_port = BE16_MAX,
            .outer.tcp.l4_port.dst_port = BE16_MAX,
            .parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV6,
            .parser_meta.outer_l4_type = DOCA_FLOW_L4_META_TCP,
        }, },
    };
    struct doca_flow_action_descs *descs_arr[1];
    struct doca_flow_actions *actions_arr[1];
    struct ovs_doca_flow_actions actions;
    struct doca_offload_esw_ctx *esw_ctx;
    struct doca_flow_pipe_entry **pentry;
    struct doca_flow_action_descs descs;
    struct doca_flow_action_desc desc;
    struct doca_flow_fwd fwd, miss;
    struct doca_flow_pipe_cfg *cfg;
    struct doca_flow_pipe **ppipe;
    char pipe_name[50];
    int ret;

    esw_ctx = doca_offload_esw_ctx_get(netdev);

    ppipe = &hash_pipe_ctx->hashes[type].pipe;
    pentry = &hash_pipe_ctx->hashes[type].entry;

    snprintf(pipe_name, sizeof pipe_name, "OVS_HASH_PIPE_type_%u", type);

    memset(&fwd, 0, sizeof(fwd));
    memset(&miss, 0, sizeof(miss));
    memset(&descs, 0, sizeof(descs));
    memset(&actions, 0, sizeof(actions));
    memset(&desc, 0, sizeof desc);

    ret = doca_flow_pipe_cfg_create(&cfg, esw_ctx->esw_port);
    if (ret) {
        VLOG_ERR("%s: Could not create doca_flow_pipe_cfg", netdev_get_name(netdev));
        return ret;
    }
    if (doca_flow_pipe_cfg_set_name(cfg, pipe_name) ||
        doca_flow_pipe_cfg_set_type(cfg, DOCA_FLOW_PIPE_HASH) ||
        doca_flow_pipe_cfg_set_nr_entries(cfg, 1)) {
        VLOG_ERR("%s: Could not set doca_flow_pipe_cfg", netdev_get_name(netdev));
        ret = -1;
        goto out;
    }
    descs_arr[0] = &descs;
    descs.desc_array = &desc;
    descs.nb_action_desc = 1;
    actions_arr[0] = &actions.d;

    desc.type = DOCA_FLOW_ACTION_COPY;
    desc.field_op.src.field_string = "parser_meta.hash.result";
    desc.field_op.src.bit_offset = 0;
    desc.field_op.dst.field_string = "meta.data";
    desc.field_op.dst.bit_offset = doca_get_reg_bit_offset(REG_FIELD_DP_HASH);
    desc.field_op.width = doca_get_reg_width(REG_FIELD_DP_HASH);

    if (type >= HASH_TYPE_IPV6_UDP_PRE && type <= HASH_TYPE_IPV6_L3_PRE) {
        uint32_t offset = HASH_TYPE_IPV6_UDP_SUF - HASH_TYPE_IPV6_UDP_PRE;

        next_pipe = hash_pipe_ctx->hashes[type + offset].pipe;

        desc.field_op.dst.bit_offset =
            doca_get_reg_bit_offset(REG_FIELD_SCRATCH);
        desc.field_op.width = doca_get_reg_width(REG_FIELD_SCRATCH);
    }

    /* For SUF pipes, it's the hash result of the PRE pipes. For others it's
     * the seed.
     */
    doca_set_reg_mask(&hash_matches[type].d.meta, REG_FIELD_SCRATCH);

    fwd.type = DOCA_FLOW_FWD_PIPE;
    fwd.next_pipe = next_pipe;
    miss.type = DOCA_FLOW_FWD_DROP;

    if (doca_flow_pipe_cfg_set_match(cfg, NULL, &hash_matches[type].d) ||
        doca_pipe_cfg_allow_one_queue(cfg, AUX_QUEUE) ||
        doca_flow_pipe_cfg_set_actions(cfg, actions_arr, NULL, descs_arr, 1)) {
        VLOG_ERR("%s: Could not set doca_flow_pipe_cfg", netdev_get_name(netdev));
        ret = -1;
        goto out;
    }
    ret = doca_flow_pipe_create(cfg, &fwd, &miss, ppipe);
    if (ret) {
        if ((doca_error_t) ret != DOCA_ERROR_TOO_BIG) {
            VLOG_ERR("%s: Failed to create hash pipe: %d (%s)", netdev_get_name(netdev),
                     ret, doca_error_get_descr(ret));
        }
        goto out;
    }

    ret = doca_offload_add_entry_hash(netdev, queue_id, *ppipe, 0, NULL, NULL, NULL,
                                      DOCA_FLOW_ENTRY_FLAGS_NO_WAIT, pentry);
    if (ret) {
        VLOG_ERR("%s: Failed to create hash pipe entry. Error: %d (%s)", netdev_get_name(netdev),
                 ret, doca_error_get_descr(ret));
        goto out;
    }
out:
    doca_flow_pipe_cfg_destroy(cfg);
    return ret;
}

static int
doca_hash_pipe_ctx_init(struct netdev *netdev, struct doca_offload_esw_ctx *ctx)
{
    unsigned int queue_id = netdev_offload_thread_id();
    struct doca_hash_pipe_ctx *hash_pipe_ctx;
    struct doca_flow_pipe_entry **pentry;
    struct ovs_doca_flow_match spec;
    struct ovs_doca_flow_match mask;
    struct doca_flow_fwd miss;
    enum hash_nw_type nw_type;
    struct doca_flow_fwd fwd;
    char pipe_name[50];
    doca_error_t err;
    int type;

    hash_pipe_ctx = xzalloc(sizeof *hash_pipe_ctx);
    ctx->hash_pipe_ctx = hash_pipe_ctx;

    hash_pipe_ctx->post_hash_group_ctx =
        doca_pipe_group_ctx_ref(netdev, POSTHASH_TABLE_ID, NULL);
    if (hash_pipe_ctx->post_hash_group_ctx == NULL) {
        VLOG_ERR("%s: Failed to create post-hash group", netdev_get_name(netdev));
        goto pipe_ctx_uninit;
    }

    for (type = 0; type < NUM_HASH_PIPE_TYPE; type++) {
        int ret;

        ret = doca_hash_pipe_init(netdev, queue_id, hash_pipe_ctx, type);
        if ((doca_error_t) ret == DOCA_ERROR_TOO_BIG && HASH_PIPE_TYPE_IS_JUMBO(type)) {
            continue;
        }
        if (ret) {
            VLOG_ERR("%s: Failed to create hash pipe ctx",
                     netdev_get_name(netdev));
            goto pipe_ctx_uninit;
        }
    }

    /* Classifier pipe. */
    for (nw_type = 0; nw_type < NUM_HASH_NW; nw_type++) {
        struct doca_flow_pipe **ppipe;

        memset(&mask, 0, sizeof mask);
        memset(&spec, 0, sizeof spec);
        memset(&fwd, 0, sizeof fwd);
        fwd.type = DOCA_FLOW_FWD_PIPE;
        memset(&miss, 0, sizeof miss);
        miss.type = DOCA_FLOW_FWD_PIPE;

        if (nw_type == HASH_NW_IP4) {
            mask.d.outer.l3_type = DOCA_FLOW_L3_TYPE_IP4;
            mask.d.outer.ip4.next_proto = 0xFF;
            mask.d.parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV4;
            miss.next_pipe = hash_pipe_ctx->hashes[HASH_TYPE_IPV4_L3].pipe;
            snprintf(pipe_name, sizeof pipe_name, "HASH_IPv4_CLASSIFIER");
        } else {
            mask.d.outer.l3_type = DOCA_FLOW_L3_TYPE_IP6;
            mask.d.outer.ip6.next_proto = 0xFF;
            mask.d.parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV6;
            miss.next_pipe = hash_pipe_ctx->hashes[HASH_TYPE_IPV6_L3_JUMBO].pipe;
            if (!miss.next_pipe) {
                miss.next_pipe = hash_pipe_ctx->hashes[HASH_TYPE_IPV6_L3_PRE].pipe;
            }
            snprintf(pipe_name, sizeof pipe_name, "HASH_IPv6_CLASSIFIER");
        }

        ppipe = &hash_pipe_ctx->classifier[nw_type].pipe;

        if (doca_offload_basic_pipe_create(netdev, &mask, NULL, NULL, NULL, NULL, NULL, &fwd,
                                           &miss, 2, UINT64_C(1) << AUX_QUEUE, pipe_name, ppipe)) {
            goto pipe_ctx_uninit;
        }

        /* TCP/UDP entries. */
        pentry = &hash_pipe_ctx->classifier[nw_type].tcpudp[HASH_TP_UDP];
        if (nw_type == HASH_NW_IP4) {
            spec.d.outer.l3_type = DOCA_FLOW_L3_TYPE_IP4;
            spec.d.outer.ip4.next_proto = IPPROTO_UDP;
            spec.d.parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV4;
            fwd.next_pipe = hash_pipe_ctx->hashes[HASH_TYPE_IPV4_UDP].pipe;
        } else {
            spec.d.outer.l3_type = DOCA_FLOW_L3_TYPE_IP6;
            spec.d.outer.ip6.next_proto = IPPROTO_UDP;
            spec.d.parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV6;
            fwd.next_pipe = hash_pipe_ctx->hashes[HASH_TYPE_IPV6_UDP_JUMBO].pipe;
            if (!fwd.next_pipe) {
               fwd.next_pipe = hash_pipe_ctx->hashes[HASH_TYPE_IPV6_UDP_PRE].pipe;
            }
        }
        err = doca_offload_add_entry(netdev, queue_id, *ppipe, &spec, NULL, NULL, &fwd,
                                     DOCA_FLOW_ENTRY_FLAGS_NO_WAIT, pentry);
        if (err) {
            VLOG_ERR("%s: Failed to create UDP classifier entry: %d (%s)",
                     netdev_get_name(netdev), err, doca_error_get_descr(err));
            goto pipe_ctx_uninit;
        }

        pentry = &hash_pipe_ctx->classifier[nw_type].tcpudp[HASH_TP_TCP];
        if (nw_type == HASH_NW_IP4) {
            spec.d.outer.l3_type = DOCA_FLOW_L3_TYPE_IP4;
            spec.d.outer.ip4.next_proto = IPPROTO_TCP;
            spec.d.parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV4;
            fwd.next_pipe = hash_pipe_ctx->hashes[HASH_TYPE_IPV4_TCP].pipe;
        } else {
            spec.d.outer.l3_type = DOCA_FLOW_L3_TYPE_IP6;
            spec.d.outer.ip6.next_proto = IPPROTO_TCP;
            spec.d.parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV6;
            fwd.next_pipe = hash_pipe_ctx->hashes[HASH_TYPE_IPV6_TCP_JUMBO].pipe;
            if (!fwd.next_pipe) {
                fwd.next_pipe = hash_pipe_ctx->hashes[HASH_TYPE_IPV6_TCP_PRE].pipe;
            }
        }
        err = doca_offload_add_entry(netdev, queue_id, *ppipe, &spec, NULL, NULL, &fwd,
                                     DOCA_FLOW_ENTRY_FLAGS_NO_WAIT, pentry);
        if (err) {
            VLOG_ERR("%s: Failed to create TCP classifier entry: %d (%s)",
                     netdev_get_name(netdev), err, doca_error_get_descr(err));
            goto pipe_ctx_uninit;
        }
    }

    return 0;

pipe_ctx_uninit:
    doca_hash_pipe_ctx_uninit(ctx);
    return -1;
}

static struct doca_flow_pipe *
get_group_root(struct doca_hash_pipe_ctx *hash_pipe_ctx,
               struct doca_pipe_group_ctx *next_group_ctx,
               struct doca_flow_match *spec,
               struct doca_flow_actions *dacts,
               struct doca_act_vars *dact_vars)
{
    enum doca_flow_l3_type outer_l3_type = DOCA_FLOW_L3_TYPE_NONE;
    enum doca_flow_tun_type tun_type = DOCA_FLOW_TUN_NONE;
    bool has_dp_hash = !!dact_vars->hash_data;
    bool is_ipv4, is_ipv6, is_tcp, is_udp;
    enum doca_flow_l3_meta l3_type;
    enum doca_flow_l4_meta l4_type;

    if (!has_dp_hash) {
        return doca_pipe_group_get_pipe(next_group_ctx);
    }

    if (dacts->encap_type == DOCA_FLOW_RESOURCE_TYPE_NON_SHARED) {
        tun_type = dacts->encap_cfg.encap.tun.type;
        outer_l3_type = dacts->encap_cfg.encap.outer.l3_type;
    } else if (dacts->encap_type == DOCA_FLOW_RESOURCE_TYPE_SHARED) {
        tun_type = dact_vars->shared_encap_tun_type;
        outer_l3_type = dact_vars->shared_encap_l3_type;
    }
    if (dacts->encap_type) {
        if (outer_l3_type == DOCA_FLOW_L3_TYPE_IP4) {
            if (tun_type == DOCA_FLOW_TUN_VXLAN ||
                tun_type == DOCA_FLOW_TUN_GENEVE) {
                return hash_pipe_ctx->hashes[HASH_TYPE_IPV4_UDP].pipe;
            }
            return hash_pipe_ctx->hashes[HASH_TYPE_IPV4_L3].pipe;
        }

        if (outer_l3_type == DOCA_FLOW_L3_TYPE_IP6) {
            struct doca_flow_pipe *jumbo_pipe;

            if (tun_type == DOCA_FLOW_TUN_VXLAN ||
                tun_type == DOCA_FLOW_TUN_GENEVE) {
                jumbo_pipe = hash_pipe_ctx->hashes[HASH_TYPE_IPV6_UDP_JUMBO].pipe;
                if (jumbo_pipe) {
                    return jumbo_pipe;
                }
                return hash_pipe_ctx->hashes[HASH_TYPE_IPV6_UDP_PRE].pipe;
            }
            jumbo_pipe = hash_pipe_ctx->hashes[HASH_TYPE_IPV6_L3_JUMBO].pipe;
            if (jumbo_pipe) {
                return jumbo_pipe;
            }
            return hash_pipe_ctx->hashes[HASH_TYPE_IPV6_L3_PRE].pipe;
        }
        return NULL;
    }

    if (dacts->decap_cfg.is_l2) {
        l3_type = spec->parser_meta.inner_l3_type;
        l4_type = spec->parser_meta.inner_l4_type;
    } else {
        l3_type = spec->parser_meta.outer_l3_type;
        l4_type = spec->parser_meta.outer_l4_type;
    }

    is_ipv4 = l3_type == DOCA_FLOW_L3_META_IPV4 ||
              dact_vars->l3_type_hint == htons(ETH_TYPE_IP);
    is_ipv6 = l3_type == DOCA_FLOW_L3_META_IPV6 ||
              dact_vars->l3_type_hint == htons(ETH_TYPE_IPV6);
    is_tcp = l4_type == DOCA_FLOW_L4_META_TCP ||
             dact_vars->l4_type_hint == IPPROTO_TCP;
    is_udp = l4_type == DOCA_FLOW_L4_META_UDP ||
             dact_vars->l4_type_hint == IPPROTO_UDP;

    if (is_ipv4) {
        if (is_tcp) {
            return hash_pipe_ctx->hashes[HASH_TYPE_IPV4_TCP].pipe;
        } else if (is_udp) {
            return hash_pipe_ctx->hashes[HASH_TYPE_IPV4_UDP].pipe;
        }
        return hash_pipe_ctx->classifier[HASH_NW_IP4].pipe;
    }

    if (is_ipv6) {
        struct doca_flow_pipe *jumbo_pipe;

        if (is_tcp) {
            jumbo_pipe = hash_pipe_ctx->hashes[HASH_TYPE_IPV6_TCP_JUMBO].pipe;
            if (jumbo_pipe) {
                return jumbo_pipe;
            }
            return hash_pipe_ctx->hashes[HASH_TYPE_IPV6_TCP_PRE].pipe;
        } else if (is_udp) {
            jumbo_pipe = hash_pipe_ctx->hashes[HASH_TYPE_IPV6_UDP_JUMBO].pipe;
            if (jumbo_pipe) {
                return jumbo_pipe;
            }
            return hash_pipe_ctx->hashes[HASH_TYPE_IPV6_UDP_PRE].pipe;
        }
        return hash_pipe_ctx->classifier[HASH_NW_IP6].pipe;
    }

    return NULL;
}

static bool
doca_translate_actions_is_valid_ethdev(struct netdev *netdev,
                                       const struct rte_flow_action_ethdev *ethdev)
{
    /* DOCA supports hairpin only for WIRE ports. In case of another
     * port, the rule is wrongly offloaded and packets will be dropped.
     * As a WA for this, deny non-wire hairpin offload. */
    if (netdev_doca_get_port_id(netdev) == ethdev->port_id &&
        netdev_doca_get_port_dir(netdev) != NETDEV_DOCA_PORT_DIR_RX) {
        return false;
    }

    return true;
}

static int
doca_translate_actions_jump(const struct rte_flow_action_jump *jump,
                            struct netdev *netdev,
                            struct doca_flow_match *spec,
                            struct doca_flow_actions *dacts,
                            struct doca_flow_fwd *fwd,
                            struct doca_flow_handle_resources *flow_res,
                            struct doca_act_vars *dact_vars,
                            struct rte_flow_error *error)
{
    struct doca_offload_esw_ctx *esw_ctx;

    esw_ctx = doca_offload_esw_ctx_get(netdev);

    fwd->type = DOCA_FLOW_FWD_PIPE;

    if (jump->group == MISS_TABLE_ID) {
        fwd->next_pipe = doca_offload_get_misspath_pipe(esw_ctx);
    } else if (doca_pipe_group_is_ct_zone_group_id(jump->group)) {
        uint32_t table_id = jump->group & ~reg_fields[REG_FIELD_CT_ZONE].mask;
        uint32_t zone = jump->group & reg_fields[REG_FIELD_CT_ZONE].mask;

        if (doca_ct_zone_init(netdev, esw_ctx, jump->group)) {
            doca_offload_rte_flow_error_set(error, EINVAL, RTE_FLOW_ERROR_TYPE_ACTION,
                                            "unable to initialize CT zone");
            return -1;
        }

        fwd->next_pipe = table_id == CT_TABLE_ID
                                     ? ovsrcu_get(struct doca_flow_pipe *,
                                                  &esw_ctx->ct_zones[zone].plain_pipe)
                                     : ovsrcu_get(struct doca_flow_pipe *,
                                                  &esw_ctx->ct_zones[zone].nat_pipe);
    } else {
        struct doca_pipe_group_ctx *next_group_ctx;

        next_group_ctx = doca_pipe_group_ctx_ref(netdev, jump->group, NULL);
        if (!next_group_ctx) {
            doca_offload_rte_flow_error_set(error, EINVAL, RTE_FLOW_ERROR_TYPE_ACTION,
                                            "unable to get pipe group context");
            return -1;
        }

        fwd->next_pipe = get_group_root(esw_ctx->hash_pipe_ctx,
                                        next_group_ctx, spec, dacts, dact_vars);
        if (!fwd->next_pipe) {
            doca_offload_rte_flow_error_set(error, EINVAL, RTE_FLOW_ERROR_TYPE_ACTION,
                                            "unable to get root group pipe");
            doca_pipe_group_ctx_unref(next_group_ctx);
            return -1;
        }
        flow_res->next_group_ctx = next_group_ctx;
    }

    return 0;
}

static int
doca_translate_actions_nested(struct netdev *netdev,
                              const struct rte_flow_action_sample *sample,
                              struct ovs_mirror_set *mirror_set,
                              struct doca_flow_handle_resources *flow_res,
                              struct rte_flow_error *error)
{
    const struct rte_flow_action *rte_actions = sample->actions;

    if (sample->ratio != 1 || !rte_actions) {
        doca_offload_rte_flow_error_set(error, EINVAL, RTE_FLOW_ERROR_TYPE_ACTION,
                                        "incorrect sample action or ratio");
        return -1;
    }

    for (; rte_actions->type != RTE_FLOW_ACTION_TYPE_END; rte_actions++) {
        struct ovs_mirror_target *target = &mirror_set->target[mirror_set->nr_targets];

        if (rte_actions->type == RTE_FLOW_ACTION_TYPE_REPRESENTED_PORT) {
            const struct rte_flow_action_ethdev *ethdev = rte_actions->conf;

            if (!doca_translate_actions_is_valid_ethdev(netdev, ethdev)) {
                doca_offload_rte_flow_error_set(error, EINVAL, RTE_FLOW_ERROR_TYPE_ACTION,
                                                "invalid ethdev");
                return -1;
            }

            if (mirror_set->nr_targets == DOCA_MIRRORS_MAX_TARGETS) {
                doca_offload_rte_flow_error_set(error, EOPNOTSUPP, RTE_FLOW_ERROR_TYPE_ACTION,
                                                "mirror targets at maximum in nested port");
                return -1;
            }

            target->fwd.type = DOCA_FLOW_FWD_PORT;
            target->fwd.port_id = ethdev->port_id;
            mirror_set->nr_targets++;
        } else if (rte_actions->type == RTE_FLOW_ACTION_TYPE_JUMP) {
            struct doca_act_vars dact_vars = { .hash_data = NULL, };

            if (mirror_set->nr_targets == DOCA_MIRRORS_MAX_TARGETS) {
                doca_offload_rte_flow_error_set(error, EOPNOTSUPP, RTE_FLOW_ERROR_TYPE_ACTION,
                                                "mirror targets at maximum in nested jump");
                return -1;
            }

            if (doca_translate_actions_jump(rte_actions->conf, netdev, NULL, NULL,
                                            &target->fwd, flow_res, &dact_vars, error)) {
                return -1;
            }

            mirror_set->nr_targets++;
        } else if (rte_actions->type == RTE_FLOW_ACTION_TYPE_RAW_ENCAP) {
            const struct raw_encap_data *data = rte_actions->conf;

            if (doca_translate_raw_encap(data, doca_offload_esw_ctx_get(netdev),
                                         &target->actions, &target->actions_mask, NULL,
                                         error)) {
                return -1;
            }
        } else if (rte_actions->type == RTE_FLOW_ACTION_TYPE_DROP) {
            /* Ignore. */
        } else {
            doca_offload_rte_flow_error_set(error, EOPNOTSUPP, RTE_FLOW_ERROR_TYPE_ACTION,
                                            "unsupported nested action type");
            return -1;
        }
    }

    return 0;
}

static int
doca_translate_actions(struct netdev *netdev,
                       struct doca_flow_match *spec,
                       struct doca_flow_match *mask,
                       const struct rte_flow_action *actions,
                       struct ovs_doca_flow_actions *odacts,
                       struct ovs_doca_flow_actions *odacts_masks,
                       struct doca_flow_fwd *fwd,
                       struct doca_flow_monitor *monitor,
                       struct doca_flow_handle_resources *flow_res,
                       struct doca_act_vars *dact_vars,
                       struct rte_flow_error *error)
{
    const struct rte_flow_action_sample *nested_sample = NULL;
    struct doca_flow_header_format *outer, *outer_masks;
    struct ovs_mirror_set mirror_set = {
        .nr_targets = 0,
        .target = (struct ovs_mirror_target[DOCA_MIRRORS_MAX_TARGETS]) {
        },
    };
    struct doca_flow_actions *dacts, *dacts_masks;
    struct doca_offload_esw_ctx *esw_ctx;
    bool vlan_act_push = false;

    dacts = &odacts->d;
    dacts_masks = &odacts_masks->d;
    outer = &dacts->outer;
    outer_masks = &dacts_masks->outer;

    esw_ctx = doca_offload_esw_ctx_get(netdev);
    for (; actions->type != RTE_FLOW_ACTION_TYPE_END; actions++) {
        int act_type = actions->type;

        if (act_type == RTE_FLOW_ACTION_TYPE_DROP) {
            fwd->type = DOCA_FLOW_FWD_DROP;
        } else if (act_type == RTE_FLOW_ACTION_TYPE_SET_MAC_SRC) {
            const struct action_set_data *asd = actions->conf;

            memcpy(&outer->eth.src_mac, asd->value, asd->size);
            memcpy(&outer_masks->eth.src_mac, asd->mask, asd->size);
        } else if (act_type == RTE_FLOW_ACTION_TYPE_SET_MAC_DST) {
            const struct action_set_data *asd = actions->conf;

            memcpy(&outer->eth.dst_mac, asd->value, asd->size);
            memcpy(&outer_masks->eth.dst_mac, asd->mask, asd->size);
        } else if (act_type == RTE_FLOW_ACTION_TYPE_OF_SET_VLAN_VID) {
            const struct rte_flow_action_of_set_vlan_vid *rte_vlan_vid;

            rte_vlan_vid = actions->conf;
            /* If preceeded by vlan push action, this is a new
             * vlan tag. Otherwise, perfrom vlan modification.
             */
            if (vlan_act_push) {
                dacts->push.type = DOCA_FLOW_PUSH_ACTION_VLAN;
                dacts_masks->push.type = DOCA_FLOW_PUSH_ACTION_VLAN;
                dacts->push.vlan.eth_type = RTE_BE16(DOCA_FLOW_ETHER_TYPE_VLAN);
                dacts->push.vlan.vlan_hdr.tci = (OVS_FORCE doca_be16_t) rte_vlan_vid->vlan_vid;
                memset(&dacts_masks->push.vlan, 0xFF, sizeof dacts_masks->push.vlan);
                dacts->has_push = true;
                dacts_masks->has_push = true;
            } else {
                outer->eth_vlan[0].tci = (OVS_FORCE doca_be16_t) rte_vlan_vid->vlan_vid;
                outer->l2_valid_headers = DOCA_FLOW_L2_VALID_HEADER_VLAN_0;
                memset(&outer_masks->eth_vlan[0].tci, 0xFF,
                       sizeof outer_masks->eth_vlan[0].tci);
                outer_masks->l2_valid_headers =
                    DOCA_FLOW_L2_VALID_HEADER_VLAN_0;
            }
        } else if (act_type == RTE_FLOW_ACTION_TYPE_SET_IPV4_SRC) {
            const struct action_set_data *asd = actions->conf;

            outer->l3_type = DOCA_FLOW_L3_TYPE_IP4;
            memcpy(&outer->ip4.src_ip, asd->value, asd->size);
            memcpy(&outer_masks->ip4.src_ip, asd->mask, asd->size);
        } else if (act_type == RTE_FLOW_ACTION_TYPE_SET_IPV4_DST) {
            const struct action_set_data *asd = actions->conf;

            outer->l3_type = DOCA_FLOW_L3_TYPE_IP4;
            memcpy(&outer->ip4.dst_ip, asd->value, asd->size);
            memcpy(&outer_masks->ip4.dst_ip, asd->mask, asd->size);
        } else if (act_type == OVS_RTE_FLOW_ACTION_TYPE(SET_IPV4_TTL)) {
            const struct action_set_data *asd = actions->conf;

            outer->l3_type = DOCA_FLOW_L3_TYPE_IP4;
            memcpy(&outer->ip4.ttl, asd->value, asd->size);
            memcpy(&outer_masks->ip4.ttl, asd->mask, asd->size);
        } else if (act_type == RTE_FLOW_ACTION_TYPE_SET_IPV4_DSCP) {
            const struct action_set_data *asd = actions->conf;

            outer->l3_type = DOCA_FLOW_L3_TYPE_IP4;
            memcpy(&outer->ip4.dscp_ecn, asd->value, asd->size);
            memcpy(&outer_masks->ip4.dscp_ecn, asd->mask, asd->size);
        } else if (act_type == OVS_RTE_FLOW_ACTION_TYPE(SET_IPV6_HOP)) {
            const struct action_set_data *asd = actions->conf;

            outer->l3_type = DOCA_FLOW_L3_TYPE_IP6;
            memcpy(&outer->ip6.hop_limit, asd->value, asd->size);
            memcpy(&outer_masks->ip6.hop_limit, asd->mask, asd->size);
        } else if (act_type == RTE_FLOW_ACTION_TYPE_SET_IPV6_SRC) {
            const struct action_set_data *asd = actions->conf;

            outer->l3_type = DOCA_FLOW_L3_TYPE_IP6;
            memcpy(&outer->ip6.src_ip, asd->value, asd->size);
            memcpy(&outer_masks->ip6.src_ip, asd->mask, asd->size);
        } else if (act_type == RTE_FLOW_ACTION_TYPE_SET_IPV6_DST) {
            const struct action_set_data *asd = actions->conf;

            outer->l3_type = DOCA_FLOW_L3_TYPE_IP6;
            memcpy(&outer->ip6.dst_ip, asd->value, asd->size);
            memcpy(&outer_masks->ip6.dst_ip, asd->mask, asd->size);
        } else if (act_type == RTE_FLOW_ACTION_TYPE_SET_IPV6_DSCP) {
            const struct action_set_data *asd = actions->conf;

            outer->l3_type = DOCA_FLOW_L3_TYPE_IP6;
            memcpy(&outer->ip6.traffic_class, asd->value, asd->size);
            memcpy(&outer_masks->ip6.traffic_class, asd->mask, asd->size);
        } else if (act_type == OVS_RTE_FLOW_ACTION_TYPE(SET_UDP_SRC)) {
            const struct action_set_data *asd = actions->conf;

            outer->l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_UDP;
            memcpy(&outer->udp.l4_port.src_port, asd->value, asd->size);
            memcpy(&outer_masks->udp.l4_port.src_port, asd->mask, asd->size);
        } else if (act_type == OVS_RTE_FLOW_ACTION_TYPE(SET_UDP_DST)) {
            const struct action_set_data *asd = actions->conf;

            outer->l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_UDP;
            memcpy(&outer->udp.l4_port.dst_port, asd->value, asd->size);
            memcpy(&outer_masks->udp.l4_port.dst_port, asd->mask, asd->size);
        } else if (act_type == OVS_RTE_FLOW_ACTION_TYPE(SET_TCP_SRC)) {
            const struct action_set_data *asd = actions->conf;

            outer->l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_TCP;
            memcpy(&outer->tcp.l4_port.src_port, asd->value, asd->size);
            memcpy(&outer_masks->tcp.l4_port.src_port, asd->mask, asd->size);
        } else if (act_type == OVS_RTE_FLOW_ACTION_TYPE(SET_TCP_DST)) {
            const struct action_set_data *asd = actions->conf;

            outer->l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_TCP;
            memcpy(&outer->tcp.l4_port.dst_port, asd->value, asd->size);
            memcpy(&outer_masks->tcp.l4_port.dst_port, asd->mask, asd->size);
        } else if (act_type == RTE_FLOW_ACTION_TYPE_REPRESENTED_PORT) {
            const struct rte_flow_action_ethdev *ethdev = actions->conf;

            if (!doca_translate_actions_is_valid_ethdev(netdev, ethdev)) {
                doca_offload_rte_flow_error_set(error, EOPNOTSUPP, RTE_FLOW_ERROR_TYPE_ACTION,
                                                "forward from VF to itself");
                return -1;
            }

            doca_port_id_match(spec, mask, dact_vars);
            fwd->type = DOCA_FLOW_FWD_PORT;
            fwd->port_id = ethdev->port_id;
        } else if (act_type == RTE_FLOW_ACTION_TYPE_NVGRE_DECAP ||
                   act_type == RTE_FLOW_ACTION_TYPE_VXLAN_DECAP ||
                   act_type == RTE_FLOW_ACTION_TYPE_RAW_DECAP) {
            dacts->decap_type = DOCA_FLOW_RESOURCE_TYPE_NON_SHARED;
            dacts->decap_cfg.is_l2 = true;
            dacts_masks->decap_type = DOCA_FLOW_RESOURCE_TYPE_NON_SHARED;
            dacts_masks->decap_cfg.is_l2 = true;
        } else if (act_type == RTE_FLOW_ACTION_TYPE_COUNT) {
            monitor->counter_type = DOCA_FLOW_RESOURCE_TYPE_NON_SHARED;
        } else if (act_type == RTE_FLOW_ACTION_TYPE_JUMP) {
            if (doca_translate_actions_jump(actions->conf, netdev, spec, dacts, fwd, flow_res,
                                            dact_vars, error)) {
                return -1;
            }
        } else if (act_type == RTE_FLOW_ACTION_TYPE_RAW_ENCAP) {
            const struct raw_encap_data *data = actions->conf;

            if (doca_translate_raw_encap(data, esw_ctx, odacts, odacts_masks, dact_vars, error)) {
                return -1;
            }
        } else if (act_type == OVS_RTE_FLOW_ACTION_TYPE(FLOW_INFO)) {
            const struct rte_flow_action_mark *mark = actions->conf;

            if (!dact_vars->mtr_flow_id) {
                doca_set_reg_val_mask(&dacts->meta, &dacts_masks->meta,
                                      REG_FIELD_FLOW_INFO, mark->id);
                dact_vars->flow_id = mark->id;
            }
        } else if (act_type == RTE_FLOW_ACTION_TYPE_SET_TAG) {
            const struct rte_flow_action_set_tag *set_tag = actions->conf;
            uint8_t index = set_tag->index;

            dacts->meta.u32[index] |= DOCA_HTOBE32(set_tag->data);
            dacts_masks->meta.u32[index] |= DOCA_HTOBE32(set_tag->mask);
        } else if (act_type == RTE_FLOW_ACTION_TYPE_OF_POP_VLAN) {
            /* Current support is for a single VLAN tag */
            if (dacts->pop_vlan) {
                doca_offload_rte_flow_error_set(error, EOPNOTSUPP, RTE_FLOW_ERROR_TYPE_ACTION,
                                                "more than single pop vlan action");
                return -1;
            }
            dacts->pop_vlan = true;
            dacts_masks->pop_vlan = true;
        } else if (act_type == RTE_FLOW_ACTION_TYPE_OF_PUSH_VLAN) {
            if (vlan_act_push) {
                doca_offload_rte_flow_error_set(error, EOPNOTSUPP, RTE_FLOW_ERROR_TYPE_ACTION,
                                                "more than single push vlan action");
                return -1;
            }
            vlan_act_push = true;
        } else if (act_type == RTE_FLOW_ACTION_TYPE_OF_SET_VLAN_PCP) {
            continue;
        } else if (act_type == OVS_RTE_FLOW_ACTION_TYPE(HASH)) {
            const struct hash_data *hash_data = actions->conf;

            if (!dact_vars->mtr_flow_id) {
                doca_set_reg_val_mask(&dacts->meta, &dacts_masks->meta,
                                      REG_FIELD_FLOW_INFO, hash_data->flow_id);
                doca_set_reg_val_mask(&dacts->meta, &dacts_masks->meta,
                                      REG_FIELD_SCRATCH, hash_data->seed);
            }
            dact_vars->hash_data = hash_data;
        } else if (act_type == RTE_FLOW_ACTION_TYPE_METER) {
            const struct meter_data *mtr_data = actions->conf;
            struct doca_flow_resource_meter_cfg meter_cfg;
            struct doca_shared_meter_ctx *meter_ctx;
            uint32_t flow_id;

            flow_id = shared_mtr_flow_id_alloc(esw_ctx, error);
            if (flow_id == -1) {
                error->type = RTE_FLOW_ERROR_TYPE_ACTION;
                VLOG_DBG_RL(&rl, "%s", error->message);
                return -1;
            }

            if (netdev_doca_meter_get_cfg(mtr_data->conf.mtr_id, &meter_cfg)) {
                doca_offload_rte_flow_error_set(error, ENXIO, RTE_FLOW_ERROR_TYPE_ACTION,
                                                "meter config not found");
                return -1;
            }

            meter_ctx = doca_get_shared_meter(mtr_data->conf.mtr_id, &meter_cfg, esw_ctx->esw_port,
                                              &esw_ctx->shared_meters_ctx);
            if (!meter_ctx) {
                doca_offload_rte_flow_error_set(error, ENXIO, RTE_FLOW_ERROR_TYPE_ACTION,
                                                "meter context not found");
                return -1;
            }

            if (monitor->meter_type == DOCA_FLOW_RESOURCE_TYPE_NONE) {
                /* first meter in a multi-meter action */
                monitor->meter_type = DOCA_FLOW_RESOURCE_TYPE_SHARED;
                monitor->shared_meter.shared_meter_id = meter_ctx->id;
                dact_vars->mtr_red_cnt_id = meter_ctx->red_id;
                dact_vars->mtr_green_cnt_id = meter_ctx->green_id;
                dact_vars->mtr_flow_id = flow_id;
                dact_vars->mtr_of_id = mtr_data->conf.mtr_id;
                dact_vars->flow_id = mtr_data->flow_id;
                doca_set_reg_val_mask(&dacts->meta, &dacts_masks->meta,
                                      REG_FIELD_FLOW_INFO, flow_id);
            } else {
                struct meter_info *mtr_info = xzalloc(sizeof *mtr_info);

                mtr_info->id = meter_ctx->id;
                mtr_info->of_id = mtr_data->conf.mtr_id;
                mtr_info->red_id = meter_ctx->red_id;
                mtr_info->green_id = meter_ctx->green_id;
                mtr_info->flow_id = flow_id;
                ovs_list_push_back(&dact_vars->next_meters,
                                   &mtr_info->list_node);
            }
        } else if (act_type == RTE_FLOW_ACTION_TYPE_SAMPLE ||
                   /* When sFlow sampling ratio is 1 the ovs action is
                    * translated into OVS_ACTION_ATTR_USERSPACE.
                    */
                   act_type == OVS_RTE_FLOW_ACTION_TYPE(USERSPACE)) {
            const struct rte_flow_action_sample *sample = actions->conf;

            dact_vars->sample_ratio = sample->ratio;
        } else if (act_type == OVS_RTE_FLOW_ACTION_TYPE(SFLOW)) {
            /* The sflow_id action parameter is not used with DOCA, but the
             * action still needs to be processed.
             */
            continue;
        } else if (act_type == RTE_FLOW_ACTION_TYPE_VOID) {
            continue;
        } else if (act_type == OVS_RTE_FLOW_ACTION_TYPE(NESTED)) {
            nested_sample = actions->conf;
            if (doca_translate_actions_nested(netdev, nested_sample, &mirror_set, flow_res,
                                              error)) {
                return -1;
            }
        } else {
            doca_offload_rte_flow_error_set(error, EOPNOTSUPP, RTE_FLOW_ERROR_TYPE_ACTION,
                                            "unsupported action type");
            return -1;
        }
    }

    outer_masks->l2_valid_headers = outer->l2_valid_headers;
    outer_masks->l3_type = outer->l3_type;
    outer_masks->l4_type_ext = outer->l4_type_ext;

    if (nested_sample && mirror_set.nr_targets != 0) {
        flow_res->mirror_ctx = doca_mirror_ctx_create(esw_ctx->doca_mirrors, &mirror_set, odacts,
                                                      odacts_masks, fwd);
        if (!flow_res->mirror_ctx) {
            doca_offload_rte_flow_error_set(error, EINVAL, RTE_FLOW_ERROR_TYPE_ACTION,
                                            "unable to get mirror context");
            return -1;
        }
    }

    return 0;
}

static uint16_t
doca_offload_ct_queue_id(unsigned int tid)
{
    if (conntrack_offload_doca_ct_enabled) {
        /* Threads that interact with DOCA-CT are either
         * polling threads (updates) or auxiliary threads (queries).
         * Queries are made with the queue ID used for entry creation.
         * Regular offload threads do not interact and do not have a queue
         * configured.
         *
         * Example with 2 offload threads, 3 polling threads:
         *
         *       [6|7|8]  DOCA-CT thread IDs
         *  x x x | | |
         * [0|1|2|3|4|5]  netdev-offload thread IDs
         *  | | | `-`-`-o polling
         *  | `-`-------o offload
         *  `-----------o main
         *
         * In this example,
         *    ovs_doca_n_offload_queues() is 6 (2 offload, 3 PMDs, 1 main)
         *    netdev_offload_thread_nb() is 2
         */
        ovs_assert(tid > netdev_offload_thread_nb());
        return ovs_doca_n_offload_queues() +
               (tid - netdev_offload_thread_nb() - 1);
    }
    return tid;
}

void
doca_offload_entry_process(struct doca_flow_pipe_entry *entry,
                           uint16_t qid,
                           enum doca_flow_entry_status status,
                           enum doca_flow_entry_op op,
                           void *aux)
{
    static const char *status_desc[] = {
        [DOCA_FLOW_ENTRY_STATUS_IN_PROCESS] = "in-process",
        [DOCA_FLOW_ENTRY_STATUS_SUCCESS] = "success",
        [DOCA_FLOW_ENTRY_STATUS_ERROR] = "failure",
    };
    static const char *op_desc[] = {
        [DOCA_FLOW_ENTRY_OP_ADD] = "add",
        [DOCA_FLOW_ENTRY_OP_DEL] = "del",
        [DOCA_FLOW_ENTRY_OP_UPD] = "mod",
        [DOCA_FLOW_ENTRY_OP_AGED] = "age",
    };
    bool error = status == DOCA_FLOW_ENTRY_STATUS_ERROR;
    struct ovs_doca_offload_queue *queues = aux;

    VLOG_RL(&rl, error ? VLL_ERR : VLL_DBG,
            "%s: [qid:%" PRIu16 "] %s aux=%p entry %p %s",
            __func__, qid, op_desc[op], aux, entry, status_desc[status]);

    if (queues && status != DOCA_FLOW_ENTRY_STATUS_IN_PROCESS) {
        queues[qid].n_waiting_entries--;
    }
}

/* Complete the queue 'qid' on the netdev's eswitch until 'min_room'
 * entries are available. If 'min_room' is 0, complete any available.
 */
static doca_error_t
doca_offload_complete_queue_n_protected(struct doca_offload_esw_ctx *esw,
                                        unsigned int qid, uint32_t min_room)
{
    struct ovs_doca_offload_queue *queue;
    long long int timeout_ms;
    unsigned int n_waiting;
    bool pipe_resizing;
    doca_error_t err;
    bool is_doca_ct;
    uint32_t room;
    int retries;

    is_doca_ct = qid >= ovs_doca_n_offload_queues();
    queue = &esw->offload_queues[qid];
    n_waiting = queue->n_waiting_entries;
    pipe_resizing = !is_doca_ct && ovs_doca_queue_use_resize(qid) &&
                    ovs_refcount_read(&esw->pipe_resizing) > 1;

    if (!pipe_resizing && n_waiting == 0) {
        COVERAGE_INC(doca_queue_empty);
        return DOCA_SUCCESS;
    }

    if (qid == AUX_QUEUE) {
        /* 10 seconds timeout when synchronizing the AUX_QUEUE. */
        timeout_ms = time_msec() + 10 * 1000;
    } else {
        timeout_ms = 0;
    }

    retries = 100;
    do {
        unsigned int n_processed;

        if (is_doca_ct) {
            err = doca_flow_ct_entries_process(esw->esw_port, qid,
                                               min_room, min_room,
                                               &room);
            /* If this assertion fails, DOCA has increased the size of their
             * internal CT queue, which breaks synchronous polling.
             * OVS_DOCA_CT_QUEUE_DEPTH must be updated. */
            if (min_room == OVS_DOCA_CT_QUEUE_DEPTH) {
                ovs_assert(room <= OVS_DOCA_CT_QUEUE_DEPTH);
            }
        } else {
            /* Use 'max_processed_entries' == 0 to always attempt processing
             * the full length of the queue. */
            err = doca_flow_entries_process(esw->esw_port, qid,
                                            ENTRY_PROCESS_TIMEOUT_US, 0);
        }
        if (err) {
            VLOG_WARN_RL(&rl, "eswitch %" PRIu32 ": Failed to process entries "
                         "in queue %u: %s", esw->esw_id, qid,
                         doca_error_get_descr(err));
            return err;
        }

        n_processed = n_waiting - queue->n_waiting_entries;
        if (min_room && n_processed == 0) {
            COVERAGE_INC(doca_queue_none_processed);
        }
        n_waiting = queue->n_waiting_entries;

        if (!is_doca_ct) {
            if (pipe_resizing && qid == FIRST_QUEUE) {
                ovs_doca_mgmt_queue_lock();
                err = doca_offload_complete_queue_n_protected(esw, AUX_QUEUE, min_room);
                ovs_doca_mgmt_queue_unlock();
            }

            doca_pipe_group_ctx_resize_end_list(&esw->resized_pipe_lists[qid]);

            room = OVS_DOCA_QUEUE_DEPTH - n_waiting;
            pipe_resizing = ovs_doca_queue_use_resize(qid) &&
                            ovs_refcount_read(&esw->pipe_resizing) > 1;
            if (pipe_resizing && n_processed == 0) {
                if (retries-- <= 0) {
                    COVERAGE_INC(doca_resize_block);
                    break;
                }
            }

            if (timeout_ms && time_msec() > timeout_ms) {
                ovs_abort(0, "Timeout reached trying to complete queue %u: %u remaining entries",
                          qid, n_waiting);
            }
        }
    } while (err == DOCA_SUCCESS && min_room &&
             ((room < min_room) || pipe_resizing));

    return err;
}

static doca_error_t
doca_offload_complete_queue_n(struct doca_offload_esw_ctx *esw,
                              unsigned int qid, uint32_t min_room)
    OVS_EXCLUDED(mgmt_queue_lock)
{
    doca_error_t err;

    if (ovs_doca_queue_use_resize(qid) || qid == AUX_QUEUE) {
        if (qid == AUX_QUEUE) {
            ovs_doca_mgmt_queue_lock();
            err = doca_offload_complete_queue_n_protected(esw, qid, min_room);
            ovs_doca_mgmt_queue_unlock();
        } else {
            err = doca_offload_complete_queue_n_protected(esw, qid, min_room);
        }
    } else {
        err = doca_offload_complete_queue_n_protected(esw, qid, min_room);
    }

    return err;
}

doca_error_t
doca_offload_complete_queue_esw(struct doca_offload_esw_ctx *esw, unsigned int qid, bool sync)
    OVS_EXCLUDED(mgmt_queue_lock)
{
    uint32_t min_room = 0;

    if (esw == NULL) {
        return DOCA_SUCCESS;
    }

    if (sync) {
        bool is_doca_ct = (qid >= ovs_doca_n_offload_queues());

        min_room = is_doca_ct ? OVS_DOCA_CT_QUEUE_DEPTH
                              : OVS_DOCA_QUEUE_DEPTH;
    }

    return doca_offload_complete_queue_n(esw, qid, min_room);
}

static void
doca_offload_resolve_pipe_resize(struct doca_offload_esw_ctx *esw)
{
    /* Use a 10 seconds timeout. */
    long long int timeout_time_ms = time_msec() + (10 * 1000);
    bool pipe_resizing;
    unsigned int qid;

    pipe_resizing = ovs_refcount_read(&esw->pipe_resizing) > 1;
    while (pipe_resizing) {
        if (time_msec() >= timeout_time_ms) {
            VLOG_ERR("Pipe resize resolution took more than 10 seconds!!");
            break;
        }
        for (qid = 0; qid < ovs_doca_n_offload_queues(); qid++) {
            if (!ovs_doca_queue_use_resize(qid)) {
                continue;
            }
            doca_offload_complete_queue_n_protected(esw, qid, 0);
        }
        pipe_resizing = ovs_refcount_read(&esw->pipe_resizing) > 1;
    }
}

static void
dpdk_offload_doca_upkeep(struct netdev *netdev, bool quiescing)
{
    unsigned int qid = netdev_offload_thread_id();
    struct doca_offload_esw_ctx *esw;
    struct destroy_pipe_elem *elem;
    bool is_doca_ct;
    bool is_pmd;

    /* Safe to get an eswitch context after verify netdev is an 'ethdev'. */
    esw = doca_offload_esw_ctx_get(netdev);

    if (!esw) {
        return;
    }

    if (!netdev_doca_is_esw_mgr(netdev)) {
        return;
    }

    doca_offload_complete_queue_esw(esw, qid, quiescing);

    is_doca_ct = qid >= ovs_doca_n_offload_queues();
    is_pmd = qid > netdev_offload_thread_nb() && !is_doca_ct;

    if (conntrack_offload_doca_ct_enabled && is_pmd) {
        /* If DOCA-CT has been initialized, poll the CT queues as well from
         * threads in the polling threads range. */
        doca_offload_complete_queue_esw(esw, doca_offload_ct_queue_id(qid), quiescing);
    }

    if (!ovs_list_is_empty(&esw->destroy_pipe_lists[qid])) {
        /* If this assert triggers, it means multiple hw-offload thread have been
         * destroying pipes. In such case, the implementation of destroy_pipe_lists
         * must be reworked. */
        ovs_assert(qid == FIRST_QUEUE);
        /* Ensure no pipe resize is currently occurring.
         * This is a necessary condition before destroying any pipe. */
        doca_offload_resolve_pipe_resize(esw);
        LIST_FOR_EACH_POP (elem, node, &esw->destroy_pipe_lists[qid]) {
            doca_flow_pipe_destroy(elem->pipe);
            free(elem);
        }
    }
}

static struct doca_flow_pipe_entry *
create_doca_pipe_group_entry(struct netdev *netdev,
                             unsigned int queue_id,
                             struct doca_pipe_group_ctx *self_group_ctx,
                             uint32_t prio,
                             struct ovs_doca_flow_match *spec,
                             struct ovs_doca_flow_match *mask,
                             struct ovs_doca_flow_actions *actions,
                             struct ovs_doca_flow_actions *actions_masks,
                             struct doca_flow_action_descs *dacts_descs,
                             struct doca_flow_monitor *monitor,
                             struct doca_flow_fwd *fwd,
                             struct doca_pipe_group_mask_ctx **mask_ctx,
                             struct rte_flow_error *error)
    OVS_EXCLUDED(mgmt_queue_lock)
{
    struct doca_flow_pipe_entry *entry;
    doca_error_t err;

    err = doca_pipe_group_add_entry(netdev, queue_id, prio, self_group_ctx,
                                    spec, mask, actions, actions_masks,
                                    dacts_descs, monitor, fwd, mask_ctx, &entry);
    if (err) {
        if ((doca_error_t) err != DOCA_ERROR_TOO_BIG) {
            VLOG_WARN_RL(&rl, "%s: Failed to create group entry. Error: %d (%s)",
                         netdev_get_name(netdev), err, doca_error_get_descr(err));
        }
        error->type = (enum rte_flow_error_type) err;
        error->message = doca_error_get_descr(err);
        return NULL;
    }

    ovs_assert(entry);

    return entry;
}

static struct doca_flow_pipe *
doca_get_ct_pipe(struct doca_offload_esw_ctx *ctx,
                 struct doca_flow_match *spec)
{
    enum ct_nw_type nw_type;
    enum ct_tp_type tp_type;

    if (ctx == NULL) {
        return NULL;
    }

    nw_type = l3_to_nw_type(spec->parser_meta.outer_l3_type);
    if (nw_type >= NUM_CT_NW) {
        VLOG_DBG_RL(&rl, "Unsupported CT network type.");
        return NULL;
    }

    /* The default mode is to handle IPv4 connections in DOCA-CT and IPv6 connections
     * on legacy basic pipes, through the split-prefix table.
     * DOCA-CT can be completely disabled, in which case all connections use the
     * basic pipes. It can also be enabled manually for IPv6.
     */
    if (ctx->ct.pipe &&
        ((nw_type == CT_NW_IP4 && conntrack_offload_doca_ct_enabled) ||
         (nw_type == CT_NW_IP6 && conntrack_offload_doca_ct_ipv6_enabled))) {
        return ctx->ct.pipe;
    }

    if (nw_type == CT_NW_IP6) {
        return ctx->ct_ip6_prefix.pipe;
    }

    tp_type = l4_to_tp_type(spec->parser_meta.outer_l4_type);
    if (tp_type >= NUM_CT_TP) {
        VLOG_DBG_RL(&rl, "Unsupported CT protocol type.");
        return NULL;
    }

    return ctx->ct_pipes[nw_type][tp_type].pipe;
}

static int
doca_split_prefix_ctx_init(void *ctx_, void *arg_)
{
    unsigned int tid = netdev_offload_thread_id();
    struct doca_split_prefix_ctx *ctx = ctx_;
    struct doca_split_prefix_arg *arg = arg_;
    uint32_t id;

    if (!id_fpool_new_id(split_prefix_id_pool, tid, &id)) {
        return -1;
    }
    ctx->id = id;

    if (!arg->prefix_is_group) {
        struct ovs_doca_flow_actions dacts;
        struct doca_flow_fwd fwd;
        unsigned int queue_id;
        doca_error_t err;

        memset(&dacts, 0, sizeof dacts);
        memset(&fwd, 0, sizeof fwd);

        doca_set_reg_val(&dacts.d.meta, REG_FIELD_SCRATCH, id);
        fwd.type = DOCA_FLOW_FWD_PIPE;
        fwd.next_pipe = arg->suffix_pipe;
        queue_id = AUX_QUEUE;

        err = doca_offload_add_entry(arg->netdev, queue_id, arg->prefix_pipe, arg->spec, &dacts,
                                     NULL, &fwd, DOCA_FLOW_ENTRY_FLAGS_WAIT_FOR_BATCH,
                                     &ctx->doh.dfh.flow);
        if (err) {
            VLOG_WARN_RL(&rl, "%s: Failed to create basic split prefix entry:"
                         " Error %d (%s)", netdev_get_name(arg->netdev),
                         err, doca_error_get_descr(err));
            return -1;
        }

        ctx->queue_id = queue_id;
    }

    ovs_mutex_init(&ctx->handle_lock);
    ctx->netdev = arg->netdev;

    return 0;
}

static void
doca_split_prefix_ctx_uninit(void *ctx_)
{
    unsigned int tid = netdev_offload_thread_id();
    struct doca_split_prefix_ctx *ctx = ctx_;
    struct doca_offload_esw_ctx *esw;
    struct rte_flow_error error;

    esw = doca_offload_esw_ctx_get(ctx->netdev);
    id_fpool_free_id(split_prefix_id_pool, tid, ctx->id);
    if (ctx->doh.dfh.flow) {
        destroy_dpdk_offload_handle__(esw, &ctx->doh, ctx->queue_id, &error);
    }
    ovs_mutex_destroy(&ctx->handle_lock);
}

static struct ds *
dump_split_prefix_id(struct ds *s, void *key_, void *ctx_, void *arg OVS_UNUSED)
{
    struct doca_split_prefix_key *key = key_;
    struct doca_split_prefix_ctx *ctx = ctx_;
    struct doca_flow_handle *hndl;

    if (ctx) {
        hndl = &ctx->doh.dfh;
        ds_put_format(s, "prefix_flow=%p, ", hndl->flow);
    }
    ds_put_format(s, "prefix_pipe=%p, ", key->prefix_pipe);

    return s;
}

static void
split_prefix_id_init(void)
{
    static struct ovsthread_once init_once = OVSTHREAD_ONCE_INITIALIZER;

    if (ovsthread_once_start(&init_once)) {
        split_prefix_id_pool = id_fpool_create(MAX_OFFLOAD_THREAD_NB, MIN_SPLIT_PREFIX_ID,
                                               NUM_SPLIT_PREFIX_ID);
        split_prefix_rfm = refmap_create("prefix-id", sizeof(struct doca_split_prefix_key),
                                         sizeof(struct doca_split_prefix_ctx),
                                         doca_split_prefix_ctx_init, doca_split_prefix_ctx_uninit,
                                         dump_split_prefix_id, false);
        ovsthread_once_done(&init_once);
    }
}

static struct doca_split_prefix_ctx *
split_prefix_ctx_ref(struct doca_split_prefix_key *key,
                     struct doca_split_prefix_arg *args,
                     uint32_t *prefix_id)
{
    struct doca_split_prefix_ctx *ctx;

    split_prefix_id_init();
    ctx = refmap_ref(split_prefix_rfm, key, args);
    if (ctx) {
        *prefix_id = ctx->id;
    }
    return ctx;
}

static void
split_prefix_ctx_unref(struct doca_split_prefix_ctx *ctx)
{
    refmap_unref(split_prefix_rfm, ctx);
}

static void
reset_split_header_proto(enum split_field_layer layer, uint32_t proto,
                         struct doca_flow_header_format *smask,
                         void *sspec_proto, void *smask_proto, size_t proto_size)
{
    switch (layer) {
    case L2_HEADERS:
        if (!is_all_zeros(&smask->eth, sizeof smask->eth)) {
            return;
        }
        break;
    case L3_HEADERS:
        if (proto == DOCA_FLOW_L3_TYPE_IP4) {
            if (!is_all_zeros(&smask->ip4, sizeof smask->ip4)) {
                return;
            }
        } else if (!is_all_zeros(&smask->ip6, sizeof smask->ip6)) {
            return;
        }
        break;
    case L4_HEADERS:
        if (proto == DOCA_FLOW_L4_TYPE_EXT_TCP) {
            if (!is_all_zeros(&smask->tcp, sizeof smask->tcp)) {
                return;
            }
        } else if (proto == DOCA_FLOW_L4_TYPE_EXT_UDP) {
            if (!is_all_zeros(&smask->udp, sizeof smask->udp)) {
                return;
            }
        } else if (proto == DOCA_FLOW_L4_TYPE_EXT_ICMP ||
                   proto == DOCA_FLOW_L4_TYPE_EXT_ICMP6) {
            if (!is_all_zeros(&smask->icmp, sizeof smask->icmp)) {
                return;
            }
        }
        break;
    }
    memset(sspec_proto, 0, proto_size);
    memset(smask_proto, 0, proto_size);
}

static void
split_match_header_fields(struct doca_flow_match *spec_,
                          struct doca_flow_match *mask_,
                          struct doca_flow_match *pspec_,
                          struct doca_flow_match *pmask_,
                          struct doca_flow_match *sspec_,
                          struct doca_flow_match *smask_,
                          enum split_field_type avoid_type,
                          bool is_inner,
                          uint32_t *num_matches,
                          uint32_t max_matches)
{
    uint32_t *pspec_pm, *pmask_pm;
    uint32_t *sspec_pm, *smask_pm;
    enum split_field_layer layer;
    uint32_t *spec_pm, *mask_pm;
    struct split_field *field;
    bool pm_matched = false;
    char *pspec, *pmask;
    char *sspec, *smask;
    char *spec, *mask;
    int i, num_fields;
    uint32_t proto;

#define MATCH_TO_CHAR(src, dst) \
    dst = (char *) (is_inner ?  &src->inner : &src->outer);
#define PARSER_META_MATCH_FIELD__(var, field) \
    var##_pm = is_inner ? &var##_->parser_meta.inner_##field : &var##_->parser_meta.outer_##field
#define PARSER_META_MATCH_FIELD(field) \
    PARSER_META_MATCH_FIELD__(spec, field); \
    PARSER_META_MATCH_FIELD__(mask, field); \
    PARSER_META_MATCH_FIELD__(pspec, field); \
    PARSER_META_MATCH_FIELD__(pmask, field); \
    PARSER_META_MATCH_FIELD__(sspec, field); \
    PARSER_META_MATCH_FIELD__(smask, field);
#define PROTO_HEADER(field) \
    is_inner ? spec_->inner.field : spec_->outer.field;

    MATCH_TO_CHAR(spec_, spec);
    MATCH_TO_CHAR(mask_, mask);
    MATCH_TO_CHAR(pspec_, pspec);
    MATCH_TO_CHAR(pmask_, pmask);
    MATCH_TO_CHAR(sspec_, sspec);
    MATCH_TO_CHAR(smask_, smask);

    for (layer = L2_HEADERS;
         layer <= L4_HEADERS && *num_matches < max_matches; layer++) {
        switch (layer) {
        case L2_HEADERS:
            proto = PROTO_HEADER(l2_valid_headers);
            num_fields = NUM_L2_FIELDS;
            PARSER_META_MATCH_FIELD(l2_type);
            break;
        case L3_HEADERS:
            proto = PROTO_HEADER(l3_type);
            num_fields = NUM_L3_FIELDS;
            PARSER_META_MATCH_FIELD(l3_type);
            break;
        case L4_HEADERS:
            proto = PROTO_HEADER(l4_type_ext);
            num_fields = NUM_L4_FIELDS;
            PARSER_META_MATCH_FIELD(l4_type);
            break;
        }

        pm_matched = false;
        for (i = 0; i < num_fields && *num_matches < max_matches; i++) {
            field = &split_fields[layer][i];
            if (field->type == avoid_type ||
                field->proto_type != proto ||
                is_all_zeros(mask + field->offset, field->size)) {
                continue;
            }
            /* Ensure that parser_meta has not been matched previously, that a match on parser_meta
             * is requested, and that there are enough bytes remaining to match parser_meta header.
             */
            if (!pm_matched && *mask_pm &&
                *num_matches + 1 <= max_matches) {
                /* Move the parser_meta matches to the prefix. */
                memcpy(pspec_pm, spec_pm, sizeof *pspec_pm);
                memcpy(pmask_pm, mask_pm, sizeof *pmask_pm);
                memset(sspec_pm, 0, sizeof *sspec_pm);
                memset(smask_pm, 0, sizeof *smask_pm);
                *num_matches += 1;
                /* Set a flag to not add another match on parser_meta in the next iteration,
                 * otherwise for each matched L2/L3/L4 split field we will
                 * wrongly increase match_bytes.
                 */
                pm_matched = true;
            }
            if (*num_matches + 1 <= max_matches) {
                memcpy(pspec + field->offset, spec + field->offset, field->size);
                memcpy(pmask + field->offset, mask + field->offset, field->size);
                memset(sspec + field->offset, 0, field->size);
                memset(smask + field->offset, 0, field->size);
                *num_matches += 1;
                memcpy(pspec + field->proto_offset, &field->proto_type, field->proto_size);
                memcpy(pmask + field->proto_offset, &field->proto_type, field->proto_size);
            }
        }
        field = &split_fields[layer][0];
        reset_split_header_proto(layer, proto, is_inner ? &smask_->inner : &smask_->outer,
                                 sspec + field->proto_offset, smask + field->proto_offset,
                                 field->proto_size);
    }
}

static int
split_tunnel_header_fields(struct doca_flow_match *spec,
                           struct doca_flow_match *mask,
                           struct doca_flow_match *pspec,
                           struct doca_flow_match *pmask,
                           struct doca_flow_match *sspec,
                           struct doca_flow_match *smask,
                           uint32_t *num_matches)
{
    pspec->tun.type = spec->tun.type;
    pmask->tun.type = spec->tun.type;
    memset(&sspec->tun, 0, sizeof smask->tun);
    memset(&smask->tun, 0, sizeof smask->tun);

    /* Assume the tunnel match will fit into the prefix header */
    if (spec->tun.type == DOCA_FLOW_TUN_GENEVE) {
        pspec->tun.geneve.vni = spec->tun.geneve.vni;
        pmask->tun.geneve.vni = mask->tun.geneve.vni;
        *num_matches += 1;

        memcpy(pspec->tun.geneve_options, spec->tun.geneve_options,
               sizeof pspec->tun.geneve_options);
        memcpy(pmask->tun.geneve_options, mask->tun.geneve_options,
               sizeof pmask->tun.geneve_options);
        *num_matches += 1;
    } else if (spec->tun.type == DOCA_FLOW_TUN_VXLAN) {
        pspec->tun.vxlan_tun_id = spec->tun.vxlan_tun_id;
        pmask->tun.vxlan_tun_id = mask->tun.vxlan_tun_id;
        *num_matches += 1;
        if (spec->tun.vxlan_type == DOCA_FLOW_TUN_EXT_VXLAN_GBP) {
            pspec->tun.vxlan_type = spec->tun.vxlan_type;
            pmask->tun.vxlan_type = mask->tun.vxlan_type;
            *num_matches += 1;
            pspec->tun.vxlan_gbp_group_policy_id = spec->tun.vxlan_gbp_group_policy_id;
            pmask->tun.vxlan_gbp_group_policy_id = mask->tun.vxlan_gbp_group_policy_id;
            *num_matches += 1;
        }
    } else if (spec->tun.type == DOCA_FLOW_TUN_GRE) {
        pspec->tun.key_present = spec->tun.key_present;
        pmask->tun.key_present = mask->tun.key_present;
        pspec->tun.gre_key = spec->tun.gre_key;
        pmask->tun.gre_key = mask->tun.gre_key;
        *num_matches += 1;
    } else {
        return -1;
    }

    return 0;
}

static int
split_ovs_doca_flow_match(struct netdev *netdev,
                          struct ovs_doca_flow_match *ospec,
                          struct ovs_doca_flow_match *omask,
                          struct ovs_doca_flow_match *opspec,
                          struct ovs_doca_flow_match *opmask,
                          struct ovs_doca_flow_match *osspec,
                          struct ovs_doca_flow_match *osmask,
                          uint8_t depth,
                          uint32_t *num_matches,
                          uint32_t max_matches)
{
    enum split_field_type avoid_type = FIELD_TYPE_INVALID;
    struct doca_flow_match *spec = &ospec->d;
    struct doca_flow_match *mask = &omask->d;
    struct doca_flow_match *pspec = &opspec->d;
    struct doca_flow_match *pmask = &opmask->d;
    struct doca_flow_match *sspec = &osspec->d;
    struct doca_flow_match *smask = &osmask->d;
    int i;

    memset(opspec, 0, sizeof *opspec);
    memset(opmask, 0, sizeof *opmask);
    memcpy(osspec, ospec, sizeof *osspec);
    memcpy(osmask, omask, sizeof *osmask);

    /* Collect meta matches to prefix first and assume we
     * always have enough room in prefix for all meta.
     * Port meta match always exists and takes 4 bytes and
     * we set it for both prefix and suffix to utilize the
     * doca flow direction optimization and reduce number
     * of flows.
     */
    pspec->parser_meta.port_id = spec->parser_meta.port_id;
    pmask->parser_meta.port_id = mask->parser_meta.port_id;
    *num_matches += 1;

    if (mask->meta.pkt_meta) {
        pspec->meta.pkt_meta = spec->meta.pkt_meta;
        pmask->meta.pkt_meta = mask->meta.pkt_meta;
        sspec->meta.pkt_meta = 0;
        smask->meta.pkt_meta = 0;
        *num_matches += 1;
    }

    for (i = 0; i < ARRAY_SIZE(spec->meta.u32); i++) {
        if (mask->meta.u32[i]) {
            pspec->meta.u32[i] = spec->meta.u32[i];
            pmask->meta.u32[i] = mask->meta.u32[i];
            sspec->meta.u32[i] = 0;
            smask->meta.u32[i] = 0;
            *num_matches += 1;
        }
    }

    if (omask->nv_mp.pid) {
        opspec->nv_mp.pid = ospec->nv_mp.pid;
        opmask->nv_mp.pid = omask->nv_mp.pid;
        osspec->nv_mp.pid = 0;
        osmask->nv_mp.pid = 0;
        *num_matches += 1;
    }
    if (omask->nv_mp.preferred) {
        opspec->nv_mp.preferred = ospec->nv_mp.preferred;
        opmask->nv_mp.preferred = omask->nv_mp.preferred;
        osspec->nv_mp.preferred = 0;
        osmask->nv_mp.preferred = 0;
        *num_matches += 1;
    }
    if (omask->nv_mp.strict) {
        opspec->nv_mp.strict = ospec->nv_mp.strict;
        opmask->nv_mp.strict = omask->nv_mp.strict;
        osspec->nv_mp.strict = 0;
        osmask->nv_mp.strict = 0;
        *num_matches += 1;
    }

    if (spec->tun.type != DOCA_FLOW_TUN_NONE) {
        if (split_tunnel_header_fields(spec, mask, pspec, pmask,
                                       sspec, smask, num_matches)) {
            return -1;
        }

        /* Tunnel outer header is common among flows so collect
         * them into the prefix flow.
         */
        split_match_header_fields(spec, mask, pspec, pmask, sspec, smask,
                                  FIELD_TYPE_INVALID, false, num_matches, max_matches);
        /* Ensure that the src IP is set as part of the suffix to
         * achive better hash distributation in Hardware.
         */
        if (spec->outer.l3_type == DOCA_FLOW_L3_TYPE_IP4) {
            sspec->outer.l3_type = DOCA_FLOW_L3_TYPE_IP4;
            smask->outer.l3_type = DOCA_FLOW_L3_TYPE_IP4;
            sspec->outer.ip4.src_ip = spec->outer.ip4.src_ip;
            smask->outer.ip4.src_ip = mask->outer.ip4.src_ip;
            pspec->outer.ip4.src_ip = 0;
            pmask->outer.ip4.src_ip = 0;
        } else if (spec->outer.l3_type == DOCA_FLOW_L3_TYPE_IP6) {
            sspec->outer.l3_type = DOCA_FLOW_L3_TYPE_IP6;
            smask->outer.l3_type = DOCA_FLOW_L3_TYPE_IP6;
            memcpy(sspec->outer.ip6.src_ip, spec->outer.ip6.src_ip, sizeof sspec->outer.ip6.src_ip);
            memset(pspec->outer.ip6.src_ip, 0, sizeof pspec->outer.ip6.src_ip);
            memset(pmask->outer.ip6.src_ip, 0, sizeof pmask->outer.ip6.src_ip);
            memcpy(smask->outer.ip6.src_ip, mask->outer.ip6.src_ip, sizeof mask->outer.ip6.src_ip);
        }
    } else {
        /* In case there's no tunnel and it is the the first split, collect fields to prefix based
         * on the flow's direction. avoid_type will indicate the flow is RX or TX, in case of RX the
         * common part is the source addresses, collect them into the split flow, otherwise consider
         * the destination.
         */
        if (depth == 0) {
            avoid_type = netdev_doca_get_port_dir(netdev) == NETDEV_DOCA_PORT_DIR_RX ?
                FIELD_TYPE_SRC : FIELD_TYPE_DST;
        }
        if (is_all_zeros(&mask->outer, sizeof mask->outer)) {
            if (!is_all_zeros(&pmask->inner.ip4, sizeof pmask->inner.ip4) ||
                !is_all_zeros(&pmask->inner.ip6, sizeof pmask->inner.ip6)) {
                pspec->inner.l3_type = spec->inner.l3_type;
                pmask->inner.l3_type = mask->inner.l3_type;
            }
            if (!is_all_zeros(&pmask->inner.icmp, sizeof pmask->inner.icmp) ||
                !is_all_zeros(&pmask->inner.tcp, sizeof pmask->inner.tcp) ||
                !is_all_zeros(&pmask->inner.udp, sizeof pmask->inner.udp) ||
                !is_all_zeros(&pmask->inner.transport, sizeof pmask->inner.transport)) {
                pspec->inner.l4_type_ext = spec->inner.l4_type_ext;
                pmask->inner.l4_type_ext = mask->inner.l4_type_ext;
            }
            split_match_header_fields(spec, mask, pspec, pmask, sspec, smask,
                                      avoid_type, true, num_matches, max_matches);
        } else {
            split_match_header_fields(spec, mask, pspec, pmask, sspec, smask,
                                      avoid_type, false, num_matches, max_matches);
        }
    }

    return 0;
}

static struct doca_flow_pipe_entry *
create_split_doca_flow_entry(struct netdev *netdev,
                             unsigned int queue_id,
                             uint32_t prio,
                             struct doca_pipe_group_ctx *group_ctx,
                             struct ovs_doca_flow_match *spec,
                             struct ovs_doca_flow_match *mask,
                             struct ovs_doca_flow_actions *actions,
                             struct ovs_doca_flow_actions *actions_masks,
                             struct doca_flow_action_descs *dacts_descs,
                             struct doca_flow_monitor *monitor,
                             struct doca_flow_fwd *fwd,
                             struct doca_flow_handle_resources *flow_res,
                             uint8_t depth,
                             struct doca_split_prefix_ctx *prev_pctx,
                             struct rte_flow_error *error)
{
    struct doca_pipe_group_mask_ctx *prefix_mask_ctx = NULL;
    uint32_t num_matches = 0, max_matches = UINT32_MAX;
    struct ovs_doca_flow_actions dacts, dacts_mask;
    struct dpdk_offload_handle suffix_entry_doh;
    struct doca_pipe_group_ctx *suffix_group;
    struct doca_split_prefix_key prefix_key;
    struct ovs_doca_flow_match prefix_spec;
    struct ovs_doca_flow_match prefix_mask;
    struct ovs_doca_flow_match suffix_spec;
    struct ovs_doca_flow_match suffix_mask;
    struct doca_split_prefix_ctx *pctx;
    struct doca_flow_pipe_entry *entry;
    struct doca_split_prefix_arg args;
    struct doca_flow_fwd pfwd;
    uint32_t pprio = prio;
    uint32_t prefix_id;

    if (unlikely(depth == MAX_SPLIT_DEPTH)) {
        error->type = RTE_FLOW_ERROR_TYPE_UNSPECIFIED;
        error->message = "Exceeded max split depth";
        VLOG_DBG_RL(&rl, "Exceeded max split depth %d", MAX_SPLIT_DEPTH);
        return NULL;
    }

    memset(&prefix_spec, 0, sizeof prefix_spec);
    memset(&prefix_mask, 0, sizeof prefix_mask);
    memset(&suffix_spec, 0, sizeof suffix_spec);
    memset(&suffix_mask, 0, sizeof suffix_mask);
    /* Split the original match to prefix and suffix keys */
    if (split_ovs_doca_flow_match(netdev, spec, mask, &prefix_spec, &prefix_mask,
                                  &suffix_spec, &suffix_mask, depth, &num_matches, max_matches)) {
        error->type = RTE_FLOW_ERROR_TYPE_UNSPECIFIED;
        error->message = "Could not split flow matches";
        return NULL;
    }
    max_matches = num_matches;

    memset(&dacts, 0, sizeof dacts);
    memset(&dacts_mask, 0, sizeof dacts_mask);
    memset(&pfwd, 0, sizeof pfwd);
    pfwd.type = DOCA_FLOW_FWD_PIPE;
    doca_set_reg_mask(&dacts_mask.d.meta, REG_FIELD_SCRATCH);
    if (depth == 0 && prefix_spec.d.tun.type != DOCA_FLOW_TUN_NONE) {
        /* First prefix of a tunnel. Set the priority by the prefix properties and not by
         * the prio argument, as it reflects the inner priority.
         */
        switch (prefix_spec.d.outer.l3_type) {
        case DOCA_FLOW_L3_TYPE_NONE:
            break;
        case DOCA_FLOW_L3_TYPE_IP4:
            pprio = DPDK_OFFLOAD_PRIORITY_HIGH;
            break;
        case DOCA_FLOW_L3_TYPE_IP6:
            pprio = DPDK_OFFLOAD_PRIORITY_MED;
            break;
        }
    }

    /* Demote the priority of the prefix. */
    if (pprio == DPDK_OFFLOAD_PRIORITY_HIGH) {
        pprio = DPDK_OFFLOAD_PRIORITY_SPLIT_HIGH;
    } else if (pprio == DPDK_OFFLOAD_PRIORITY_MED) {
        pprio = DPDK_OFFLOAD_PRIORITY_SPLIT_MED;
    }

    do {
        prefix_mask_ctx =
            doca_pipe_group_mask_ctx_ref(netdev, group_ctx, &prefix_spec, &prefix_mask,
                                         &dacts, &dacts_mask, NULL, NULL, &pfwd,
                                         pprio, NULL, UINT64_C(1) << queue_id);
        if (prefix_mask_ctx) {
            break;
        }

        memset(&prefix_spec, 0, sizeof prefix_spec);
        memset(&prefix_mask, 0, sizeof prefix_mask);
        memset(&suffix_spec, 0, sizeof suffix_spec);
        memset(&suffix_mask, 0, sizeof suffix_mask);
        /* Reduce the allowed matches by one and attempt to split again. */
        max_matches--;
        num_matches = 0;
        if (split_ovs_doca_flow_match(netdev, spec, mask, &prefix_spec, &prefix_mask,
                                      &suffix_spec, &suffix_mask, depth, &num_matches,
                                      max_matches)) {
            error->type = RTE_FLOW_ERROR_TYPE_UNSPECIFIED;
            error->message = "Could not split flow matches";
            return NULL;
        }
    } while (max_matches > 1);

    if (!prefix_mask_ctx) {
        error->type = RTE_FLOW_ERROR_TYPE_UNSPECIFIED;
        error->message = "Could not get prefix mask_ctx for split flow";
        return NULL;
    }

    suffix_group = doca_pipe_group_ctx_ref(netdev, SPLIT_POSTPREFIX_TABLE_ID, prefix_mask_ctx);
    if (!suffix_group) {
        error->type = RTE_FLOW_ERROR_TYPE_UNSPECIFIED;
        error->message = "Could not get suffix group for split flow";
        goto err_suffix_group;
    }
    flow_res->self_group_ctx = suffix_group;

    memset(&prefix_key, 0, sizeof prefix_key);
    args.netdev = netdev;
    args.prefix_is_group = true;
    args.suffix_pipe = NULL;

    prefix_key.prefix_mask_ctx = prefix_mask_ctx;
    memcpy(&prefix_key.spec, &prefix_spec, sizeof prefix_key.spec);
    pctx = split_prefix_ctx_ref(&prefix_key, &args, &prefix_id);
    if (!pctx) {
        error->type = RTE_FLOW_ERROR_TYPE_UNSPECIFIED;
        error->message = "Could not get split flow prefix id";
        goto err_split_prefix_ctx;
    }
    flow_res->split_prefix_ctx = pctx;
    if (depth > 0) {
        /* If this is not the first split attempt, suffix_spec and mask are
         * copied from the previous split suffix which included a match on the
         * previous prefix_id, mask that match before setting a match on
         * the current prefix_id
         */
        memset(&suffix_spec.d.meta, 0, sizeof suffix_spec.d.meta);
    }

    /* Add prefix ID to suffix match. */
    doca_set_reg_val_mask(&suffix_spec.d.meta, &suffix_mask.d.meta,
                          REG_FIELD_SCRATCH, prefix_id);
    /* Create suffix entry. */
    entry = create_doca_pipe_group_entry(netdev, queue_id, suffix_group, prio,
                                         &suffix_spec, &suffix_mask, actions,
                                         actions_masks, dacts_descs, monitor,
                                         fwd, &flow_res->mask_ctx, error);
    if (!entry) {
        if (depth != MAX_SPLIT_DEPTH &&
            (doca_error_t) error->type == DOCA_ERROR_TOO_BIG) {
            VLOG_DBG_RL(&rl, "%s: Split attempt %d flow entry is too big to "
                        "insert directly. Attempt another flow split.",
                        netdev_get_name(netdev), depth);
            entry = create_split_doca_flow_entry(netdev, queue_id, prio,
                                                 suffix_group, &suffix_spec,
                                                 &suffix_mask, actions,
                                                 actions_masks, dacts_descs,
                                                 monitor, fwd, flow_res,
                                                 depth + 1, pctx, error);
        }
        if (!entry) {
            error->type = RTE_FLOW_ERROR_TYPE_UNSPECIFIED;
            error->message = "Could not create split flow";
            goto err_suffix_entry;
        }
    }

    /* Create prefix entry. */
    ovs_mutex_lock(&pctx->handle_lock);
    if (!pctx->doh.dfh.flow) {
        struct doca_flow_handle *phndl = &pctx->doh.dfh;

        phndl->flow_res.split_prefix_ctx = prev_pctx;
        doca_set_reg_val_mask(&dacts.d.meta, &dacts_mask.d.meta, REG_FIELD_SCRATCH, prefix_id);
        pfwd.next_pipe = doca_pipe_group_get_pipe(suffix_group);
        phndl->flow = create_doca_pipe_group_entry(netdev, queue_id,
                                                   group_ctx, pprio, &prefix_spec, &prefix_mask,
                                                   &dacts, &dacts_mask, NULL, NULL, &pfwd,
                                                   &phndl->flow_res.mask_ctx, error);
        ovs_assert(phndl->flow_res.mask_ctx == prefix_mask_ctx);
        if (!phndl->flow) {
            ovs_mutex_unlock(&pctx->handle_lock);
            goto err_prefix_entry;
        }
        doca_pipe_group_set_suffix_group(prefix_mask_ctx, suffix_group);
        phndl->flow_res.self_group_ctx = group_ctx;
        pctx->queue_id = queue_id;
    }
    ovs_mutex_unlock(&pctx->handle_lock);
    doca_pipe_group_mask_ctx_unref(prefix_mask_ctx);

    error->type = (enum rte_flow_error_type) DOCA_SUCCESS;
    return entry;

err_prefix_entry:
    memset(&suffix_entry_doh, 0, sizeof suffix_entry_doh);
    suffix_entry_doh.dfh.flow = entry;
    memcpy(&suffix_entry_doh.dfh.flow_res, flow_res, sizeof *flow_res);
    destroy_dpdk_offload_handle__(doca_offload_esw_ctx_get(netdev), &suffix_entry_doh, queue_id,
                                  error);
err_suffix_entry:
    split_prefix_ctx_unref(pctx);
err_split_prefix_ctx:
    doca_pipe_group_ctx_unref(suffix_group);
err_suffix_group:
    doca_pipe_group_mask_ctx_unref(prefix_mask_ctx);

    return NULL;
}

static struct doca_flow_handle *
create_doca_flow_handle(struct netdev *netdev,
                        unsigned int queue_id,
                        uint32_t prio,
                        uint32_t group,
                        struct ovs_doca_flow_match *spec,
                        struct ovs_doca_flow_match *mask,
                        struct ovs_doca_flow_actions *actions,
                        struct ovs_doca_flow_actions *actions_masks,
                        struct doca_flow_action_descs *dacts_descs,
                        struct doca_flow_monitor *monitor,
                        struct doca_flow_fwd *fwd,
                        struct doca_flow_handle_resources *flow_res,
                        struct dpdk_offload_handle *doh,
                        struct rte_flow_error *error)
{
    struct doca_pipe_group_ctx *group_ctx = NULL;
    struct doca_flow_handle *hndl;
    bool is_split = false;

    hndl = &doh->dfh;

    /* get self table pointer */
    group_ctx = doca_pipe_group_ctx_ref(netdev, group, NULL);
    if (!group_ctx) {
        error->type = RTE_FLOW_ERROR_TYPE_UNSPECIFIED;
        error->message = "pipe error: could not create group";
        goto err_pipe;
    }
    /* insert rule */
    if (!spec || spec->d.tun.type == DOCA_FLOW_TUN_NONE) {
        hndl->flow = create_doca_pipe_group_entry(netdev, queue_id, group_ctx,
                                                  prio, spec, mask, actions,
                                                  actions_masks, dacts_descs,
                                                  monitor, fwd, &flow_res->mask_ctx,
                                                  error);
    } else {
        hndl->flow = NULL;
        error->type = (enum rte_flow_error_type) DOCA_ERROR_TOO_BIG;
    }
    if (!hndl->flow && spec && mask && (doca_error_t) error->type == DOCA_ERROR_TOO_BIG) {
        VLOG_DBG_RL(&rl, "%s: Flow entry is too big to insert directly. "
                    "Attempt flow split.",
                    netdev_get_name(netdev));
        is_split = true;
        hndl->flow = create_split_doca_flow_entry(netdev, queue_id, prio,
                                                  group_ctx, spec, mask,
                                                  actions, actions_masks,
                                                  dacts_descs, monitor,
                                                  fwd, flow_res, 0, NULL, error);
    }

    if (!hndl->flow) {
        goto err_insert;
    }

    memcpy(&hndl->flow_res, flow_res, sizeof *flow_res);
    if (!is_split) {
        hndl->flow_res.self_group_ctx = group_ctx;
    }

    return hndl;

err_insert:
    doca_pipe_group_ctx_unref(group_ctx);
err_pipe:
    return NULL;
}

static struct doca_flow_handle *
create_doca_flow_handle_control(struct netdev *netdev,
                                unsigned int queue_id,
                                uint32_t prio,
                                struct doca_flow_pipe *pipe,
                                struct ovs_doca_flow_match *spec,
                                struct ovs_doca_flow_match *mask,
                                struct ovs_doca_flow_actions *actions,
                                struct ovs_doca_flow_actions *actions_masks,
                                struct doca_flow_monitor *monitor,
                                struct doca_flow_fwd *fwd,
                                struct doca_flow_handle_resources *flow_res,
                                struct dpdk_offload_handle *doh,
                                struct rte_flow_error *error)
{
    struct doca_offload_esw_ctx *esw;
    struct doca_flow_handle *hndl;
    doca_error_t err;

    hndl = &doh->dfh;
    esw = doca_offload_esw_ctx_get(netdev);
    err = doca_offload_add_entry_control(queue_id, prio, pipe, spec, mask, actions,
                                         actions_masks, monitor, fwd, esw, &hndl->flow);
    if (err) {
        goto err_add;
    }

    dpdk_offload_counter_inc(netdev);
    memcpy(&hndl->flow_res, flow_res, sizeof *flow_res);
    return hndl;

err_add:
    error->type = RTE_FLOW_ERROR_TYPE_UNSPECIFIED;
    error->message = "pipe error: cannot add control pipe entry";
    return NULL;
}

static struct doca_flow_pipe_entry *
add_doca_post_meter_red_entry(struct netdev *netdev,
                              unsigned int queue_id,
                              uint32_t meter_counter_id,
                              uint32_t flow_id,
                              struct doca_pipe_group_mask_ctx **mask_ctx,
                              struct rte_flow_error *error)
{
    struct doca_pipe_group_ctx *post_meter_pipe_group_ctx;
    struct doca_offload_esw_ctx *esw_ctx;
    struct ovs_doca_flow_match red_match;
    struct ovs_doca_flow_match red_mask;
    struct doca_flow_pipe_entry *entry;
    struct doca_flow_monitor monitor;
    struct doca_flow_fwd fwd;

    esw_ctx = doca_offload_esw_ctx_get(netdev);
    post_meter_pipe_group_ctx = esw_ctx->post_meter_pipe_group_ctx;

    memset(&red_match, 0, sizeof(red_match));
    memset(&red_mask, 0, sizeof(red_mask));
    memset(&monitor, 0, sizeof(monitor));
    memset(&fwd, 0, sizeof fwd);

    fwd.type = DOCA_FLOW_FWD_DROP;
    monitor.counter_type = DOCA_FLOW_RESOURCE_TYPE_SHARED;
    monitor.shared_counter.shared_counter_id = meter_counter_id;

    doca_set_reg_val_mask(&red_match.d.meta, &red_mask.d.meta,
                          REG_FIELD_FLOW_INFO, flow_id);

    /* Insert red rule with low priority, lower than the corresponding
     * green rule, optimizing for the case when traffic stays below metered
     * rate.
     */
    entry = create_doca_pipe_group_entry(netdev, queue_id, post_meter_pipe_group_ctx,
                                         DPDK_OFFLOAD_PRIORITY_MED, &red_match,
                                         &red_mask, NULL, NULL, NULL, &monitor,
                                         &fwd, mask_ctx, error);
    if (!entry) {
        VLOG_ERR_RL(&rl, "%s: Failed to create shared meter red rule for flow ID %u",
                    netdev_get_name(netdev), flow_id);
        return NULL;
    }

    return entry;
}

static struct doca_flow_pipe_entry *
add_doca_post_meter_green_entry(struct netdev *netdev,
                                unsigned int queue_id,
                                uint32_t meter_counter_id,
                                uint32_t flow_id,
                                const struct doca_act_vars *dact_vars,
                                const struct meter_info *next_meter,
                                struct doca_flow_fwd *orig_fwd,
                                struct doca_pipe_group_mask_ctx **mask_ctx,
                                struct rte_flow_error *error)
{
    struct doca_pipe_group_ctx *post_meter_pipe_group_ctx;
    struct ovs_doca_flow_actions dacts, dacts_masks;
    struct ovs_doca_flow_match green_match;
    struct ovs_doca_flow_match green_mask;
    struct doca_offload_esw_ctx *esw_ctx;
    struct doca_flow_fwd *fwd = orig_fwd;
    struct doca_flow_pipe_entry *entry;
    struct doca_flow_fwd next_mtr_fwd;
    uint32_t fid = dact_vars->flow_id;
    struct doca_flow_monitor monitor;

    esw_ctx = doca_offload_esw_ctx_get(netdev);
    post_meter_pipe_group_ctx = esw_ctx->post_meter_pipe_group_ctx;

    memset(&next_mtr_fwd, 0, sizeof next_mtr_fwd);
    memset(&green_match, 0, sizeof green_match);
    memset(&dacts_masks, 0, sizeof dacts_masks);
    memset(&green_mask, 0, sizeof green_mask);
    memset(&monitor, 0, sizeof monitor);
    memset(&dacts, 0, sizeof dacts);

    monitor.counter_type = DOCA_FLOW_RESOURCE_TYPE_SHARED;
    monitor.shared_counter.shared_counter_id = meter_counter_id;

    doca_set_reg_val_mask(&green_match.d.meta, &green_mask.d.meta,
                          REG_FIELD_FLOW_INFO, flow_id);
    green_match.d.parser_meta.meter_color = DOCA_FLOW_METER_COLOR_GREEN;
    green_mask.d.parser_meta.meter_color = 0xff;

    if (next_meter) {
        monitor.meter_type = DOCA_FLOW_RESOURCE_TYPE_SHARED;
        monitor.shared_meter.shared_meter_id = next_meter->id;
        fid = next_meter->flow_id;
        next_mtr_fwd.type = DOCA_FLOW_FWD_PIPE;
        next_mtr_fwd.next_pipe = doca_pipe_group_get_pipe(post_meter_pipe_group_ctx);
        fwd = &next_mtr_fwd;
    } else if (dact_vars->hash_data) {
        doca_set_reg_val_mask(&dacts.d.meta, &dacts_masks.d.meta, REG_FIELD_SCRATCH,
                              dact_vars->hash_data->seed);
    }

    doca_set_reg_val_mask(&dacts.d.meta, &dacts_masks.d.meta,
                          REG_FIELD_FLOW_INFO, fid);

    /* Insert green rule with high prio, which is higher than the corresponding
     * red rule, optimizing for the case when traffic stays below metered rate.
     */
    entry = create_doca_pipe_group_entry(netdev, queue_id, post_meter_pipe_group_ctx,
                                         DPDK_OFFLOAD_PRIORITY_HIGH, &green_match,
                                         &green_mask, &dacts, &dacts_masks, NULL,
                                         &monitor, fwd, mask_ctx, error);
    if (!entry) {
        VLOG_ERR_RL(&rl, "%s: Failed to create shared meter green rule for flow ID %u",
                    netdev_get_name(netdev), flow_id);
        return NULL;
    }

    if (!next_meter) {
        /* replace original fwd with the internal meter pipe */
        memset(orig_fwd, 0, sizeof *orig_fwd);
        orig_fwd->type = DOCA_FLOW_FWD_PIPE;
        orig_fwd->next_pipe = doca_pipe_group_get_pipe(post_meter_pipe_group_ctx);
    }

    return entry;
}

static int
destroy_meter_hierarchy(struct netdev *netdev, unsigned int queue_id,
                        struct doca_flow_handle_resources *flow_res,
                        struct rte_flow_error *error)
{
    struct doca_offload_esw_ctx *esw_ctx = doca_offload_esw_ctx_get(netdev);
    struct doca_meter_ctx *mtr_ctx;
    int err;

    if (!flow_res->meters_ctx) {
        return 0;
    }

    LIST_FOR_EACH_SAFE (mtr_ctx, list_node, flow_res->meters_ctx) {
        /* RED entry */
        if (mtr_ctx->post_meter_red_entry_mctx.entry) {
            err = doca_offload_remove_entry(esw_ctx, queue_id, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                            &mtr_ctx->post_meter_red_entry_mctx.entry);
            if (err) {
                if (error) {
                    error->type = RTE_FLOW_ERROR_TYPE_HANDLE;
                    error->message = doca_error_get_descr(err);
                }
                return -1;
            }
            doca_pipe_group_mask_ctx_unref(mtr_ctx->post_meter_red_entry_mctx.mctx);
            dpdk_offload_counter_dec(netdev);
        }

        /* GREEN entry */
        if (mtr_ctx->post_meter_green_entry_mctx.entry) {
            err = doca_offload_remove_entry(esw_ctx, queue_id, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                            &mtr_ctx->post_meter_green_entry_mctx.entry);
            if (err) {
                if (error) {
                    error->type = RTE_FLOW_ERROR_TYPE_HANDLE;
                    error->message = doca_error_get_descr(err);
                }
                return -1;
            }
            doca_pipe_group_mask_ctx_unref(mtr_ctx->post_meter_green_entry_mctx.mctx);
            dpdk_offload_counter_dec(netdev);
        }

        /* Meter internal flow ID */
        if (mtr_ctx->post_meter_flow_id) {
            shared_mtr_flow_id_free(esw_ctx, mtr_ctx->post_meter_flow_id);
        }

        doca_put_shared_meter(mtr_ctx->of_meter_id, &esw_ctx->shared_meters_ctx);
        ovs_list_remove(&mtr_ctx->list_node);
        free(mtr_ctx);
    }

    free(flow_res->meters_ctx);
    flow_res->meters_ctx = NULL;

    return 0;
}

static void
destroy_post_hash_entry(struct netdev *netdev, unsigned int queue_id,
                        struct doca_flow_handle_resources *flow_res)
{
    struct doca_offload_esw_ctx *esw;

    if (!flow_res->post_hash_entry_mctx.entry) {
        return;
    }

    esw = doca_offload_esw_ctx_get(netdev);
    doca_offload_remove_entry(esw, queue_id, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                              &flow_res->post_hash_entry_mctx.entry);
    doca_pipe_group_mask_ctx_unref(flow_res->post_hash_entry_mctx.mctx);
    dpdk_offload_counter_dec(netdev);
    flow_res->post_hash_entry_mctx.mctx = NULL;
}

static struct doca_flow_pipe_entry *
create_post_hash_entry(struct netdev *netdev, unsigned int queue_id,
                       uint32_t flow_id,
                       struct doca_pipe_group_ctx *next_group_ctx,
                       struct doca_pipe_group_mask_ctx **mask_ctx,
                       struct rte_flow_error *error)
{
    struct doca_offload_esw_ctx *esw_ctx = doca_offload_esw_ctx_get(netdev);
    struct doca_pipe_group_ctx *post_hash_group_ctx;
    struct doca_flow_pipe_entry *entry;
    struct ovs_doca_flow_match spec;
    struct ovs_doca_flow_match mask;
    struct doca_flow_fwd fwd;

    memset(&spec, 0, sizeof spec);
    memset(&mask, 0, sizeof mask);
    memset(&fwd, 0, sizeof fwd);

    post_hash_group_ctx = esw_ctx->hash_pipe_ctx->post_hash_group_ctx;

    doca_set_reg_val_mask(&spec.d.meta, &mask.d.meta, REG_FIELD_FLOW_INFO, flow_id);
    fwd.type = DOCA_FLOW_FWD_PIPE;
    fwd.next_pipe = doca_pipe_group_get_pipe(next_group_ctx);

    entry = create_doca_pipe_group_entry(netdev, queue_id, post_hash_group_ctx, 0,
                                         &spec, &mask, NULL, NULL, NULL, NULL,
                                         &fwd, mask_ctx, error);
    return entry;
}

static int
create_meter_hierarchy(struct netdev *netdev, unsigned int queue_id,
                       struct doca_act_vars *dact_vars,
                       struct doca_flow_handle_resources *flow_res,
                       struct doca_flow_fwd *fwd, struct rte_flow_error *error)
{
    struct doca_offload_esw_ctx *esw = doca_offload_esw_ctx_get(netdev);
    struct meter_info *next_mtr = NULL;
    struct doca_meter_ctx *mtr_ctx;
    struct meter_info *cur_mtr;

    if (!flow_res->meters_ctx) {
        flow_res->meters_ctx = xzalloc(sizeof *flow_res->meters_ctx);
        ovs_list_init(flow_res->meters_ctx);
    }

    /* Meter entries are added in reverse order to have the full hierarchy
     * already in place at time when the first packet arrives. The entries are
     * "chained" by the means of mark and match on meter_info->flow_id.
     */
    LIST_FOR_EACH_REVERSE (cur_mtr, list_node, &dact_vars->next_meters) {
        mtr_ctx = xzalloc(sizeof *mtr_ctx);

        /* RED entry */
        mtr_ctx->post_meter_red_entry_mctx.entry =
            add_doca_post_meter_red_entry(netdev, queue_id, cur_mtr->red_id,
                                          cur_mtr->flow_id,
                                          &mtr_ctx->post_meter_red_entry_mctx.mctx, error);
        if (!mtr_ctx->post_meter_red_entry_mctx.entry) {
            if (error) {
                error->type = RTE_FLOW_ERROR_TYPE_ACTION;
                error->message = "Could not create red post multi-meter rule";
            }
            goto err;
        }

        /* GREEN entry */
        mtr_ctx->post_meter_green_entry_mctx.entry =
            add_doca_post_meter_green_entry(netdev, queue_id, cur_mtr->green_id,
                                            cur_mtr->flow_id, dact_vars,
                                            next_mtr, fwd,
                                            &mtr_ctx->post_meter_green_entry_mctx.mctx,
                                            error);
        if (!mtr_ctx->post_meter_green_entry_mctx.entry) {
            if (error) {
                error->type = RTE_FLOW_ERROR_TYPE_ACTION;
                error->message = "Could not create green post multi-meter rule";
            }
            doca_offload_remove_entry(esw, queue_id, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                      &mtr_ctx->post_meter_red_entry_mctx.entry);
            doca_pipe_group_mask_ctx_unref(mtr_ctx->post_meter_red_entry_mctx.mctx);
            goto err;
        }

        mtr_ctx->of_meter_id = cur_mtr->of_id;
        mtr_ctx->post_meter_flow_id = cur_mtr->flow_id;
        ovs_list_push_front(flow_res->meters_ctx, &mtr_ctx->list_node);
        next_mtr = cur_mtr;
    }

    mtr_ctx = xzalloc(sizeof *mtr_ctx);

    /* First RED entry */
    mtr_ctx->post_meter_red_entry_mctx.entry =
        add_doca_post_meter_red_entry(netdev, queue_id, dact_vars->mtr_red_cnt_id,
                                      dact_vars->mtr_flow_id,
                                      &mtr_ctx->post_meter_red_entry_mctx.mctx,
                                      error);
    if (!mtr_ctx->post_meter_red_entry_mctx.entry) {
        if (error) {
            error->type = RTE_FLOW_ERROR_TYPE_ACTION;
            error->message = "Could not create red post meter rule";
        }
        goto err;
    }

    /* First GREEN entry */
    mtr_ctx->post_meter_green_entry_mctx.entry =
        add_doca_post_meter_green_entry(netdev, queue_id, dact_vars->mtr_green_cnt_id,
                                        dact_vars->mtr_flow_id, dact_vars,
                                        next_mtr, fwd,
                                        &mtr_ctx->post_meter_green_entry_mctx.mctx,
                                        error);
    if (!mtr_ctx->post_meter_green_entry_mctx.entry) {
        if (error) {
            error->type = RTE_FLOW_ERROR_TYPE_ACTION;
            error->message = "Could not create green post meter rule";
        }
        doca_offload_remove_entry(esw, queue_id, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                  &mtr_ctx->post_meter_red_entry_mctx.entry);
        doca_pipe_group_mask_ctx_unref(mtr_ctx->post_meter_red_entry_mctx.mctx);
        goto err;
    }

    mtr_ctx->of_meter_id = dact_vars->mtr_of_id;
    mtr_ctx->post_meter_flow_id = dact_vars->mtr_flow_id;
    ovs_list_push_front(flow_res->meters_ctx, &mtr_ctx->list_node);

    return 0;

err:
    free(mtr_ctx);
    destroy_meter_hierarchy(netdev, queue_id, flow_res, NULL);
    if (error) {
        ovs_assert(error->message);
        ovs_assert(error->type);
    }
    return -1;
}

/* Random mask HW register is limited to 16 bits. Because it's a mask, it
 * doesn't support arbitrary sampling ratios. Only a fixed number of ratios that
 * correspond to each bit in the mask are supported - starting from the maximum
 * supported ratio of 50%, down to the minimal supported ratio of 0.0015258789%.
 * The percentage value is halved on each step:
 *
 *     50, 25, 12.5, ... , 0.0015258789
 *
 *      1:  50.000000% -> 0x1
 *      2:  25.000000% -> 0x3
 *      3:  12.500000% -> 0x7
 *      4:   6.250000% -> 0xf
 *      5:   3.125000% -> 0x1f
 *      6:   1.562500% -> 0x3f
 *      7:   0.781250% -> 0x7f
 *      8:   0.390625% -> 0xff
 *      9:   0.195312% -> 0x1ff
 *     10:   0.097656% -> 0x3ff
 *     11:   0.048828% -> 0x7ff
 *     12:   0.024414% -> 0xfff
 *     13:   0.012207% -> 0x1fff
 *     14:   0.006104% -> 0x3fff
 *     15:   0.003052% -> 0x7fff
 *     16:   0.001526% -> 0xffff
 *
 * It finds the nearest supported percentage which is not greater than the
 * requested ratio and calculates corresponding mask value.
 */
static doca_be16_t
sample_ratio_to_random_mask(uint32_t ratio)
{
    double next_highest_supported_percentage = 50;
    double percentage = 1.0 / ratio * 100;
    uint16_t mask;
    int i;

    for (i = 1; i <= UINT16_WIDTH; ++i) {
        if (percentage >= next_highest_supported_percentage) {
            break;
        }

        next_highest_supported_percentage /= 2;
    }

    mask = (1 << i) - 1;

    VLOG_INFO("Using random mask 0x%04x for sampling %g%% of traffic"
              " (%g%% requested)", mask, next_highest_supported_percentage,
              percentage);

    return DOCA_HTOBE16(mask);
}

static void
doca_sample_pipe_miss_del_entry__(struct doca_offload_esw_ctx *ctx)
{
    if (!ctx->sample_ctx.sample_pipe_miss_entry_mctx.entry) {
        return;
    }

    doca_offload_remove_entry(ctx, AUX_QUEUE, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                              &ctx->sample_ctx.sample_pipe_miss_entry_mctx.entry);
    doca_pipe_group_mask_ctx_unref(ctx->sample_ctx.sample_pipe_miss_entry_mctx.mctx);
}

static void
doca_sample_pipe_del_entry__(struct doca_offload_esw_ctx *ctx)
{
    if (!ctx->sample_ctx.sample_pipe_entry_mctx.entry) {
        return;
    }

    doca_offload_remove_entry(ctx, AUX_QUEUE, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                              &ctx->sample_ctx.sample_pipe_entry_mctx.entry);
    doca_pipe_group_mask_ctx_unref(ctx->sample_ctx.sample_pipe_entry_mctx.mctx);
    ctx->sample_ctx.ratio = 0;
}

static void
doca_sample_pipe_del_entry(struct netdev *netdev)
{
    doca_sample_pipe_del_entry__(doca_offload_esw_ctx_get(netdev));
    dpdk_offload_counter_dec(netdev);
}

static int
doca_sample_pipe_add_entry(struct netdev *netdev, uint32_t ratio,
                           struct rte_flow_error *error)
{
    struct doca_offload_esw_ctx *ctx = doca_offload_esw_ctx_get(netdev);
    struct doca_pipe_group_mask_ctx **entry_mctx;
    struct doca_flow_monitor monitor = {
        .counter_type = DOCA_FLOW_RESOURCE_TYPE_NON_SHARED,
    };
    struct ovs_doca_flow_actions act_mask;
    struct doca_flow_pipe_entry *entry;
    struct ovs_doca_flow_actions act;
    struct ovs_doca_flow_match spec;
    struct ovs_doca_flow_match mask;
    struct doca_flow_fwd fwd = {
        .type = DOCA_FLOW_FWD_PIPE,
        .next_pipe = ctx->sample_ctx.mirror_pipe_ctx.pipe,
    };

    memset(&act_mask, 0, sizeof act_mask);
    memset(&spec, 0, sizeof spec);
    memset(&mask, 0, sizeof mask);
    memset(&act, 0, sizeof act);

    /* For ratio == 1 the match mask doesn't need to be set, effectively meaning
     * "matchall".
     */
    if (ratio != 1) {
        mask.d.parser_meta.random = sample_ratio_to_random_mask(ratio);
    }
    doca_set_reg_val_mask(&act.d.meta, &act_mask.d.meta, REG_FIELD_FLOW_INFO,
                          NONSAMPLE_FLOW_MARK);
    entry_mctx = &ctx->sample_ctx.sample_pipe_entry_mctx.mctx;

    /* Main sampling entry, it matches with a probability lower than the
     * requested sampling ratio and forwards packets to the sample-mirror pipe.
     * Everything else, that is above the requested sampling ratio probability,
     * is captured by the second lower priority match-all entry and is forwarded
     * back to the root pipe.
     */
    entry = create_doca_pipe_group_entry(netdev, AUX_QUEUE, ctx->sample_ctx.sample_pipe_ctx,
                                         DPDK_OFFLOAD_PRIORITY_HIGH, &spec,
                                         &mask, &act, &act_mask, NULL, &monitor,
                                         &fwd, entry_mctx, error);
    if (!entry) {
            VLOG_ERR_RL(&rl, "%s: Failed to create sample entry.",
                        netdev_get_name(netdev));
            return -1;
    }
    ctx->sample_ctx.sample_pipe_entry_mctx.entry = entry;
    dpdk_offload_counter_inc(netdev);

    /* Skip creating sample pipe miss entry if already exists. */
    if (ctx->sample_ctx.sample_pipe_miss_entry_mctx.entry) {
        goto out;
    }

    entry_mctx = &ctx->sample_ctx.sample_pipe_miss_entry_mctx.mctx;
    fwd.next_pipe = doca_pipe_group_get_pipe(ctx->root_pipe_group_ctx);

    entry = create_doca_pipe_group_entry(netdev, AUX_QUEUE, ctx->sample_ctx.sample_pipe_ctx,
                                         DPDK_OFFLOAD_PRIORITY_LOW, &spec,
                                         &spec, &act, &act_mask, NULL, NULL, &fwd,
                                         entry_mctx, error);
    if (!entry) {
            VLOG_ERR_RL(&rl, "%s: Failed to create sample miss entry.",
                        netdev_get_name(netdev));
            doca_sample_pipe_del_entry(netdev);
            return -1;
    }
    ctx->sample_ctx.sample_pipe_miss_entry_mctx.entry = entry;
    dpdk_offload_counter_inc(netdev);

out:
    ctx->sample_ctx.ratio = ratio;
    doca_offload_complete_queue_esw(ctx, AUX_QUEUE, true);
    return 0;
}

static int
doca_sample_root_add_entry(struct netdev *netdev, struct rte_flow_error *error)
{
    struct doca_offload_esw_ctx *ctx = doca_offload_esw_ctx_get(netdev);
    struct doca_pipe_group_mask_ctx **entry_mctx;
    struct doca_flow_pipe_entry *entry;
    struct ovs_doca_flow_match spec;
    struct ovs_doca_flow_match mask;
    struct doca_flow_fwd fwd = {
        .type = DOCA_FLOW_FWD_PIPE,
        .next_pipe = doca_pipe_group_get_pipe(ctx->sample_ctx.sample_pipe_ctx),
    };

    memset(&spec, 0, sizeof spec);
    memset(&mask, 0, sizeof mask);

    entry_mctx = &ctx->sample_ctx.root_sample_entry_mctx.mctx;
    doca_set_reg_mask(&mask.d.meta, REG_FIELD_FLOW_INFO);

    entry = create_doca_pipe_group_entry(netdev, AUX_QUEUE, ctx->root_pipe_group_ctx,
                                         DPDK_OFFLOAD_PRIORITY_DOCA_SAMPLE, &spec,
                                         &mask, NULL, NULL, NULL, NULL, &fwd,
                                         entry_mctx, error);
    if (!entry) {
            VLOG_ERR_RL(&rl, "%s: Failed to add sampling entry to root pipe: "
                        "%s.", netdev_get_name(netdev), error->message);
            return -1;
    }
    ctx->sample_ctx.root_sample_entry_mctx.entry = entry;
    ovs_refcount_init(&ctx->sample_ctx.root_sample_entry_refcount);
    doca_offload_complete_queue_esw(ctx, AUX_QUEUE, true);
    VLOG_DBG("%s: Added sampling entry to root pipe.", netdev_get_name(netdev));

    return 0;
}

static void
doca_sample_root_del_entry__(struct doca_offload_esw_ctx *ctx)
{
    if (!ctx->sample_ctx.root_sample_entry_mctx.entry) {
        return;
    }

    doca_offload_remove_entry(ctx, AUX_QUEUE, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                              &ctx->sample_ctx.root_sample_entry_mctx.entry);
    doca_pipe_group_mask_ctx_unref(ctx->sample_ctx.root_sample_entry_mctx.mctx);
}

static void doca_sample_root_del_entry(struct netdev *netdev)
{
    doca_sample_root_del_entry__(doca_offload_esw_ctx_get(netdev));
    dpdk_offload_counter_dec(netdev);
    VLOG_DBG("%s: Removed sampling entry from root pipe.",
             netdev_get_name(netdev));
}

static int
dpdk_offload_doca_create__(struct netdev *netdev,
                           const struct rte_flow_attr *attr,
                           struct rte_flow_item *items,
                           struct rte_flow_action *actions,
                           struct dpdk_offload_handle *doh,
                           struct rte_flow_error *error)
{
    struct doca_offload_esw_ctx *esw_ctx = doca_offload_esw_ctx_get(netdev);
    struct ovs_doca_flow_actions dacts, dacts_masks;
    unsigned int tid = netdev_offload_thread_id();
    struct doca_flow_handle_resources flow_res;
    struct doca_flow_action_descs dacts_descs;
    struct doca_flow_action_desc desc_array;
    struct doca_flow_monitor monitor;
    struct ovs_doca_flow_match spec;
    struct ovs_doca_flow_match mask;
    struct doca_act_vars dact_vars;
    struct doca_flow_handle *hndl;
    struct meter_info *next_mtr;
    unsigned int queue_id = tid;
    struct doca_flow_fwd fwd;

    memset(error, 0, sizeof *error);
    memset(&dact_vars, 0, sizeof dact_vars);
    ovs_list_init(&dact_vars.next_meters);

    memset(&dacts_masks, 0, sizeof dacts_masks);
    memset(&dacts_descs, 0, sizeof dacts_descs);
    memset(&desc_array, 0, sizeof desc_array);
    memset(&flow_res, 0, sizeof flow_res);
    memset(&monitor, 0, sizeof monitor);
    memset(&dacts, 0, sizeof dacts);
    memset(&mask, 0, sizeof mask);
    memset(&spec, 0, sizeof spec);
    memset(&fwd, 0, sizeof fwd);

    /* If it's a post ct rule, check for eswitch ct offload support */
    if (attr->group == POSTCT_TABLE_ID &&
        !(ovs_doca_max_ct_counters_per_esw() || conntrack_offload_doca_ct_enabled)) {
        error->type = RTE_FLOW_ERROR_TYPE_UNSPECIFIED;
        error->message = "CT offload disabled on this eswitch";
        goto err;
    }

    if (doca_translate_items(netdev, attr, items, &spec, &mask, &dact_vars, error)) {
        goto err;
    }

    /* parse actions */
    dacts_descs.desc_array = &desc_array;
    dacts_descs.nb_action_desc = 1;
    if (doca_translate_actions(netdev, &spec.d, &mask.d, actions, &dacts, &dacts_masks,
                               &fwd, &monitor, &flow_res, &dact_vars, error)) {
        goto err;
    }

    /* Create post-hash entry */
    if (dact_vars.hash_data && flow_res.next_group_ctx) {
        struct doca_pipe_group_ctx *next_group_ctx = flow_res.next_group_ctx;
        uint32_t flow_id = dact_vars.hash_data->flow_id;

        flow_res.post_hash_entry_mctx.entry =
            create_post_hash_entry(netdev, queue_id, flow_id, next_group_ctx,
                                   &flow_res.post_hash_entry_mctx.mctx, error);
        if (flow_res.post_hash_entry_mctx.entry == NULL) {
            goto err;
        }
    }

    if (monitor.meter_type == DOCA_FLOW_RESOURCE_TYPE_SHARED) {
        if (create_meter_hierarchy(netdev, queue_id, &dact_vars, &flow_res, &fwd, error)) {
            goto err;
        }
    }

    if (dact_vars.sample_ratio) {
        if (!esw_ctx->sample_ctx.sample_pipe_entry_mctx.entry) {
            if (doca_sample_pipe_add_entry(netdev, dact_vars.sample_ratio, error)) {
                goto err;
            }
        }
        if (!esw_ctx->sample_ctx.root_sample_entry_mctx.entry) {
            if (dact_vars.sample_ratio != esw_ctx->sample_ctx.ratio) {
                doca_sample_pipe_del_entry(netdev);
                if (doca_sample_pipe_add_entry(netdev, dact_vars.sample_ratio, error)) {
                    goto err;
                }
            }

            if (doca_sample_root_add_entry(netdev, error)) {
                goto err;
            }
        }
        if (dact_vars.sample_ratio != esw_ctx->sample_ctx.ratio) {
            error->type = RTE_FLOW_ERROR_TYPE_ACTION;
            error->message = "Could not offload sample action";
            VLOG_WARN_RL(&rl, "%s: Cannot add sFlow ratio %u while another "
                         " ratio %u is active", netdev_get_name(netdev),
                         dact_vars.sample_ratio, esw_ctx->sample_ctx.ratio);
            goto err;
        }
    }

    hndl = create_doca_flow_handle(netdev, queue_id, attr->priority, attr->group, &spec, &mask,
                                   &dacts, &dacts_masks, &dacts_descs, &monitor, &fwd, &flow_res,
                                   doh, error);
    if (!hndl) {
        destroy_post_hash_entry(netdev, queue_id, &flow_res);
        doca_pipe_group_ctx_unref(flow_res.next_group_ctx);
        if (monitor.meter_type == DOCA_FLOW_RESOURCE_TYPE_SHARED) {
            destroy_meter_hierarchy(netdev, queue_id, &flow_res, NULL);
        }
        goto err;
    }

    if (dact_vars.sample_ratio) {
        hndl->flow_res.sampled = true;
        ovs_refcount_ref(&esw_ctx->sample_ctx.root_sample_entry_refcount);
    }

    LIST_FOR_EACH_POP (next_mtr, list_node, &dact_vars.next_meters) {
        free(next_mtr);
    }

    return 0;

err:
    doca_mirror_ctx_destroy(flow_res.mirror_ctx);
    LIST_FOR_EACH_POP (next_mtr, list_node, &dact_vars.next_meters) {
        free(next_mtr);
    }
    ovs_assert(error->message);
    ovs_assert(error->type);
    VLOG_DBG_RL(&rl, "%s", error->message);
    doh->rte_flow = NULL;
    return -1;
}

static int
dpdk_offload_doca_create(struct netdev *netdev,
                         const struct rte_flow_attr *attr,
                         struct rte_flow_item *items,
                         struct rte_flow_action *actions,
                         struct dpdk_offload_handle *doh,
                         struct rte_flow_error *error)
{
    if (!doca_offload_esw_ctx_get(netdev)) {
        memset(error, 0, sizeof *error);
        error->type = RTE_FLOW_ERROR_TYPE_UNSPECIFIED;
        error->message = "Port offloads are disabled";
        return EOPNOTSUPP;
    }
    return doca_action_split_create(dpdk_offload_doca_create__, dpdk_offload_doca_destroy__,
                                    netdev, attr, items, actions, doh, error);
}

static int
destroy_dpdk_offload_handle__(struct doca_offload_esw_ctx *esw,
                              struct dpdk_offload_handle *doh,
                              unsigned int queue_id,
                              struct rte_flow_error *error)
{
    doca_error_t err;

    /* Deletion is always synchronous.
     *
     * If async deletion is implemented, aux-table uninit calls deleting
     * entries will use the offload queues in conflict with offload threads
     * polling them during upkeep. It should result in a crash or
     * in a lockup of the queues. */
    err = doca_offload_remove_entry(esw, queue_id, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT, &doh->dfh.flow);
    if (err) {
        if (error) {
            error->type = RTE_FLOW_ERROR_TYPE_HANDLE;
            error->message = doca_error_get_descr(err);
        }
        return -1;
    }
    doca_pipe_group_mask_ctx_unref(doh->dfh.flow_res.mask_ctx);

    return 0;
}

static int
destroy_dpdk_offload_handle(struct doca_offload_esw_ctx *esw,
                            struct netdev *netdev,
                            struct dpdk_offload_handle *doh,
                            unsigned int queue_id,
                            struct rte_flow_error *error)
{
    struct doca_split_prefix_ctx *split_prefixes[MAX_SPLIT_DEPTH];
    struct doca_split_prefix_ctx *split_prefix_ctx;
    uint32_t num_of_prefixes;
    doca_error_t err;

    /* Collect the chain of prefixes that are used. Free them afterwards from most prefix to
     * most suffix order. It cannot be recursive as refmap lock is not recursive.
     */
    num_of_prefixes = 0;
    split_prefix_ctx = doh->dfh.flow_res.split_prefix_ctx;
    while (split_prefix_ctx) {
        split_prefixes[num_of_prefixes++] = split_prefix_ctx;
        ovs_assert(num_of_prefixes < MAX_SPLIT_DEPTH);
        split_prefix_ctx = split_prefix_ctx->doh.dfh.flow_res.split_prefix_ctx;
    }
    for (int i = num_of_prefixes - 1; i >= 0; i--) {
        split_prefix_ctx_unref(split_prefixes[i]);
    }

    if (destroy_dpdk_offload_handle__(esw, doh, queue_id, error)) {
        return -1;
    }

    if (doh->dfh.flow_res.meters_ctx) {
        err = destroy_meter_hierarchy(netdev, queue_id, &doh->dfh.flow_res,
                                      error);
        if (err) {
            return -1;
        }
    }

    destroy_post_hash_entry(netdev, queue_id, &doh->dfh.flow_res);

    if (doh->dfh.flow_res.sampled) {
        struct doca_offload_esw_ctx *esw_ctx = doca_offload_esw_ctx_get(netdev);
        /* root_sample_entry_refcount is initialized to 1 when entry is added.
         * There is also additional ovs_refcount_ref() for every offloaded flow
         * with sample action. So when there is only a single flow, the count is
         * 2. When this flow is destroyed, there will be no other sample action
         * flow, so the root entry needs to be removed and the counter value to
         * check for this condition is 2.
         */
        if (ovs_refcount_unref(&esw_ctx->sample_ctx.root_sample_entry_refcount) == 2) {
            doca_sample_root_del_entry(netdev);
        }
    }

    /* Netdev can only be NULL during aux tables uninit. */
    if (netdev) {
        dpdk_offload_counter_dec(netdev);
    }

    doca_pipe_group_ctx_unref(doh->dfh.flow_res.next_group_ctx);
    doca_pipe_group_ctx_unref(doh->dfh.flow_res.self_group_ctx);
    /* The self is the most suffix. The end anchor's miss is from the prefix group.
     * Thus, the order of unref the groups should be the most suffix upwards.
     */
    for (int i = 0; i < num_of_prefixes; i++) {
        doca_pipe_group_ctx_unref(split_prefixes[i]->doh.dfh.flow_res.self_group_ctx);
    }

    doca_mirror_ctx_destroy(doh->dfh.flow_res.mirror_ctx);

    return 0;
}

static int
dpdk_offload_doca_destroy__(struct netdev *netdev,
                            struct dpdk_offload_handle *doh,
                            struct rte_flow_error *error)
{
    struct doca_offload_esw_ctx *esw_ctx = doca_offload_esw_ctx_get(netdev);
    unsigned int queue_id = netdev_offload_thread_id();

    return destroy_dpdk_offload_handle(esw_ctx, netdev, doh, queue_id, error);
}

static int
dpdk_offload_doca_destroy(struct netdev *netdev,
                          struct dpdk_offload_handle *doh,
                          struct rte_flow_error *error,
                          bool esw_port_id OVS_UNUSED)
{
    return doca_action_split_destroy(dpdk_offload_doca_destroy__, netdev, doh, error);
}

static int
doca_query_count(struct netdev *netdev,
                 struct doca_flow_pipe_entry *entry,
                 struct rte_flow_query_count *query,
                 struct rte_flow_error *error)
{
    struct doca_flow_resource_query stats;
    doca_error_t err;

    memset(query, 0, sizeof *query);
    memset(&stats, 0, sizeof stats);

    if (entry == NULL) {
        /* The async entry has not yet been completed,
         * it cannot have done anything yet. */
        return 0;
    }

    err = doca_flow_resource_query_entry(entry, &stats);
    if (err) {
        VLOG_WARN_RL(&rl, "%s: Failed to query entry: %p. Error %d (%s)",
                     netdev_get_name(netdev), entry, err,
                     doca_error_get_descr(err));
        error->type = RTE_FLOW_ERROR_TYPE_UNSPECIFIED;
        error->message = doca_error_get_descr(err);
        return -1;
    }

    query->hits = stats.counter.total_pkts;
    query->bytes = stats.counter.total_bytes;

    return 0;
}

static int
dpdk_offload_doca_query_count(struct netdev *netdev,
                              struct dpdk_offload_handle *doh,
                              struct rte_flow_query_count *query,
                              struct rte_flow_error *error)
{
    return doca_query_count(netdev, doh->dfh.flow, query, error);
}

static int
dpdk_offload_doca_shared_create(struct ovs_shared_ctx *ctx,
                                struct ovs_shared_arg *arg,
                                struct rte_flow_error *error)
{
    const struct rte_flow_action *action = arg->action;
    struct doca_flow_shared_resource_cfg shared_cfg;
    struct doca_offload_esw_ctx *esw_ctx;
    struct netdev *netdev = arg->netdev;
    unsigned int tid = arg->tid;
    uint32_t id = 0;

    esw_ctx = doca_offload_esw_ctx_get(netdev);

    if (action && action->type != RTE_FLOW_ACTION_TYPE_COUNT) {
        return -1;
    }

    error->type = RTE_FLOW_ERROR_TYPE_UNSPECIFIED;
    error->message = NULL;

    ctx->esw_offload_ctx = esw_ctx;

    switch (ctx->res_type) {
    case OVS_SHARED_CT_COUNT:
        if (doca_flow_port_shared_resource_get(esw_ctx->esw_port,
                                               DOCA_FLOW_SHARED_RESOURCE_COUNTER, &id)) {
            error->message = "CT counter exhausted: no free ID";
        } else {
            memset(&shared_cfg, 0, sizeof shared_cfg);
            if (doca_flow_port_shared_resource_set_cfg(esw_ctx->esw_port,
                                                       DOCA_FLOW_SHARED_RESOURCE_COUNTER, id,
                                                       &shared_cfg)) {
                error->message = "CT counter configuration failed";
            }
        }
        if (error->message) {
            /* CT-related message can be in high-volume, do not DoS our
             * syslog with errors. */
            VLOG_ERR_RL(&rl, "%s", error->message);
            return -1;
        }
        break;
    case OVS_SHARED_CT_ACTIONS:
        /* This is only compatible with DOCA-CT. */
        if (!esw_ctx->ct.pipe) {
            error->message = "Shared CT actions requires DOCA-CT";
            return -1;
        }
        if (!doca_flow_ct_shared_actions_create(esw_ctx, tid, arg->ct_actions_key, &id, error)) {
            return -1;
        }
        break;
    case OVS_SHARED_UNDEFINED:
        error->message = "Unsupported shared resource type requested";
        return -1;
    }

    ctx->res_id = id;
    if (action) {
        ctx->act_type = action->type;
    }

    return 0;
}

static int
dpdk_offload_doca_shared_destroy(struct ovs_shared_ctx *ctx,
                                 struct rte_flow_error *error OVS_UNUSED)
{
    struct doca_offload_esw_ctx *esw_ctx = ctx->esw_offload_ctx;

    switch (ctx->res_type) {
    case OVS_SHARED_CT_COUNT:
        doca_flow_port_shared_resource_put(esw_ctx->esw_port, DOCA_FLOW_SHARED_RESOURCE_COUNTER,
                                           ctx->res_id);
        break;
    case OVS_SHARED_CT_ACTIONS:
        if (conntrack_offload_doca_ct_enabled) {
            doca_flow_ct_shared_actions_destroy(esw_ctx, ctx->res_id);
        }
        break;
    case OVS_SHARED_UNDEFINED:
        VLOG_ERR("Unsupported shared resource type deletion");
        return -1;
    }

    return 0;
}

static int
dpdk_offload_doca_shared_query(struct ovs_shared_ctx *ctx,
                               void *data,
                               struct rte_flow_error *error)
{
    struct doca_offload_esw_ctx *esw_ctx = ctx->esw_offload_ctx;
    struct doca_flow_resource_query query_results;
    struct rte_flow_query_count *query;
    doca_error_t ret;
    uint32_t cnt_id;

    /* Only shared counter supported at the moment */
    if (ctx->act_type != RTE_FLOW_ACTION_TYPE_COUNT) {
        return -1;
    }

    query = (struct rte_flow_query_count *) data;
    memset(query, 0, sizeof *query);
    memset(&query_results, 0, sizeof query_results);

    cnt_id = ctx->res_id;
    ret = doca_flow_port_shared_resources_query(esw_ctx->esw_port,
                                                DOCA_FLOW_SHARED_RESOURCE_COUNTER, &cnt_id,
                                                &query_results, 1);
    if (ret != DOCA_SUCCESS) {
        VLOG_ERR("Failed to query shared counter id 0x%.8x: %s",
                 ctx->res_id, doca_error_get_descr(ret));
        error->type = RTE_FLOW_ERROR_TYPE_UNSPECIFIED;
        error->message = doca_error_get_descr(ret);
        return -1;
    }

    query->hits = query_results.counter.total_pkts;
    query->bytes = query_results.counter.total_bytes;

    return 0;
}

static int
ipv4_packet_hw_hash(struct doca_hash_pipe_ctx *hash_pipe_ctx,
                    struct dp_packet *packet,
                    uint32_t seed,
                    uint32_t *hash)
{
    struct ovs_doca_flow_match ofield_values;
    struct doca_flow_match *field_values;
    struct doca_flow_pipe *pipe;
    struct udp_header *udp;
    struct tcp_header *tcp;
    struct ip_header *ip;

    ip = dp_packet_l3(packet);
    field_values = &ofield_values.d;

    memset(&field_values->meta, 0, sizeof field_values->meta);
    doca_set_reg_val(&field_values->meta, REG_FIELD_SCRATCH, seed);
    if (ip->ip_proto == IPPROTO_UDP) {
        pipe = hash_pipe_ctx->hashes[HASH_TYPE_IPV4_UDP].pipe;
        udp = (struct udp_header *) (ip + 1);
        field_values->outer.udp.l4_port.src_port = (OVS_FORCE doca_be16_t) udp->udp_src;
        field_values->outer.udp.l4_port.dst_port = (OVS_FORCE doca_be16_t) udp->udp_dst;
        field_values->outer.l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_UDP;
        field_values->parser_meta.outer_l4_type = DOCA_FLOW_L4_META_UDP;
    } else if (ip->ip_proto == IPPROTO_TCP) {
        pipe = hash_pipe_ctx->hashes[HASH_TYPE_IPV4_TCP].pipe;
        tcp = (struct tcp_header *) (ip + 1);
        field_values->outer.tcp.l4_port.src_port = (OVS_FORCE doca_be16_t) tcp->tcp_src;
        field_values->outer.tcp.l4_port.dst_port = (OVS_FORCE doca_be16_t) tcp->tcp_dst;
        field_values->outer.l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_TCP;
        field_values->parser_meta.outer_l4_type = DOCA_FLOW_L4_META_TCP;
    } else {
        pipe = hash_pipe_ctx->hashes[HASH_TYPE_IPV4_L3].pipe;
    }
    field_values->parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV4;

    field_values->outer.l3_type = DOCA_FLOW_L3_TYPE_IP4;
    field_values->outer.ip4.src_ip = (OVS_FORCE doca_be32_t) get_16aligned_be32(&ip->ip_src);
    field_values->outer.ip4.dst_ip = (OVS_FORCE doca_be32_t) get_16aligned_be32(&ip->ip_dst);

    return doca_flow_pipe_calc_hash(pipe, field_values, hash);
}

static int
ipv6_packet_hw_hash(struct doca_hash_pipe_ctx *hash_pipe_ctx,
                    struct dp_packet *packet,
                    uint32_t seed,
                    uint32_t *hash)
{
    struct doca_flow_pipe *pre, *suf, *jumbo;
    struct ovs_doca_flow_match ofield_values;
    struct doca_flow_match *field_values;
    struct ovs_16aligned_ip6_hdr *ip6;
    struct udp_header *udp;
    struct tcp_header *tcp;
    int rv;

    ip6 = dp_packet_l3(packet);
    field_values = &ofield_values.d;

    memset(&field_values->meta, 0, sizeof field_values->meta);
    doca_set_reg_val(&field_values->meta, REG_FIELD_SCRATCH, seed);
    if (ip6->ip6_nxt == IPPROTO_UDP) {
        pre = hash_pipe_ctx->hashes[HASH_TYPE_IPV6_UDP_PRE].pipe;
        suf = hash_pipe_ctx->hashes[HASH_TYPE_IPV6_UDP_SUF].pipe;
        jumbo = hash_pipe_ctx->hashes[HASH_TYPE_IPV6_UDP_JUMBO].pipe;
        udp = (struct udp_header *) (ip6 + 1);
        field_values->outer.udp.l4_port.src_port = (OVS_FORCE doca_be16_t) udp->udp_src;
        field_values->outer.udp.l4_port.dst_port = (OVS_FORCE doca_be16_t) udp->udp_dst;
        field_values->outer.l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_UDP;
        field_values->parser_meta.outer_l4_type = DOCA_FLOW_L4_META_UDP;
    } else if (ip6->ip6_nxt == IPPROTO_TCP) {
        pre = hash_pipe_ctx->hashes[HASH_TYPE_IPV6_TCP_PRE].pipe;
        suf = hash_pipe_ctx->hashes[HASH_TYPE_IPV6_TCP_SUF].pipe;
        jumbo = hash_pipe_ctx->hashes[HASH_TYPE_IPV6_TCP_JUMBO].pipe;
        tcp = (struct tcp_header *) (ip6 + 1);
        field_values->outer.tcp.l4_port.src_port = (OVS_FORCE doca_be16_t) tcp->tcp_src;
        field_values->outer.tcp.l4_port.dst_port = (OVS_FORCE doca_be16_t) tcp->tcp_dst;
        field_values->outer.l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_TCP;
        field_values->parser_meta.outer_l4_type = DOCA_FLOW_L4_META_TCP;
    } else {
        pre = hash_pipe_ctx->hashes[HASH_TYPE_IPV6_L3_PRE].pipe;
        suf = hash_pipe_ctx->hashes[HASH_TYPE_IPV6_L3_SUF].pipe;
        jumbo = hash_pipe_ctx->hashes[HASH_TYPE_IPV6_L3_JUMBO].pipe;
    }
    field_values->parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV6;

    field_values->outer.l3_type = DOCA_FLOW_L3_TYPE_IP6;
    memcpy(&field_values->outer.ip6.src_ip, &ip6->ip6_src, sizeof ip6->ip6_src);
    memcpy(&field_values->outer.ip6.dst_ip, &ip6->ip6_dst, sizeof ip6->ip6_dst);

    if (jumbo) {
        return doca_flow_pipe_calc_hash(jumbo, field_values, hash);
    }

    rv = doca_flow_pipe_calc_hash(pre, field_values, hash);
    if (rv) {
        return rv;
    }

    memset(&field_values->meta, 0, sizeof field_values->meta);
    doca_set_reg_val(&field_values->meta, REG_FIELD_SCRATCH, *hash);

    return doca_flow_pipe_calc_hash(suf, field_values, hash);
}

static int
dpdk_offload_doca_packet_hw_hash(struct netdev *netdev,
                                 struct dp_packet *packet,
                                 uint32_t seed,
                                 uint32_t *hash)
{
    struct doca_hash_pipe_ctx *hash_pipe_ctx;
    struct doca_offload_esw_ctx *esw_ctx;
    ovs_be16 eth_proto;
    int rv = -1;

    esw_ctx = doca_offload_esw_ctx_get(netdev);
    if (!esw_ctx) {
        return -1;
    }
    hash_pipe_ctx = esw_ctx->hash_pipe_ctx;

    parse_tcp_flags(packet, &eth_proto, NULL, NULL);
    if (eth_proto == htons(ETH_TYPE_IP)) {
        rv = ipv4_packet_hw_hash(hash_pipe_ctx, packet, seed, hash);
    } else if (eth_proto == htons(ETH_TYPE_IPV6)) {
        rv = ipv6_packet_hw_hash(hash_pipe_ctx, packet, seed, hash);
    } else {
        /* Only IPv4/IPv6 are supported. */
        return -1;
    }

    if (rv) {
        *hash = 0;
    } else {
        *hash &= 0x0000FFFF;
        *hash |= 0xd0ca0000;
    }
    return rv;
}

/* Parse the hardware register values embedded within the encapsulated
 * IP header. Extract the source (src) and destination (dst) IP addresses,
 * which hold registers of index 0 and 1. Additionally, it decapsulates
 * the VXLAN header that was used to encapsulate and convey these metadata.
 */
static bool
dpdk_offload_doca_packet_parse_meta(struct dp_packet *packet, struct reg_tags *reg_tags)
{
    struct vxlan_meta_header *meta_hdr;

    /* Get and drop vxlan header used for storing metadata */
    meta_hdr = dp_packet_pull(packet, sizeof *meta_hdr);

    /* Parse reg tags */
    reg_tags->ct_state = meta_hdr->eth_dst.reg_field_ct_state;
    reg_tags->ct_zone = ntohs(meta_hdr->eth_dst.reg_field_ct_zone);
    reg_tags->ct_mark = ntohl(get_16aligned_be32(&meta_hdr->reg_field_ct_mark));
    reg_tags->ct_label = ntohl(get_16aligned_be32(&meta_hdr->reg_field_ct_label_id));
    reg_tags->nv_mp_pid = meta_hdr->eth_src.nv_mp.pid;
    reg_tags->nv_mp_preferred = meta_hdr->eth_src.nv_mp.preferred;
    reg_tags->nv_mp_strict = meta_hdr->eth_src.nv_mp.strict;
    packet->md.nv_mp.valid = true;

    return true;
}

static bool
dpdk_offload_doca_get_pkt_recover_info(struct dp_packet *p,
                                       struct dpdk_offload_recovery_info *info)
{
    memset(info, 0, sizeof *info);

    if (OVS_UNLIKELY(!dpdk_offload_doca_packet_parse_meta(p, &info->reg_tags))) {
        COVERAGE_INC(doca_invalid_meta_header);
        return false;
    }

    if (dpdk_offload_get_reg_field(p, REG_FIELD_FLOW_INFO, &info->flow_miss_id)) {
        if (OVS_UNLIKELY(info->flow_miss_id == SAMPLE_FLOW_MARK ||
                         info->flow_miss_id == NONSAMPLE_FLOW_MARK)) {
            if (info->flow_miss_id == SAMPLE_FLOW_MARK) {
                p->md.sample = true;
            }
            memset(info, 0, sizeof *info);
            return true;
        }
        dpdk_offload_get_reg_field(p, REG_FIELD_DP_HASH, &info->dp_hash);
    }

    return true;
}

static void
dpdk_offload_doca_update_stats(struct dpif_flow_stats *stats,
                               struct dpif_flow_stats *baseline_stats,
                               struct dpif_flow_attrs *attrs,
                               struct rte_flow_query_count *query)
{
    if (attrs) {
        attrs->dp_layer = "doca";
    }

    if (!stats || !query) {
        return;
    }

    if (stats->n_packets != query->hits) {
        query->hits_set = 1;
        query->bytes_set = 1;
    }

    stats->n_packets = query->hits;
    stats->n_bytes = query->bytes;
    if (baseline_stats) {
        stats->n_packets += baseline_stats->n_packets;
        stats->n_bytes += baseline_stats->n_bytes;
    }
}

static int
doca_offload_query_conn_dirs(struct netdev *netdevs[CT_DIR_NUM],
                             struct conn *conn,
                             struct dpif_flow_stats *stats,
                             struct dpif_flow_attrs *attrs,
                             struct rte_flow_error *error,
                             long long int now)
{
    struct rte_flow_query_count query = { .reset = 1 };
    struct ct_offload_handle *coh;
    int ret;

    if (attrs) {
        attrs->dp_extra_info = NULL;
        attrs->dp_offload_info = NULL;
        attrs->offloaded = true;
        attrs->dp_layer = "doca";
    }

    coh = conntrack_offload_get(conn);
    for (int dir = 0; dir < CT_DIR_NUM; dir++) {
        struct conn_item *ci = &coh->dir[dir].conn_item;
        struct ovs_shared_ctx *ctx;

        if (dir == CT_DIR_REP && conntrack_offload_is_unidir(conn)) {
            continue;
        }
        if (!coh->act_resources.shared_count_ctx) {
            ret = doca_query_count(netdevs[dir], ci->doca_flow, &query, error);
        } else {
            ctx = coh->act_resources.shared_count_ctx;
            ret = dpdk_offload_doca_shared_query(ctx, &query, error);
        }

        if (ret == 0) {
            break;
        }

        VLOG_DBG_RL(&rl, "%s: Failed to query conn %p",
                    netdev_get_name(netdevs[dir]), conn);
    }

    if (ret == 0) {
        dpdk_offload_doca_update_stats(&coh->stats, NULL, attrs, &query);
        if (query.hits_set && query.hits) {
            coh->stats.used = now;
        }

        if (stats) {
            memcpy(stats, &coh->stats, sizeof *stats);
        }
    }

    return ret;
}

static int
doca_offload_query_conn_doca_ct(struct netdev *netdevs[CT_DIR_NUM],
                                struct conn *conn,
                                struct dpif_flow_stats *stats,
                                struct dpif_flow_attrs *attrs,
                                struct rte_flow_error *error,
                                long long int now)
{
    struct doca_flow_resource_query queries[CT_DIR_NUM];
    struct doca_offload_esw_ctx *esw;
    struct ct_offload_handle *coh;
    struct netdev *netdev;
    struct conn_item *ci;
    uint64_t last_hit_s;
    unsigned int qid;
    doca_error_t err;

    if (attrs) {
        attrs->dp_extra_info = NULL;
        attrs->dp_offload_info = NULL;
        attrs->offloaded = true;
        attrs->dp_layer = "doca";
    }

    netdev = netdevs[CT_DIR_INIT] ? netdevs[CT_DIR_INIT]
                                  : netdevs[CT_DIR_REP];
    if (!netdev) {
        error->type = RTE_FLOW_ERROR_TYPE_UNSPECIFIED;
        error->message = "Missing netdev";
        return -ENODEV;
    }

    coh = conntrack_offload_get(conn);
    ci = &coh->dir[0].conn_item;
    qid = doca_offload_ct_queue_id(conntrack_offload_get_insertion_tid(conn));
    esw = doca_offload_esw_ctx_get(netdev);

    if (ci->doca_flow == NULL) {
        error->type = RTE_FLOW_ERROR_TYPE_HANDLE;
        error->message = "missing entry handle";
        return -1;
    }

    err = doca_flow_ct_query_entry(qid, esw->ct.pipe,
                                   DOCA_FLOW_CT_ENTRY_FLAGS_NO_WAIT,
                                   ci->doca_flow, &queries[CT_DIR_INIT],
                                   &queries[CT_DIR_REP], &last_hit_s);
    if (DOCA_IS_ERROR(err)) {
        VLOG_DBG_RL(&rl, "%s/%s: Failed to query conn %p",
                    netdev_get_name(netdevs[CT_DIR_INIT]),
                    netdev_nullable_get_name(netdevs[CT_DIR_REP]),
                    conn);
        error->type = RTE_FLOW_ERROR_TYPE_UNSPECIFIED;
        error->message = doca_error_get_descr(err);
        return -1;
    }

    if (coh->stats.n_packets != queries[CT_DIR_INIT].counter.total_pkts) {
        coh->stats.used = now;
    }
    coh->stats.n_packets = queries[CT_DIR_INIT].counter.total_pkts;
    coh->stats.n_bytes = queries[CT_DIR_INIT].counter.total_bytes;

    if (stats) {
        memcpy(stats, &coh->stats, sizeof *stats);
    }

    return 0;
}

static int
dpdk_offload_doca_query_conn(struct netdev *netdevs[CT_DIR_NUM],
                             struct conn *conn,
                             struct dpif_flow_stats *stats,
                             struct dpif_flow_attrs *attrs,
                             struct rte_flow_error *error,
                             long long int now)
{
    struct ct_offload_handle *coh;
    struct conn_item *ci;

    coh = conntrack_offload_get(conn);
    ci = &coh->dir[0].conn_item;

    if (ci->use_doca_ct) {
        return doca_offload_query_conn_doca_ct(netdevs, conn, stats, attrs, error, now);
    } else {
        return doca_offload_query_conn_dirs(netdevs, conn, stats, attrs, error, now);
    }
}

static int
doca_create_ct_zone_revisit_rule(struct netdev *netdev,
                                 struct doca_flow_pipe *pipe,
                                 uint16_t zone, int nat,
                                 struct dpdk_offload_handle *doh)
{
    struct doca_flow_handle_resources flow_res;
    struct doca_pipe_group_ctx *next_group_ctx;
    uint32_t ct_state_spec, ct_state_mask;
    struct ovs_doca_flow_match spec;
    struct ovs_doca_flow_match mask;
    struct doca_flow_handle *hndl;
    struct rte_flow_error error;
    struct doca_flow_fwd fwd;

    memset(&flow_res, 0, sizeof flow_res);
    memset(&spec, 0, sizeof spec);
    memset(&mask, 0, sizeof mask);
    memset(&fwd, 0, sizeof fwd);

    /* If the zone is the same, and already visited ct/ct-nat, skip
     * ct/ct-nat and jump directly to post-ct.
     */
    ct_state_spec = OVS_CS_F_TRACKED;
    if (nat) {
        ct_state_spec |= OVS_CS_F_NAT_MASK;
    }
    ct_state_mask = ct_state_spec;

    doca_set_reg_val_mask(&spec.d.meta, &mask.d.meta, REG_FIELD_CT_ZONE, zone);
    doca_set_reg_val(&spec.d.meta, REG_FIELD_CT_STATE, ct_state_spec);
    doca_set_reg_val(&mask.d.meta, REG_FIELD_CT_STATE, ct_state_mask);

    next_group_ctx = doca_pipe_group_ctx_ref(netdev, POSTCT_TABLE_ID, NULL);
    if (!next_group_ctx) {
        return -1;
    }

    fwd.type = DOCA_FLOW_FWD_PIPE;
    fwd.next_pipe = doca_pipe_group_get_pipe(next_group_ctx);
    flow_res.next_group_ctx = next_group_ctx;

    hndl = create_doca_flow_handle_control(netdev, AUX_QUEUE,
                                           DPDK_OFFLOAD_PRIORITY_HIGH, pipe,
                                           &spec, &mask, NULL, NULL, NULL, &fwd,
                                           &flow_res, doh, &error);
    if (!hndl) {
        doca_pipe_group_ctx_unref(next_group_ctx);
        return -1;
    }
    doh->dfh.flow_res.next_group_ctx = flow_res.next_group_ctx;

    return 0;
}

static int
doca_create_ct_zone_uphold_rule(struct netdev *netdev,
                                struct doca_offload_esw_ctx *ctx,
                                struct doca_flow_pipe *pipe,
                                uint16_t zone,
                                enum ct_zone_cls_flow_type type,
                                struct dpdk_offload_handle *doh)
{
    struct ovs_doca_flow_actions dacts, dacts_masks;
    struct doca_flow_handle_resources flow_res;
    struct ovs_doca_flow_match spec;
    struct ovs_doca_flow_match mask;
    struct doca_flow_handle *hndl;
    struct rte_flow_error error;
    struct doca_flow_fwd fwd;

    memset(&dacts_masks, 0, sizeof dacts_masks);
    memset(&flow_res, 0, sizeof flow_res);
    memset(&dacts, 0, sizeof dacts);
    memset(&fwd, 0, sizeof fwd);
    memset(&spec, 0, sizeof spec);
    memset(&mask, 0, sizeof mask);

    if (type == CT_ZONE_FLOW_UPHOLD_IP6_TCP ||
        type == CT_ZONE_FLOW_UPHOLD_IP6_UDP) {
        spec.d.parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV6;
        mask.d.parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV6;
    } else {
        spec.d.parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV4;
        mask.d.parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV4;
    }

    if (type == CT_ZONE_FLOW_UPHOLD_IP4_TCP ||
        type == CT_ZONE_FLOW_UPHOLD_IP6_TCP) {
        spec.d.outer.l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_TCP;
        mask.d.outer.l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_TCP;
        /* Ensure that none of SYN | RST | FIN flag is set in
         * packets going to CT: they must miss and go to SW. */
        mask.d.outer.tcp.flags = TCP_SYN | TCP_RST | TCP_FIN;
        spec.d.parser_meta.outer_l4_type = DOCA_FLOW_L4_META_TCP;
        mask.d.parser_meta.outer_l4_type = DOCA_FLOW_L4_META_TCP;
    } else {
        spec.d.parser_meta.outer_l4_type = DOCA_FLOW_L4_META_UDP;
        mask.d.parser_meta.outer_l4_type = DOCA_FLOW_L4_META_UDP;
    }

    doca_set_reg_val_mask(&dacts.d.meta, &dacts_masks.d.meta, REG_FIELD_CT_ZONE, zone);

    fwd.type = DOCA_FLOW_FWD_PIPE;
    fwd.next_pipe = doca_get_ct_pipe(ctx, &spec.d);
    flow_res.next_group_ctx = NULL;

    hndl = create_doca_flow_handle_control(netdev, AUX_QUEUE,
                                           DPDK_OFFLOAD_PRIORITY_MED, pipe, &spec,
                                           &mask, &dacts, &dacts_masks, NULL,
                                           &fwd, &flow_res, doh, &error);
    if (!hndl) {
        return -1;
    }
    return 0;
}

static int
doca_create_ct_zone_miss_rule(struct netdev *netdev,
                              struct doca_offload_esw_ctx *ctx,
                              struct doca_flow_pipe *pipe,
                              struct dpdk_offload_handle *doh)
{
    struct ovs_doca_flow_actions dacts, dacts_masks;
    struct doca_flow_handle_resources flow_res;
    struct ovs_doca_flow_match spec;
    struct ovs_doca_flow_match mask;
    struct doca_flow_handle *hndl;
    struct rte_flow_error error;
    struct doca_flow_fwd fwd;

    memset(&dacts_masks, 0, sizeof dacts_masks);
    memset(&flow_res, 0, sizeof flow_res);
    memset(&dacts, 0, sizeof dacts);
    memset(&fwd, 0, sizeof fwd);
    memset(&spec, 0, sizeof spec);
    memset(&mask, 0, sizeof mask);

    fwd.type = DOCA_FLOW_FWD_PIPE;
    fwd.next_pipe = doca_offload_get_misspath_pipe(ctx);
    hndl = create_doca_flow_handle_control(netdev, AUX_QUEUE,
                                           DPDK_OFFLOAD_PRIORITY_MISS, pipe, &spec,
                                           &mask, &dacts, &dacts_masks, NULL,
                                           &fwd, &flow_res, doh, &error);
    return hndl ? 0 : -1;
}

static void
doca_ct_zone_rules_uninit(struct doca_offload_esw_ctx *esw_ctx, uint32_t zone)
    OVS_REQUIRES(esw_ctx->ct_zones_mutex)
{
    struct dpdk_offload_handle *doh;
    int nat, i;

    if (netdev_is_zone_tables_disabled()) {
        VLOG_ERR("Disabling ct zones is not supported with doca");
        return;
    }

    for (nat = 0; nat < 2; nat++) {
        for (i = 0; i < CT_ZONE_FLOWS_NUM; i++) {
            doh = &esw_ctx->ct_zones[zone].entries[nat][i];
            doca_offload_remove_entry(esw_ctx, AUX_QUEUE, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                      &doh->dfh.flow);
            doca_pipe_group_ctx_unref(doh->dfh.flow_res.next_group_ctx);
        }
    }
}

static void
doca_ct_zone_ctx_uninit(struct doca_offload_esw_ctx *esw_ctx, uint32_t zone)
    OVS_REQUIRES(esw_ctx->ct_zones_mutex)
{
    struct doca_flow_pipe *plain_pipe, *nat_pipe;

    nat_pipe = ovsrcu_get(struct doca_flow_pipe *, &esw_ctx->ct_zones[zone].nat_pipe);
    plain_pipe = ovsrcu_get(struct doca_flow_pipe *, &esw_ctx->ct_zones[zone].plain_pipe);

    doca_pipe_group_pipe_destroy(esw_ctx, plain_pipe);
    doca_pipe_group_pipe_destroy(esw_ctx, nat_pipe);
    ovsrcu_set(&esw_ctx->ct_zones[zone].plain_pipe, NULL);
    ovsrcu_set(&esw_ctx->ct_zones[zone].nat_pipe, NULL);
}

static int
doca_ct_zone_rules_init(struct netdev *netdev,
                        struct doca_offload_esw_ctx *esw_ctx, uint32_t zone)
    OVS_REQUIRES(esw_ctx->ct_zones_mutex)
{
    struct dpdk_offload_handle *doh;
    int nat;

    for (nat = 0; nat < 2; nat++) {
        struct doca_flow_pipe *pipe;

        pipe = nat
               ? ovsrcu_get(struct doca_flow_pipe *, &esw_ctx->ct_zones[zone].plain_pipe)
               : ovsrcu_get(struct doca_flow_pipe *, &esw_ctx->ct_zones[zone].nat_pipe);
        /* If the zone is already set, then CT for this zone has already
         * been executed: skip to post-ct. */

        doh = &esw_ctx->ct_zones[zone].entries[nat][CT_ZONE_FLOW_REVISIT];
        if (doca_create_ct_zone_revisit_rule(netdev, pipe, zone, nat, doh)) {
            goto err;
        }

        /* Otherwise, set the zone and go to CT/CT-NAT. */

        doh = &esw_ctx->ct_zones[zone].entries[nat][CT_ZONE_FLOW_UPHOLD_IP4_UDP];
        if (doca_create_ct_zone_uphold_rule(netdev, esw_ctx,
                                            pipe, zone,
                                            CT_ZONE_FLOW_UPHOLD_IP4_UDP,
                                            doh)) {
            goto err;
        }

        doh = &esw_ctx->ct_zones[zone].entries[nat][CT_ZONE_FLOW_UPHOLD_IP4_TCP];
        if (doca_create_ct_zone_uphold_rule(netdev, esw_ctx,
                                            pipe,
                                            zone,
                                            CT_ZONE_FLOW_UPHOLD_IP4_TCP,
                                            doh)) {
            goto err;
        }

        doh = &esw_ctx->ct_zones[zone].entries[nat][CT_ZONE_FLOW_MISS];
        if (doca_create_ct_zone_miss_rule(netdev, esw_ctx, pipe, doh)) {
            goto err;
        }

        if (!conntrack_offload_ipv6_is_enabled()) {
            continue;
        }

        doh = &esw_ctx->ct_zones[zone].entries[nat][CT_ZONE_FLOW_UPHOLD_IP6_UDP];
        if (doca_create_ct_zone_uphold_rule(netdev, esw_ctx,
                                            pipe,
                                            zone,
                                            CT_ZONE_FLOW_UPHOLD_IP6_UDP,
                                            doh)) {
            goto err;
        }

        doh = &esw_ctx->ct_zones[zone].entries[nat][CT_ZONE_FLOW_UPHOLD_IP6_TCP];
        if (doca_create_ct_zone_uphold_rule(netdev, esw_ctx,
                                            pipe,
                                            zone,
                                            CT_ZONE_FLOW_UPHOLD_IP6_TCP,
                                            doh)) {
            goto err;
        }
    }

    return 0;

err:
    doca_ct_zone_rules_uninit(esw_ctx, zone);
    doca_ct_zone_ctx_uninit(esw_ctx, zone);
    return -1;
}

static doca_error_t
doca_ct_zone_ctx_init_create_control_pipe(struct netdev *netdev,
                                          uint32_t zone, bool nat,
                                          struct doca_flow_pipe **pipe)
{
    struct doca_flow_port *doca_port;
    struct doca_flow_pipe_cfg *cfg;
    char pipe_name[100];
    doca_error_t ret;

    doca_port = netdev_doca_port_get(netdev);
    ret = doca_flow_pipe_cfg_create(&cfg, doca_flow_port_switch_get(doca_port));
    if (ret) {
        return ret;
    }

    snprintf(pipe_name, sizeof pipe_name, "OVS_CT_ZONE_CONTROL_PIPE_%s_%u_%s",
             netdev_get_name(netdev), zone, nat ? "NAT" : "PLAIN");
    if (doca_flow_pipe_cfg_set_nr_entries(cfg, CT_ZONE_FLOWS_NUM) ||
        doca_flow_pipe_cfg_set_name(cfg, pipe_name) ||
        doca_pipe_cfg_allow_one_queue(cfg, AUX_QUEUE) ||
        doca_flow_pipe_cfg_set_type(cfg, DOCA_FLOW_PIPE_CONTROL)) {
        ret = DOCA_ERROR_UNKNOWN;
        VLOG_ERR("%s: Could not set doca_flow_pipe_cfg for group control pipe",
                 netdev_get_name(netdev));
        goto out;
    }

    ret = doca_flow_pipe_create(cfg, NULL, NULL, pipe);
out:
    doca_flow_pipe_cfg_destroy(cfg);
    return ret;
}

static int
doca_ct_zone_ctx_init(struct netdev *netdev, struct doca_offload_esw_ctx *esw_ctx,
                      uint32_t zone)
    OVS_REQUIRES(esw_ctx->ct_zones_mutex)
{
    struct ct_zone_ctx *ct_zone_ctx = &esw_ctx->ct_zones[zone];
    struct doca_flow_pipe *pipe_ct, *pipe_ct_nat;
    doca_error_t err;

    err = doca_ct_zone_ctx_init_create_control_pipe(netdev, zone, true, &pipe_ct);
    if (err) {
        goto pipe_create_err;
    }

    err = doca_ct_zone_ctx_init_create_control_pipe(netdev, zone, false, &pipe_ct_nat);
    if (err) {
        goto pipe_create_nat_err;
    }

    ovsrcu_set(&ct_zone_ctx->nat_pipe, pipe_ct_nat);
    ovsrcu_set(&ct_zone_ctx->plain_pipe, pipe_ct);

    return 0;

pipe_create_nat_err:
    doca_flow_pipe_destroy(pipe_ct);
pipe_create_err:
    return -1;
}

static int
doca_ct_zone_init(struct netdev *netdev, struct doca_offload_esw_ctx *esw_ctx,
                  uint32_t zone_group)
    OVS_EXCLUDED(esw_ctx->ct_zones_mutex)
{
    uint32_t zone = zone_group & reg_fields[REG_FIELD_CT_ZONE].mask;
    int ret = 0;

    if (!doca_pipe_group_is_ct_zone_group_id(zone_group)) {
        return -1;
    }

    if (ovsrcu_get(struct doca_flow_pipe *,
                   &esw_ctx->ct_zones[zone].plain_pipe) != NULL) {
        return ret;
    }

    ovs_mutex_lock(&esw_ctx->ct_zones_mutex);
    ovs_assert(!netdev_is_zone_tables_disabled());
    /* Read again after locking to resolve concurrent initialization. */
    if (ovsrcu_get(struct doca_flow_pipe *,
                   &esw_ctx->ct_zones[zone].plain_pipe) != NULL) {
        goto unlock;
    }

    if (doca_ct_zone_ctx_init(netdev, esw_ctx, zone)) {
        ret = -1;
        goto unlock;
    }

    if (doca_ct_zone_rules_init(netdev, esw_ctx, zone)) {
        doca_ct_zone_ctx_uninit(esw_ctx, zone);
        ret = -1;
        goto unlock;
    }

unlock:
    ovs_mutex_unlock(&esw_ctx->ct_zones_mutex);
    return ret;
}

static void
doca_ct_zones_uninit(struct doca_offload_esw_ctx *esw_ctx)
    OVS_EXCLUDED(esw_ctx->ct_zones_mutex)
{
    uint32_t zone;

    if (netdev_is_zone_tables_disabled()) {
        VLOG_ERR("Disabling ct zones is not supported with doca");
        return;
    }

    ovs_mutex_lock(&esw_ctx->ct_zones_mutex);
    for (zone = 0; zone < NUM_CT_ZONES; zone++) {
        if (ovsrcu_get(struct doca_flow_pipe *,
                       &esw_ctx->ct_zones[zone].plain_pipe) == NULL) {
            continue;
        }

        doca_ct_zone_rules_uninit(esw_ctx, zone);
        doca_ct_zone_ctx_uninit(esw_ctx, zone);
    }
    ovs_mutex_unlock(&esw_ctx->ct_zones_mutex);
}

static void
doca_ct_pipe_destroy(struct doca_offload_esw_ctx *ctx,
                     struct doca_basic_pipe_ctx *pipe_ctx)
{
    doca_basic_pipe_ctx_uninit(ctx, pipe_ctx);
}

static void
doca_ct_pipes_destroy(struct doca_offload_esw_ctx *ctx)
{
    struct doca_basic_pipe_ctx *pipe_ctx;
    int i, j;

    doca_ct_pipe_destroy(ctx, &ctx->ct_ip6_prefix);

    for (i = 0; i < NUM_CT_NW; i++) {
        for (j = 0; j < NUM_CT_TP; j++) {
            pipe_ctx = &ctx->ct_pipes[i][j];
            doca_ct_pipe_destroy(ctx, pipe_ctx);
        }
    }
}

static void
doca_basic_pipe_name(struct ds *s, enum ct_nw_type nw_type, enum ct_tp_type tp_type)
{
    ds_put_cstr(s, "BASIC_CT_PIPE");

    switch (nw_type) {
    case CT_NW_IP4:
        ds_put_cstr(s, "_IP4");
        break;
    case CT_NW_IP6:
        ds_put_cstr(s, "_IP6_SUFFIX");
        break;
    case NUM_CT_NW:
       OVS_NOT_REACHED();
    }

    switch (tp_type) {
    case CT_TP_UDP:
        ds_put_cstr(s, "_UDP");
        break;
    case CT_TP_TCP:
        ds_put_cstr(s, "_TCP");
        break;
    case NUM_CT_TP:
       OVS_NOT_REACHED();
    }
}

static struct ovs_doca_flow_match ct_matches[NUM_CT_NW][NUM_CT_TP] = {
    [CT_NW_IP4] = {
        [CT_TP_UDP] = { .d = {
            .outer.l3_type = DOCA_FLOW_L3_TYPE_IP4,
            .outer.ip4.src_ip = BE32_MAX,
            .outer.ip4.dst_ip = BE32_MAX,
            .outer.l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_UDP,
            .outer.udp.l4_port.src_port = BE16_MAX,
            .outer.udp.l4_port.dst_port = BE16_MAX,
            .parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV4,
            .parser_meta.outer_l4_type = DOCA_FLOW_L4_META_UDP,
            /* For pure functionality this match is not needed, instead use it
             * as a WA to take advantage of the lower level optimization.
             */
            .parser_meta.port_id = UINT16_MAX,
        }, },
        [CT_TP_TCP] = { .d = {
            .outer.l3_type = DOCA_FLOW_L3_TYPE_IP4,
            .outer.ip4.src_ip = BE32_MAX,
            .outer.ip4.dst_ip = BE32_MAX,
            .outer.l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_TCP,
            .outer.tcp.l4_port.src_port = BE16_MAX,
            .outer.tcp.l4_port.dst_port = BE16_MAX,
            .parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV4,
            .parser_meta.outer_l4_type = DOCA_FLOW_L4_META_TCP,
            /* For pure functionality this match is not needed, instead use it
             * as a WA to take advantage of the lower level optimization.
             */
            .parser_meta.port_id = UINT16_MAX,
        }, },
    },
    [CT_NW_IP6] = {
        [CT_TP_UDP] = { .d = {
            .outer.l3_type = DOCA_FLOW_L3_TYPE_IP6,
            .outer.ip6.dst_ip[0] = BE32_MAX,
            .outer.ip6.dst_ip[1] = BE32_MAX,
            .outer.ip6.dst_ip[2] = BE32_MAX,
            .outer.ip6.dst_ip[3] = BE32_MAX,
            .outer.l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_UDP,
            .outer.udp.l4_port.src_port = BE16_MAX,
            .outer.udp.l4_port.dst_port = BE16_MAX,
            .parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV6,
            .parser_meta.outer_l4_type = DOCA_FLOW_L4_META_UDP,
            /* For pure functionality this match is not needed, instead use it
             * as a WA to take advantage of the lower level optimization.
             */
            .parser_meta.port_id = UINT16_MAX,
        }, },
        [CT_TP_TCP] = { .d = {
            .outer.l3_type = DOCA_FLOW_L3_TYPE_IP6,
            .outer.ip6.dst_ip[0] = BE32_MAX,
            .outer.ip6.dst_ip[1] = BE32_MAX,
            .outer.ip6.dst_ip[2] = BE32_MAX,
            .outer.ip6.dst_ip[3] = BE32_MAX,
            .outer.l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_TCP,
            .outer.tcp.l4_port.src_port = BE16_MAX,
            .outer.tcp.l4_port.dst_port = BE16_MAX,
            .parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV6,
            .parser_meta.outer_l4_type = DOCA_FLOW_L4_META_TCP,
            /* For pure functionality this match is not needed, instead use it
             * as a WA to take advantage of the lower level optimization.
             */
            .parser_meta.port_id = UINT16_MAX,
        }, },
    },
};

static int
doca_ct_ip6_prefix_pipe_init(struct netdev *netdev,
                             struct doca_offload_esw_ctx *ctx)
{
    struct ovs_doca_flow_match omatch, omatch_mask;
    struct doca_flow_match *match, *match_mask;
    struct doca_basic_pipe_ctx *pipe_ctx;
    struct ovs_doca_flow_actions actions;
    struct doca_flow_fwd miss;
    struct doca_flow_fwd fwd;

    pipe_ctx = &ctx->ct_ip6_prefix;

    memset(&fwd, 0, sizeof fwd);
    memset(&miss, 0, sizeof miss);
    memset(&omatch, 0, sizeof omatch);
    memset(&actions, 0, sizeof actions);
    memset(&omatch_mask, 0, sizeof omatch_mask);
    match = &omatch.d;
    match_mask = &omatch_mask.d;

    miss.type = DOCA_FLOW_FWD_PIPE;
    miss.next_pipe = doca_offload_get_misspath_pipe(ctx);
    /* Next pipe will be determined per connection based on L4 type */
    fwd.type = DOCA_FLOW_FWD_PIPE;
    fwd.next_pipe = NULL;

    doca_set_reg_mask(&actions.d.meta, REG_FIELD_SCRATCH);

    /* Set ip6 ct tuple match template (zone + src_ip + l4_type) */
    doca_set_reg_mask(&match->meta, REG_FIELD_CT_ZONE);
    doca_set_reg_mask(&match_mask->meta, REG_FIELD_CT_ZONE);
    match->parser_meta.port_id = UINT16_MAX;
    match_mask->parser_meta.port_id = UINT16_MAX;
    match->outer.l3_type = DOCA_FLOW_L3_TYPE_IP6;
    match_mask->outer.l3_type = DOCA_FLOW_L3_TYPE_IP6;
    match->parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV6;
    match_mask->parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV6;
    memset(&match->outer.ip6.src_ip, 0xFF, sizeof match->outer.ip6.src_ip);
    memset(&match_mask->outer.ip6.src_ip, 0xFF, sizeof match->outer.ip6.src_ip);
    match->outer.ip6.next_proto = UINT8_MAX;
    match_mask->outer.ip6.next_proto = UINT8_MAX;

    return doca_offload_basic_pipe_create(netdev, &omatch, &omatch_mask, NULL, &actions, NULL,
                                          NULL, &fwd, &miss, ovs_doca_max_ct_rules(),
                                          UINT64_C(1) << AUX_QUEUE, "BASIC_CT_PIPE_IP6_PREFIX",
                                          &pipe_ctx->pipe);
}

static void
build_ct_actions_templates(struct ovs_doca_flow_actions *oactions,
                           struct ovs_doca_flow_actions *oactions_masks,
                           enum ct_nw_type nw_type, enum ct_tp_type tp_type, bool nat)
{
    struct doca_flow_actions *actions = &oactions->d, *actions_masks = &oactions_masks->d;
    enum dpdk_reg_id regs[] = {
        REG_FIELD_CT_STATE,
        REG_FIELD_CT_MARK,
        REG_FIELD_CT_LABEL_ID,
    };

    memset(oactions, 0, sizeof *oactions);
    if (oactions_masks) {
        memset(oactions_masks, 0, sizeof *oactions_masks);
    }

    for (size_t i = 0; i < ARRAY_SIZE(regs); i++) {
        /* Use 0xFFs values to set meta.u32 in the action upon pipe create
         * and have the mask in the actions_masks
         */
        doca_offload_set_reg_template(&actions->meta, regs[i]);
        if (actions_masks && actions_masks != actions) {
            doca_set_reg_mask(&actions_masks->meta, regs[i]);
        }
    }

    if (nat) {
        struct doca_flow_header_format *outer = &actions->outer;

        if (nw_type == CT_NW_IP4) {
            outer->l3_type = DOCA_FLOW_L3_TYPE_IP4;
            outer->ip4.src_ip = BE32_MAX;
            outer->ip4.dst_ip = BE32_MAX;
        } else if (nw_type == CT_NW_IP6) {
            outer->l3_type = DOCA_FLOW_L3_TYPE_IP6;
            memset(&outer->ip6.src_ip, UINT8_MAX, sizeof outer->ip6.src_ip) ;
            memset(&outer->ip6.dst_ip, UINT8_MAX, sizeof outer->ip6.dst_ip) ;
        } else {
            OVS_NOT_REACHED();
        }

        if (tp_type == CT_TP_UDP) {
            outer->l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_UDP;
            outer->udp.l4_port.src_port = BE16_MAX;
            outer->udp.l4_port.dst_port = BE16_MAX;
        } else {
            outer->l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_TCP;
            outer->tcp.l4_port.src_port = BE16_MAX;
            outer->tcp.l4_port.dst_port = BE16_MAX;
        }
        if (actions_masks && actions_masks != actions) {
            memcpy(&actions_masks->outer, &actions->outer, sizeof actions_masks->outer);
        }
    }
}

static int
doca_ct_pipe_init(struct netdev *netdev, struct doca_offload_esw_ctx *ctx,
                  enum ct_nw_type nw_type, enum ct_tp_type tp_type)
{
    struct doca_pipe_group_ctx *fwd_group_ctx = NULL;
    struct ovs_doca_flow_actions actions_masks;
    struct doca_basic_pipe_ctx *pipe_ctx;
    struct ovs_doca_flow_actions actions;
    struct doca_flow_monitor monitor;
    struct doca_flow_fwd miss;
    struct doca_flow_fwd fwd;
    struct ds pipe_name;
    int ret = 0;

    pipe_ctx = &ctx->ct_pipes[nw_type][tp_type];

    /* Do not re-init a pipe if already done. */
    if (pipe_ctx->pipe != NULL) {
        return 0;
    }

    /* Do not initialize IPv6 pipes if not enabled. */
    if (nw_type == CT_NW_IP6 &&
        (!conntrack_offload_ipv6_is_enabled() || conntrack_offload_doca_ct_ipv6_enabled)) {
        return 0;
    }

    /* IPv4 uses DOCA-CT instead of basic pipes if enabled. */
    if (conntrack_offload_doca_ct_enabled && nw_type == CT_NW_IP4) {
        return 0;
    }

    memset(&fwd, 0, sizeof fwd);
    memset(&miss, 0, sizeof miss);
    memset(&monitor, 0, sizeof monitor);

    /* CT offloads on basic pipes *always* modify the packet header,
     * even for plain-CT. In such case, all fields are written with
     * their own value. */
    build_ct_actions_templates(&actions, &actions_masks, nw_type, tp_type, true);

    /* Finalize the match templates. */
    doca_set_reg_mask(&ct_matches[CT_NW_IP4][CT_TP_UDP].d.meta, REG_FIELD_CT_ZONE);
    doca_set_reg_mask(&ct_matches[CT_NW_IP4][CT_TP_TCP].d.meta, REG_FIELD_CT_ZONE);
    /* Add prefix id match to IPv6 pipes */
    doca_set_reg_mask(&ct_matches[CT_NW_IP6][CT_TP_UDP].d.meta, REG_FIELD_SCRATCH);
    doca_set_reg_mask(&ct_matches[CT_NW_IP6][CT_TP_TCP].d.meta, REG_FIELD_SCRATCH);

    monitor.counter_type = DOCA_FLOW_RESOURCE_TYPE_SHARED;
    monitor.shared_counter.shared_counter_id = UINT32_MAX;

    fwd_group_ctx = doca_pipe_group_ctx_ref(netdev, POSTCT_TABLE_ID, NULL);
    if (fwd_group_ctx == NULL) {
        VLOG_ERR("%s: Failed to take a reference on post-ct table",
                 netdev_get_name(netdev));
        return -1;
    }
    fwd.type = DOCA_FLOW_FWD_PIPE;
    fwd.next_pipe = doca_pipe_group_get_pipe(fwd_group_ctx);

    miss.type = DOCA_FLOW_FWD_PIPE;
    miss.next_pipe = doca_offload_get_misspath_pipe(ctx);

    ds_init(&pipe_name);
    doca_basic_pipe_name(&pipe_name, nw_type, tp_type);

    if (doca_offload_basic_pipe_create(netdev, &ct_matches[nw_type][tp_type], NULL, &monitor,
                                       &actions, &actions_masks, NULL, &fwd, &miss,
                                       ovs_doca_max_ct_rules(), PMD_QUEUES_BITMAP,
                                       ds_cstr(&pipe_name), &pipe_ctx->pipe)) {
        ret = -1;
        goto out;
    }

    pipe_ctx->fwd_group_ctx = fwd_group_ctx;

out:
    ds_destroy(&pipe_name);
    if (ret) {
        doca_pipe_group_ctx_unref(fwd_group_ctx);
    }
    return ret;
}

static int
doca_ct_pipes_init(struct netdev *netdev, struct doca_offload_esw_ctx *ctx)
{
    int i, j;

    for (i = 0; i < NUM_CT_NW; i++) {
        for (j = 0; j < NUM_CT_TP; j++) {
            if (doca_ct_pipe_init(netdev, ctx, i, j)) {
                goto error;
            }
        }
    }

    if (conntrack_offload_ipv6_is_enabled() && !conntrack_offload_doca_ct_ipv6_enabled) {
        if (doca_ct_ip6_prefix_pipe_init(netdev, ctx)) {
            goto error;
        }
    }

    return 0;

error:
    /* Rollback any pipe creation. */
    doca_ct_pipes_destroy(ctx);
    return -1;
}

static void
doca_offload_ct_pipe_uninit(struct doca_offload_esw_ctx *esw)
{
    doca_pipe_group_pipe_destroy(esw, esw->ct.pipe);
    esw->ct.pipe = NULL;

    doca_pipe_group_ctx_unref(esw->ct.post_ct_ctx);
    esw->ct.post_ct_ctx = NULL;
}

static int
doca_offload_ct_pipe_init(struct netdev *netdev, struct doca_offload_esw_ctx *esw)
{
    struct doca_flow_actions *actions_masks_arr[NUM_CT_ACTIONS];
    struct ovs_doca_flow_actions actions_masks[NUM_CT_ACTIONS];
    struct doca_flow_actions *actions_arr[NUM_CT_ACTIONS];
    struct ovs_doca_flow_actions actions[NUM_CT_ACTIONS];
    struct doca_pipe_group_ctx *post_ct_ctx = NULL;
    doca_error_t err = DOCA_ERROR_UNKNOWN;
    struct doca_flow_pipe_cfg *cfg = NULL;
    struct doca_flow_pipe *ct_pipe = NULL;
    struct doca_flow_fwd fwd[NUM_CT_FWD];
    struct ovs_doca_flow_match match;
    struct doca_flow_port *port;
    unsigned int n_actions;
    char pipe_name[50];

    memset(&fwd, 0, sizeof fwd);
    memset(&match, 0, sizeof match);

    /* Create forward to post-ct. */
    post_ct_ctx = doca_pipe_group_ctx_ref(netdev, POSTCT_TABLE_ID, NULL);
    if (post_ct_ctx == NULL) {
        VLOG_ERR("%s: Failed to take a reference on post-ct table",
                 netdev_get_name(netdev));
        goto out;
    }
    fwd[CT_FWD_POST].type = DOCA_FLOW_FWD_PIPE;
    fwd[CT_FWD_POST].next_pipe = doca_pipe_group_get_pipe(post_ct_ctx);

    /* Create forward to miss handling. */
    fwd[CT_FWD_MISS].type = DOCA_FLOW_FWD_PIPE;
    fwd[CT_FWD_MISS].next_pipe = doca_offload_get_misspath_pipe(esw);

    port = doca_flow_port_switch_get(netdev_doca_port_get(netdev));

    n_actions = 0;
    for (int nw = 0; nw < NUM_CT_NW; nw++) {
        int nat_start;

        if (nw == CT_NW_IP6 && !conntrack_offload_doca_ct_ipv6_enabled) {
            continue;
        }

        /* For IPv4: create both plain CT (nat=0) and CT-NAT (nat=1)
         * For IPv6: only create CT-NAT (nat=1)
         */
        nat_start = (nw == CT_NW_IP6) ? 1 : 0;

        for (int nat = nat_start; nat < 2; nat++) {
            size_t idx;

            idx = CT_ACTIONS_IDX(nw, nat);
            ovs_assert(idx < ARRAY_SIZE(actions));
            /* Transport type is irrelevant for the DOCA-CT pipe. */
            build_ct_actions_templates(&actions[idx], &actions_masks[idx], nw, 0, nat);
            actions_arr[idx] = &actions[idx].d;
            actions_masks_arr[idx] = &actions_masks[idx].d;
            n_actions++;
        }
    }

    snprintf(pipe_name, sizeof pipe_name, "OVS_DOCA_CT_PIPE_%s", netdev_get_name(netdev));
    if (doca_flow_pipe_cfg_create(&cfg, port) ||
        doca_flow_pipe_cfg_set_name(cfg, pipe_name) ||
        doca_flow_pipe_cfg_set_match(cfg, &match.d, NULL) ||
        doca_flow_pipe_cfg_set_actions(cfg, actions_arr, actions_masks_arr, NULL, n_actions) ||
        doca_flow_pipe_cfg_set_type(cfg, DOCA_FLOW_PIPE_CT)) {
        VLOG_ERR("%s: Could not create DOCA-CT pipe configuration", netdev_get_name(netdev));
        goto out;
    }

    err = doca_flow_ct_fwd_register(port, NUM_CT_FWD, fwd, esw->ct.fwd_handles);
    if (DOCA_IS_ERROR(err)) {
        VLOG_ERR("%s: Failed to register CT fwd handles: %s",
                 netdev_get_name(netdev), doca_error_get_descr(err));
        goto out;
    }
    /* Next pipe will be determined per connection */
    fwd[CT_FWD_POST].next_pipe = NULL;

    err = doca_flow_pipe_create(cfg, &fwd[CT_FWD_POST], &fwd[CT_FWD_MISS], &ct_pipe);
    if (DOCA_IS_ERROR(err)) {
        VLOG_ERR("%s: Failed to create CT pipe: %s",
                 netdev_get_name(netdev), doca_error_get_descr(err));
        goto out;
    }

    esw->ct.pipe = ct_pipe;
    esw->ct.post_ct_ctx = post_ct_ctx;

out:
    if (DOCA_IS_ERROR(err)) {
        doca_pipe_group_ctx_unref(post_ct_ctx);
    }

    if (cfg) {
        doca_flow_pipe_cfg_destroy(cfg);
    }
    return -((int) DOCA_IS_ERROR(err));
}

static void
doca_esw_id_pool_init(struct doca_offload_esw_ctx *ctx)
{
    static struct ovsthread_once init_once = OVSTHREAD_ONCE_INITIALIZER;

    if (ovsthread_once_start(&init_once)) {
        esw_id_pool = id_pool_create(0, OVS_DOCA_MAX_SUPPORTED_ESW);
        ovsthread_once_done(&init_once);
    }
    if (!esw_id_pool || !id_pool_alloc_id(esw_id_pool, &ctx->esw_id)) {
        VLOG_ERR("Failed to alloc a new esw id");
        return;
    }
}

static int
shared_mtr_flow_id_alloc(struct doca_offload_esw_ctx *esw_ctx,
                         struct rte_flow_error *error)
{
    unsigned int tid = netdev_offload_thread_id();
    uint32_t id;

    if (!esw_ctx->shared_mtr_flow_id_pool) {
        if (error) {
            error->type = RTE_FLOW_ERROR_TYPE_UNSPECIFIED;
            error->message = "Could not allocate meter flow ID, id-pool is not initialized";
        }
        goto err;
    }

    if (!id_fpool_new_id(esw_ctx->shared_mtr_flow_id_pool, tid, &id)) {
        VLOG_ERR("Failed to alloc a new shared meter flow id");
        if (error) {
            error->type = RTE_FLOW_ERROR_TYPE_UNSPECIFIED;
            error->message = "Failed to alloc a new shared meter flow id";
        }
        goto err;
    }

    return id;
err:
    if (error) {
        ovs_assert(error->message);
        ovs_assert(error->type);
    }
    return -1;
}

static void
shared_mtr_flow_id_free(struct doca_offload_esw_ctx *esw_ctx, uint32_t id)
{
    unsigned int tid = netdev_offload_thread_id();

    id_fpool_free_id(esw_ctx->shared_mtr_flow_id_pool, tid, id);
}

static uint32_t
shared_mtr_n_ids(void)
{
    return ovs_doca_get_max_megaflows_counters() * DPDK_OFFLOAD_MAX_METERS_PER_FLOW;
}

static void
shared_mtr_flow_id_pool_init(struct doca_offload_esw_ctx *ctx)
{
    uint32_t base_id;

    base_id = shared_mtr_n_ids() * ctx->esw_id + MIN_SHARED_MTR_FLOW_ID;
    ctx->shared_mtr_flow_id_pool = id_fpool_create(MAX_OFFLOAD_THREAD_NB,
                                                   base_id, shared_mtr_n_ids());
}

static int
gpr_meter_id_get(struct doca_offload_esw_ctx *ctx, int base_id, uint32_t *id)
{
    int ret;

    ovs_assert(base_id < OVS_DOCA_MAX_GPR_METERS_PER_ESW);

    ret = doca_flow_port_shared_resource_get(ctx->esw_port, DOCA_FLOW_SHARED_RESOURCE_METER, id);
    if (ret) {
        VLOG_WARN("%s: Failed to get DOCA meter ID: %d (%s)", netdev_get_name(ctx->esw_netdev),
                  ret, doca_error_get_descr(ret));
        return -1;
    }
    ctx->shared_meters_ctx.gpr_meter_ids[base_id].valid = true;
    ctx->shared_meters_ctx.gpr_meter_ids[base_id].id = *id;

    return 0;
}

static int
sw_meter_id_get(struct doca_offload_esw_ctx *ctx, int base_id, uint32_t *id)
{
    struct doca_shared_meter_ctx *meter_ctx;
    struct doca_shared_meter_key key = {
        .of_meter_id = base_id
    };

    meter_ctx = refmap_find(ctx->shared_meters_ctx.rfm, &key);
    if (meter_ctx) {
        *id = meter_ctx->id;
        return 0;
    }

    return ENOENT;
}

static int
doca_post_meter_pipe_init(struct netdev *netdev,
                          struct doca_offload_esw_ctx *ctx)
{
    struct doca_pipe_group_ctx *group_ctx;

    group_ctx = doca_pipe_group_ctx_ref(netdev, POSTMETER_TABLE_ID, NULL);
    if (!group_ctx) {
        return -1;
    }
    ctx->post_meter_pipe_group_ctx = group_ctx;

    return 0;
}

static void
doca_post_meter_pipe_uninit(struct doca_offload_esw_ctx *ctx)
{
    doca_pipe_group_ctx_unref(ctx->post_meter_pipe_group_ctx);
}

static int
doca_sample_pipe_init(struct netdev *netdev, struct doca_offload_esw_ctx *ctx)
{
    struct doca_pipe_group_ctx *pipe_ctx;

    pipe_ctx = doca_pipe_group_ctx_ref(netdev, SAMPLE_TABLE_ID, NULL);
    if (!pipe_ctx) {
        return -1;
    }
    ctx->sample_ctx.sample_pipe_ctx = pipe_ctx;

    return 0;
}

static void
doca_sample_pipe_uninit(struct doca_offload_esw_ctx *ctx)
{
    doca_pipe_group_ctx_unref(ctx->sample_ctx.sample_pipe_ctx);
}

static int
doca_sample_mirror_pipe_init(struct netdev *netdev,
                             struct doca_offload_esw_ctx *ctx)
{
    struct doca_basic_pipe_ctx *pipe_ctx = &ctx->sample_ctx.mirror_pipe_ctx;
    struct ovs_doca_flow_actions odacts_mask;
    struct doca_flow_pipe_entry **pentry;
    struct ovs_doca_flow_actions odacts;
    struct ovs_doca_flow_match matchall;
    struct ovs_mirror_set mirror_set = {
        .nr_targets = 1,
        .target = (struct ovs_mirror_target[1]) {
            {
              .fwd.type = DOCA_FLOW_FWD_PIPE,
              .fwd.next_pipe = doca_pipe_group_get_pipe(ctx->sample_ctx.postmirror_pipe_ctx),
            },
         },
    };
    struct doca_flow_fwd fwd_pipe = {
        .type = DOCA_FLOW_FWD_PIPE,
        .next_pipe = NULL,
    };
    struct doca_flow_fwd fwd = {
        .type = DOCA_FLOW_FWD_PIPE,
        .next_pipe = doca_pipe_group_get_pipe(ctx->root_pipe_group_ctx),
    };
    int err;

    memset(&matchall, 0, sizeof matchall);
    memset(&odacts, 0, sizeof odacts);
    memset(&odacts_mask, 0, sizeof odacts_mask);

    doca_set_reg_val_mask(&odacts.d.meta, &odacts_mask.d.meta, REG_FIELD_FLOW_INFO,
                          NONSAMPLE_FLOW_MARK);
    ctx->sflow_mirror_ctx = doca_mirror_ctx_create(ctx->doca_mirrors, &mirror_set, &odacts,
                                                   &odacts_mask, &fwd);
    if (!ctx->sflow_mirror_ctx) {
        return -1;
    }

    err = doca_offload_basic_pipe_create(netdev, &matchall, NULL, NULL, &odacts, &odacts_mask,
                                         NULL, &fwd_pipe, NULL, 1, UINT64_C(1) << AUX_QUEUE,
                                         "SAMPLE_MIRROR", &pipe_ctx->pipe);
    if (err) {
        VLOG_ERR("%s: Failed to create sample mirror pipe. err=%d", netdev_get_name(netdev), err);
        goto err_pipe;
    }

    pentry = &ctx->sample_ctx.mirror_pipe_entry;
    err = doca_offload_add_entry(netdev, AUX_QUEUE, pipe_ctx->pipe, &matchall, &odacts, NULL,
                                 &fwd, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT, pentry);
    if (err) {
        VLOG_ERR("%s: Failed to add entry to sample mirror pipe. err=%d",
                 netdev_get_name(netdev), err);
        goto err_entry;
    }

    return 0;

err_entry:
    doca_flow_pipe_destroy(pipe_ctx->pipe);
    ctx->sample_ctx.mirror_pipe_ctx.pipe = NULL;
err_pipe:
    doca_mirror_ctx_destroy(ctx->sflow_mirror_ctx);
    ctx->sflow_mirror_ctx = NULL;
    return err;
}

static void
doca_sample_mirror_pipe_uninit(struct doca_offload_esw_ctx *ctx)
{
    if (ctx->sample_ctx.mirror_pipe_entry) {
        doca_offload_remove_entry(ctx, AUX_QUEUE, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                  &ctx->sample_ctx.mirror_pipe_entry);
        dpdk_offload_counter_dec(ctx->esw_netdev);
    }
    doca_pipe_group_pipe_destroy(ctx, ctx->sample_ctx.mirror_pipe_ctx.pipe);
    doca_mirror_ctx_destroy(ctx->sflow_mirror_ctx);
}

static int
doca_sample_post_mirror_pipe_init(struct netdev *netdev,
                                  struct doca_offload_esw_ctx *ctx)
{
    struct doca_pipe_group_mask_ctx **entry_mctx;
    struct ovs_doca_flow_actions act_mask;
    struct doca_pipe_group_ctx *pipe_ctx;
    struct ovs_doca_flow_match drop_spec;
    struct ovs_doca_flow_match matchall;
    struct doca_flow_pipe_entry *entry;
    struct doca_flow_fwd fwd_miss = {
        .type = DOCA_FLOW_FWD_PIPE,
        .next_pipe = NULL,
    };
    struct doca_flow_fwd fwd_drop = {
        .type = DOCA_FLOW_FWD_DROP,
    };
    struct ovs_doca_flow_actions act;
    struct rte_flow_error error;
    int i;

    memset(&drop_spec, 0, sizeof drop_spec);
    memset(&act_mask, 0, sizeof act_mask);
    memset(&matchall, 0, sizeof matchall);
    memset(&act, 0, sizeof act);

    pipe_ctx = doca_pipe_group_ctx_ref(netdev, SAMPLE_POSTMIRROR_TABLE_ID, NULL);
    if (!pipe_ctx) {
        return -1;
    }
    ctx->sample_ctx.postmirror_pipe_ctx = pipe_ctx;

    entry_mctx = &ctx->sample_ctx.postmirror_pipe_entry_mctx.mctx;
    fwd_miss.next_pipe = doca_offload_get_misspath_pipe(ctx);
    doca_set_reg_val_mask(&act.d.meta, &act_mask.d.meta, REG_FIELD_FLOW_INFO,
                          SAMPLE_FLOW_MARK);

    entry = create_doca_pipe_group_entry(netdev, AUX_QUEUE, pipe_ctx,
                                         DPDK_OFFLOAD_PRIORITY_LOW, &matchall,
                                         &matchall, &act, &act_mask, NULL, NULL,
                                         &fwd_miss, entry_mctx, &error);
    if (!entry) {
        VLOG_ERR("%s: Failed to create sample postmirror sflow entry: %s.",
                 netdev_get_name(netdev), error.message);
        goto err;
    }
    ctx->sample_ctx.postmirror_pipe_entry_mctx.entry = entry;

    /* LACP, LLDP and 802.1X packets should not be duplicated so if such packet
     * has been sampled it must be dropped.
     */
    drop_spec.d.parser_meta.outer_l2_type = DOCA_FLOW_L2_META_NO_VLAN;
    for (i = 0 ; i < NUM_SEND_TO_KERNEL ; i++) {
        drop_spec.d.outer.eth.type = htons(pre_miss_mapping[i]);
        entry_mctx = &ctx->sample_ctx.postmirror_drop[i].mctx;
        entry = create_doca_pipe_group_entry(netdev, AUX_QUEUE, pipe_ctx,
                                             DPDK_OFFLOAD_PRIORITY_HIGH, &drop_spec,
                                             &drop_spec, NULL, NULL, NULL, NULL,
                                             &fwd_drop, entry_mctx, &error);
        if (!entry) {
            VLOG_ERR("%s: Failed to create sample postmirror %x entry: %s.",
                     netdev_get_name(netdev), pre_miss_mapping[i], error.message);
            goto err_send2kernel;
        }
        ctx->sample_ctx.postmirror_drop[i].entry = entry;
    }

    doca_offload_complete_queue_esw(ctx, AUX_QUEUE, true);
    return 0;

err_send2kernel:
    for (; i >= 0 && i < NUM_SEND_TO_KERNEL; i--) {
        doca_offload_remove_entry(ctx, AUX_QUEUE, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                  &ctx->sample_ctx.postmirror_drop[i].entry);
        doca_pipe_group_mask_ctx_unref(ctx->sample_ctx.postmirror_drop[i].mctx);
        dpdk_offload_counter_dec(netdev);
    }
err:
    doca_pipe_group_ctx_unref(pipe_ctx);
    ctx->sample_ctx.postmirror_pipe_ctx = NULL;
    return -1;
}

static void
doca_sample_post_mirror_pipe_uninit(struct doca_offload_esw_ctx *ctx)
{
    if (ctx->sample_ctx.postmirror_pipe_entry_mctx.entry) {
        doca_offload_remove_entry(ctx, AUX_QUEUE, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                  &ctx->sample_ctx.postmirror_pipe_entry_mctx.entry);
        doca_pipe_group_mask_ctx_unref(ctx->sample_ctx.postmirror_pipe_entry_mctx.mctx);
    }

    for (int i = 0; i < NUM_SEND_TO_KERNEL; i++) {
        if (ctx->sample_ctx.postmirror_drop[i].entry) {
            doca_offload_remove_entry(ctx, AUX_QUEUE, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                      &ctx->sample_ctx.postmirror_drop[i].entry);
            doca_pipe_group_mask_ctx_unref(ctx->sample_ctx.postmirror_drop[i].mctx);
        }
    }

    doca_pipe_group_ctx_unref(ctx->sample_ctx.postmirror_pipe_ctx);
}

static void
doca_post_sw_meter_pipe_uninit(struct doca_offload_esw_ctx *ctx)
{
    doca_basic_pipe_ctx_uninit(ctx, &ctx->post_sw_meter_pipe_ctx);
}

static int
doca_post_sw_meter_pipe_init(struct netdev *netdev, struct doca_offload_esw_ctx *ctx)
{
    struct doca_basic_pipe_ctx *miss_pipe_ctx = &ctx->meta_transfer_ctx.meta_push_pipe_ctx;
    struct ovs_doca_flow_match match = { .d = {
        .parser_meta.port_id = UINT16_MAX,
        .parser_meta.meter_color = 0xff,
    }, };
    struct doca_flow_monitor monitor = {
        .counter_type = DOCA_FLOW_RESOURCE_TYPE_NON_SHARED,
    };
    struct doca_flow_fwd miss = {
        .type = DOCA_FLOW_FWD_PIPE,
        .next_pipe = miss_pipe_ctx->pipe,
    };
    struct doca_flow_fwd fwd = {
        .type = DOCA_FLOW_FWD_CHANGEABLE,
    };

    ovs_assert(ctx->post_sw_meter_pipe_ctx.pipe == NULL);

    return doca_offload_basic_pipe_create(netdev, &match, NULL, &monitor, NULL, NULL, NULL, &fwd,
                                          &miss, OVS_DOCA_MAX_SW_METERS_PER_ESW,
                                          UINT64_C(1) << AUX_QUEUE, "POST_SW_METER",
                                          &ctx->post_sw_meter_pipe_ctx.pipe);
}

static void
doca_sw_meter_pipe_uninit(struct doca_offload_esw_ctx *ctx)
{
    doca_basic_pipe_ctx_uninit(ctx, &ctx->sw_meter_pipe_ctx);
}

/* The sw-meter table is a basic pipe with a default miss decision
 * to forward to the miss-pipe. The miss-pipe, being empty, executes
 * the RSS config to send the packet to SW. */
static int
doca_sw_meter_pipe_init(struct netdev *netdev,
                        struct doca_offload_esw_ctx *ctx)
{
    struct doca_basic_pipe_ctx *miss_pipe_ctx = &ctx->meta_transfer_ctx.meta_push_pipe_ctx;
    struct doca_flow_monitor monitor = {
        .meter_type = DOCA_FLOW_RESOURCE_TYPE_SHARED,
        .shared_meter.shared_meter_id = UINT32_MAX,
    };
    struct ovs_doca_flow_match match = { .d = {
        .parser_meta.port_id = UINT16_MAX,
    }, };
    struct doca_flow_fwd miss = {
        .type = DOCA_FLOW_FWD_PIPE,
        .next_pipe = miss_pipe_ctx->pipe,
    };
    struct doca_flow_fwd fwd = {
        .type = DOCA_FLOW_FWD_PIPE,
        .next_pipe = ctx->post_sw_meter_pipe_ctx.pipe,
    };

    return doca_offload_basic_pipe_create(netdev, &match, NULL, &monitor, NULL, NULL, NULL, &fwd,
                                          &miss, OVS_DOCA_MAX_SW_METERS_PER_ESW,
                                          UINT64_C(1) << AUX_QUEUE, "POST_SW_METER",
                                          &ctx->sw_meter_pipe_ctx.pipe);
}

static void
doca_pre_miss_pipe_uninit(struct doca_offload_esw_ctx *ctx)
{
    doca_basic_pipe_ctx_uninit(ctx, &ctx->pre_miss_ctx.pre_miss_pipe_ctx);
}

static int
doca_pre_miss_pipe_init(struct netdev *netdev, struct doca_offload_esw_ctx *ctx)
{
    struct doca_basic_pipe_ctx *pre_miss_pipe_ctx, *sw_meter_pipe_ctx;
    struct doca_flow_target *kernel_target;
    struct doca_flow_fwd fwd, miss;
    struct ovs_doca_flow_match match = { .d = {
        .parser_meta.outer_l2_type = DOCA_FLOW_L2_META_NO_VLAN,
        .outer.eth.type = UINT16_MAX,
    }, };

    memset(&miss, 0, sizeof miss);
    memset(&fwd, 0, sizeof fwd);

    pre_miss_pipe_ctx = &ctx->pre_miss_ctx.pre_miss_pipe_ctx;

    sw_meter_pipe_ctx = &ctx->sw_meter_pipe_ctx;
    miss.type = DOCA_FLOW_FWD_PIPE;
    miss.next_pipe = ovs_doca_gpr_mode() != OVS_DOCA_GPR_MODE_DISABLED ?
                     ctx->port_meter_pipe_ctx.pipe : sw_meter_pipe_ctx->pipe;

    if (doca_flow_get_target(DOCA_FLOW_TARGET_KERNEL, &kernel_target)) {
        VLOG_ERR("%s: Could not get miss to kernel target",
                 netdev_get_name(netdev));
        return -1;
    }
    fwd.type = DOCA_FLOW_FWD_TARGET;
    fwd.target = kernel_target;

    return doca_offload_basic_pipe_create(netdev, &match, NULL, NULL, NULL, NULL, NULL, &fwd,
                                          &miss, OVS_DOCA_MAX_PRE_MISS_RULES,
                                          UINT64_C(1) << AUX_QUEUE, "PRE_MISS",
                                          &pre_miss_pipe_ctx->pipe);
}

static int
doca_pre_miss_rule_add(struct netdev *netdev,
                       struct doca_offload_esw_ctx *ctx,
                       uint16_t eth_type,
                       struct doca_pre_miss_rule *rule)
{
    struct ovs_doca_flow_match match;
    int ret;

    memset(&match, 0, sizeof match);
    match.d.outer.eth.type = htons(eth_type);
    ret = doca_offload_add_entry(netdev, AUX_QUEUE, ctx->pre_miss_ctx.pre_miss_pipe_ctx.pipe,
                                 &match, NULL, NULL, NULL, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                 &rule->entry);
    if (ret) {
        VLOG_ERR("%s: Failed to create pre_miss 0x%"PRIx16" rule",
                 netdev_get_name(netdev), eth_type);
        return -1;
    }
    VLOG_INFO("%s: Created pre_miss 0x%"PRIx16" rule", netdev_get_name(netdev), eth_type);
    rule->eth_type = eth_type;
    return 0;
}

static void
doca_pre_miss_rules_uninit(struct doca_offload_esw_ctx *ctx)
{
    for (int i = 0; i < OVS_DOCA_MAX_PRE_MISS_RULES; i++) {
        if (!ctx->pre_miss_ctx.pre_miss_rules[i].entry) {
            continue;
        }
        doca_offload_remove_entry(ctx, AUX_QUEUE, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                  &ctx->pre_miss_ctx.pre_miss_rules[i].entry);
        dpdk_offload_counter_dec(ctx->esw_netdev);
    }
}

static int
doca_pre_miss_rules_init(struct netdev *netdev, struct doca_offload_esw_ctx *ctx)
{
    for (int i = 0 ; i < NUM_SEND_TO_KERNEL ; i++) {
        if (doca_pre_miss_rule_add(netdev, ctx, pre_miss_mapping[i],
                                   &ctx->pre_miss_ctx.pre_miss_rules[i])) {
            doca_pre_miss_rules_uninit(ctx);
            return -1;
        }
    }
    return 0;
}

static bool
doca_pre_miss_rules_match(struct doca_offload_esw_ctx *ctx,
                          uint16_t *eth_types,
                          size_t n_eth_types)
{
    size_t existing_count = 0;
    size_t i, j;
    bool found;

    /* Count existing rules */
    for (i = 0; i < OVS_DOCA_MAX_PRE_MISS_RULES; i++) {
        if (ctx->pre_miss_ctx.pre_miss_rules[i].entry) {
            existing_count++;
        }
    }

    /* Different counts = not matching */
    if (existing_count != n_eth_types) {
        return false;
    }

    /* Both empty = match */
    if (n_eth_types == 0) {
        return true;
    }

    /* Verify each new rule exists in existing rules */
    for (i = 0; i < n_eth_types; i++) {
        found = false;
        for (j = 0; j < OVS_DOCA_MAX_PRE_MISS_RULES; j++) {
            if (ctx->pre_miss_ctx.pre_miss_rules[j].entry &&
                ctx->pre_miss_ctx.pre_miss_rules[j].eth_type == eth_types[i]) {
                found = true;
                break;
            }
        }
        if (!found) {
            return false;
        }
    }

    return true;
}

static int
doca_offload_pre_miss_rules_update__(struct netdev *netdev,
                                     uint16_t *eth_types,
                                     size_t n_eth_types)
{
    struct doca_offload_esw_ctx *ctx = doca_offload_esw_ctx_get(netdev);

    /* Check if the new rules are identical to existing ones */
    if (doca_pre_miss_rules_match(ctx, eth_types, n_eth_types)) {
        return 0;
    }

    /* Remove all existing rules before adding new ones */
    doca_pre_miss_rules_uninit(ctx);

    for (size_t i = 0; i < n_eth_types; i++) {
        if (doca_pre_miss_rule_add(netdev, ctx, eth_types[i],
                                   &ctx->pre_miss_ctx.pre_miss_rules[i])) {
            VLOG_ERR("%s: Failed to create pre_miss rule %"PRIx16,
                     netdev_get_name(netdev), eth_types[i]);
            return -1;
        }
    }
    return 0;

}

static int
parse_pre_miss_ethertypes(struct netdev *netdev,
                          const char *pre_miss_rules,
                          uint16_t *eth_types,
                          size_t *n_types)
{
    char *str_copy = xstrdup(pre_miss_rules);
    char *token, *saveptr = NULL;
    int ret = 0;

    *n_types = 0;
    for (token = strtok_r(str_copy, ",", &saveptr);
         token != NULL;
         token = strtok_r(NULL, ",", &saveptr)) {
        unsigned int parsed_val;

        /* Check if we've reached the maximum number of rules */
        if (*n_types >= OVS_DOCA_MAX_PRE_MISS_RULES) {
            VLOG_ERR("%s: Too many doca-pre-miss rules requested, max is %d",
                     netdev_get_name(netdev), OVS_DOCA_MAX_PRE_MISS_RULES);
            ret = -1;
            goto out;
        }

        /* Trim leading and trailing white-space */
        token += strspn(token, " \t\r\n");

        /* Parse hex value (with or without 0x prefix) */
        if (!str_to_uint(token, 0, &parsed_val) ||
            parsed_val > UINT16_MAX) {
            VLOG_WARN("%s: Invalid ethertype value '%s' in doca-pre-miss-rules, ignoring list",
                      netdev_get_name(netdev), token);
            ret = -1;
            goto out;
        }

        eth_types[(*n_types)++] = (uint16_t) parsed_val;
    }
out:
    free(str_copy);
    return ret;
}

int
doca_offload_pre_miss_rules_update(struct netdev *netdev, const char *pre_miss_rules)
{
    struct doca_offload_esw_ctx *ctx = doca_offload_esw_ctx_get(netdev);
    uint16_t eth_types[OVS_DOCA_MAX_PRE_MISS_RULES];
    size_t n_eth_types = 0;

    /* If no rules specified, and the default rules are not already set, insert them. */
    if (!pre_miss_rules) {
        if (!doca_pre_miss_rules_match(ctx, pre_miss_mapping, NUM_SEND_TO_KERNEL)) {
            doca_pre_miss_rules_uninit(ctx);
            VLOG_INFO("%s: Cleared doca-pre-miss-rules", netdev_get_name(netdev));
            if (doca_pre_miss_rules_init(netdev, ctx)) {
                VLOG_ERR("%s: Failed to insert default pre-miss rules", netdev_get_name(netdev));
                return -1;
            }
        }
        return 0;
    }

    /* If rules contain only white-space, clear them. */
    if (pre_miss_rules[strspn(pre_miss_rules, " \t\r\n")] == '\0') {
        doca_pre_miss_rules_uninit(ctx);
        VLOG_INFO("%s: Cleared doca-pre-miss-rules", netdev_get_name(netdev));
        return 0;
    }

    if (parse_pre_miss_ethertypes(netdev, pre_miss_rules, eth_types, &n_eth_types)) {
        return 0;  /* Not fatal, silently ignore */
    }

    if (doca_offload_pre_miss_rules_update__(netdev, eth_types, n_eth_types)) {
        VLOG_ERR("%s: Failed to update pre-miss rules", netdev_get_name(netdev));
        return -1;
    }
    return 0;
}

static void
doca_meta_push_pipe_uninit(struct doca_offload_esw_ctx *ctx)
{
    doca_basic_pipe_ctx_uninit(ctx, &ctx->meta_transfer_ctx.meta_push_pipe_ctx);
}

/* The meta-push table functions as a basic pipe, employing a catch-all rule
 * for encapsulating metadata before forwarding it to the miss-pipe.
 * The miss-pipe, being empty, sends the packet to SW.
 * Metadata encapsulation is accomplished through both the meta-push
 * and meta-copy pipes, where the former adds a header, and the latter coppies
 * the data into the header.
 */
static int
doca_meta_push_pipe_init(struct netdev *netdev, struct doca_offload_esw_ctx *ctx)
{
    struct ovs_doca_flow_actions actions_masks;
    struct ovs_doca_flow_actions actions;
    struct doca_basic_pipe_ctx *pipe_ctx;
    struct ovs_doca_flow_match match;
    struct doca_flow_fwd fwd = {
        .type = DOCA_FLOW_FWD_PIPE,
        .next_pipe = ctx->meta_transfer_ctx.meta_copy_pipe_ctx.pipe,
    };

    ovs_assert(ctx->meta_transfer_ctx.meta_push_pipe_ctx.pipe == NULL);

    memset(&match, 0, sizeof match);
    memset(&actions, 0, sizeof actions);
    memset(&actions_masks, 0, sizeof actions_masks);

    /* Build basic outer VXLAN encap data */
    actions.d.encap_type = DOCA_FLOW_RESOURCE_TYPE_NON_SHARED;
    actions.d.encap_cfg.is_l2 = true;
    actions.d.encap_cfg.encap.outer.l3_type = DOCA_FLOW_L3_TYPE_IP4;
    memset(&actions.d.encap_cfg.encap.outer.ip4.src_ip, 0xFF,
           sizeof actions.d.encap_cfg.encap.outer.ip4.src_ip);
    memset(&actions.d.encap_cfg.encap.outer.ip4.dst_ip, 0xFF,
           sizeof actions.d.encap_cfg.encap.outer.ip4.dst_ip);
    memset(&actions.d.encap_cfg.encap.outer.eth.src_mac, 0xFF,
           sizeof actions.d.encap_cfg.encap.outer.eth.src_mac);
    memset(&actions.d.encap_cfg.encap.outer.eth.dst_mac, 0xFF,
           sizeof actions.d.encap_cfg.encap.outer.eth.dst_mac);
    memset(&actions.d.encap_cfg.encap.outer.ip4.ttl, 0xFF,
           sizeof actions.d.encap_cfg.encap.outer.ip4.ttl);
    memset(&actions.d.encap_cfg.encap.outer.udp.l4_port.dst_port, 0xFF,
           sizeof actions.d.encap_cfg.encap.outer.udp.l4_port.dst_port);
    memset(&actions.d.encap_cfg.encap.tun.vxlan_tun_id, 0xFF,
           sizeof actions.d.encap_cfg.encap.tun.vxlan_tun_id);
    actions.d.encap_cfg.encap.outer.l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_UDP;
    actions.d.encap_cfg.encap.tun.type = DOCA_FLOW_TUN_VXLAN;

    pipe_ctx = &ctx->meta_transfer_ctx.meta_push_pipe_ctx;
    return doca_offload_basic_pipe_create(netdev, &match, NULL, NULL, &actions, &actions_masks,
                                          NULL, &fwd, NULL, 1, UINT64_C(1) << AUX_QUEUE,
                                          "META_PUSH", &pipe_ctx->pipe);
}

static void
doca_meta_tag0_pipe_uninit(struct doca_offload_esw_ctx *ctx)
{
    doca_basic_pipe_ctx_uninit(ctx, &ctx->meta_transfer_ctx.meta_tag0_pipe_ctx);
}

static void
doca_meta_copy_pipe_uninit(struct doca_offload_esw_ctx *ctx)
{
    doca_basic_pipe_ctx_uninit(ctx, &ctx->meta_transfer_ctx.meta_copy_pipe_ctx);
}

static int
doca_meta_push_rule_init(struct netdev *netdev, struct doca_offload_esw_ctx *ctx)
{
    struct doca_meta_transfer_ctx *meta_trans_ctx;
    struct doca_flow_pipe_entry **pentry;
    struct ovs_doca_flow_actions dacts;
    struct doca_flow_fwd fwd;
    int ret = 0;

    memset(&dacts, 0, sizeof dacts);

    /* Eth */
    memset(&dacts.d.encap_cfg.encap.outer.eth.src_mac, 0,
           sizeof dacts.d.encap_cfg.encap.outer.eth.src_mac);
    dacts.d.encap_cfg.encap.outer.eth.type = (OVS_FORCE doca_be16_t) htons(ETH_TYPE_IP);
    /* IP */
    dacts.d.encap_cfg.encap.outer.l3_type = DOCA_FLOW_L3_TYPE_IP4;
    dacts.d.encap_cfg.encap.outer.ip4.next_proto = IPPROTO_UDP;
    /* UDP */
    dacts.d.encap_cfg.encap.outer.l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_UDP;
    dacts.d.encap_cfg.encap.outer.udp.l4_port.dst_port =
        (OVS_FORCE doca_be16_t) htons(META_PUSH_VXLAN_UDP_PORT);
    /* VXLAN */
    dacts.d.encap_cfg.encap.tun.type = DOCA_FLOW_TUN_VXLAN;
    dacts.d.encap_cfg.encap.tun.vxlan_tun_id = (OVS_FORCE doca_be32_t) htonl(META_PUSH_VNI << 8);

    fwd.type = DOCA_FLOW_FWD_PIPE;
    fwd.next_pipe = ctx->meta_transfer_ctx.meta_copy_pipe_ctx.pipe;

    pentry = &ctx->meta_transfer_ctx.meta_push_hdr;
    meta_trans_ctx = &ctx->meta_transfer_ctx;

    ret = doca_offload_add_entry(netdev, AUX_QUEUE, meta_trans_ctx->meta_push_pipe_ctx.pipe, NULL,
                                 &dacts, NULL, &fwd, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT, pentry);
    if (ret) {
        VLOG_ERR("%s: Failed to create meta-push rule", netdev_get_name(netdev));
    }

    return ret;
}

static void
doca_meta_push_rule_uninit(struct netdev *netdev)
{
    struct doca_offload_esw_ctx *esw = doca_offload_esw_ctx_get(netdev);

    if (!esw->meta_transfer_ctx.meta_push_hdr) {
        return;
    }

    doca_offload_remove_entry(esw, AUX_QUEUE, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                              &esw->meta_transfer_ctx.meta_push_hdr);
    dpdk_offload_counter_dec(netdev);
}

static int
doca_meta_copy_rule_init(struct netdev *netdev, struct doca_offload_esw_ctx *ctx)
{
    struct doca_meta_transfer_ctx *meta_trans_ctx;
    struct doca_flow_pipe_entry **pentry;
    int ret = 0;

    pentry = &ctx->meta_transfer_ctx.meta_copy_to_hdr;
    meta_trans_ctx = &ctx->meta_transfer_ctx;

    ret = doca_offload_add_entry(netdev, AUX_QUEUE, meta_trans_ctx->meta_copy_pipe_ctx.pipe, NULL,
                                 NULL, NULL, NULL, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT, pentry);
    if (ret) {
        VLOG_ERR("%s: Failed to create meta-copy rule", netdev_get_name(netdev));
    }

    return ret;
}

static void
doca_meta_copy_rule_uninit(struct netdev *netdev)
{
    struct doca_offload_esw_ctx *esw = doca_offload_esw_ctx_get(netdev);

    if (!esw->meta_transfer_ctx.meta_copy_to_hdr) {
        return;
    }

    doca_offload_remove_entry(esw, AUX_QUEUE, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                              &esw->meta_transfer_ctx.meta_copy_to_hdr);
    dpdk_offload_counter_dec(netdev);
}

static int
doca_meta_tag0_rule_init(struct netdev *netdev, struct doca_offload_esw_ctx *ctx)
{
    struct doca_flow_pipe_entry **pentry;
    struct doca_flow_pipe *pipe;
    int ret;

    pentry = &ctx->meta_transfer_ctx.meta_tag0_hdr;
    pipe = ctx->meta_transfer_ctx.meta_tag0_pipe_ctx.pipe;

    ret = doca_offload_add_entry(netdev, AUX_QUEUE, pipe, NULL,
                                 NULL, NULL, NULL, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT, pentry);
    if (ret) {
        VLOG_ERR("%s: Failed to create meta-tag0 rule", netdev_get_name(netdev));
    }

    return ret;
}

static void
doca_meta_tag0_rule_uninit(struct netdev *netdev)
{
    struct doca_offload_esw_ctx *esw = doca_offload_esw_ctx_get(netdev);

    if (!esw->meta_transfer_ctx.meta_tag0_hdr) {
        return;
    }

    doca_offload_remove_entry(esw, AUX_QUEUE, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                              &esw->meta_transfer_ctx.meta_tag0_hdr);
    dpdk_offload_counter_dec(netdev);
}

static void
doca_meta_transfer_uninit(struct doca_offload_esw_ctx *ctx)
{
    doca_meta_push_rule_uninit(ctx->esw_netdev);
    doca_meta_push_pipe_uninit(ctx);
    doca_meta_copy_rule_uninit(ctx->esw_netdev);
    doca_meta_copy_pipe_uninit(ctx);
    doca_meta_tag0_rule_uninit(ctx->esw_netdev);
    doca_meta_tag0_pipe_uninit(ctx);
}

static int
doca_meta_copy_pipe_init__(struct netdev *netdev, struct doca_offload_esw_ctx *ctx,
                           bool nv_mp)
{
    const uint32_t n_descs = sizeof(struct reg_tags) / sizeof(uint32_t);
    struct ovs_doca_flow_actions actions, actions_masks;
    struct doca_flow_action_desc desc_array[n_descs];
    struct doca_basic_pipe_ctx *pipe_ctx;
    struct doca_flow_action_descs descs;
    struct ovs_doca_flow_match match;
    struct doca_flow_fwd fwd = {
        .type = DOCA_FLOW_FWD_PIPE,
        .next_pipe = ctx->meta_transfer_ctx.meta_tag0_pipe_ctx.pipe,
    };
    int desc_ind = 0;

    ovs_assert(ctx->meta_transfer_ctx.meta_copy_pipe_ctx.pipe == NULL);

    memset(&match, 0, sizeof match);
    memset(&actions, 0, sizeof actions);
    memset(&actions_masks, 0, sizeof actions_masks);
    memset(&descs, 0, sizeof descs);

    descs.desc_array = desc_array;

    /* Copy REG_FIELD_CT_STATE */
    desc_array[desc_ind].type = DOCA_FLOW_ACTION_COPY;
    desc_array[desc_ind].field_op.src.field_string = "meta.data";
    desc_array[desc_ind].field_op.src.bit_offset = doca_get_reg_bit_offset(REG_FIELD_CT_STATE);
    desc_array[desc_ind].field_op.dst.field_string = "outer.eth.dst_mac";
    desc_array[desc_ind].field_op.dst.bit_offset =
        REV_BIT_OFFSETOF(union meta_eth_dst, reg_field_ct_state);
    desc_array[desc_ind].field_op.width = doca_get_reg_width(REG_FIELD_CT_STATE);
    desc_ind++;

    /* Copy REG_FIELD_CT_ZONE */
    desc_array[desc_ind].type = DOCA_FLOW_ACTION_COPY;
    desc_array[desc_ind].field_op.src.field_string = "meta.data";
    desc_array[desc_ind].field_op.src.bit_offset = doca_get_reg_bit_offset(REG_FIELD_CT_ZONE);
    desc_array[desc_ind].field_op.dst.field_string = "outer.eth.dst_mac";
    desc_array[desc_ind].field_op.dst.bit_offset =
        REV_BIT_OFFSETOF(union meta_eth_dst, reg_field_ct_zone);
    desc_array[desc_ind].field_op.width = doca_get_reg_width(REG_FIELD_CT_ZONE);
    desc_ind++;

    /* Copy REG_FIELD_CT_MARK */
    desc_array[desc_ind].type = DOCA_FLOW_ACTION_COPY;
    desc_array[desc_ind].field_op.src.field_string = "meta.data";
    desc_array[desc_ind].field_op.src.bit_offset = doca_get_reg_bit_offset(REG_FIELD_CT_MARK);
    desc_array[desc_ind].field_op.dst.field_string = "outer.ipv4.dst_ip";
    desc_array[desc_ind].field_op.dst.bit_offset = 0;
    desc_array[desc_ind].field_op.width = doca_get_reg_width(REG_FIELD_CT_MARK);
    desc_ind++;

    /* Copy REG_FIELD_CT_LABEL_ID */
    desc_array[desc_ind].type = DOCA_FLOW_ACTION_COPY;
    desc_array[desc_ind].field_op.src.field_string = "meta.data";
    desc_array[desc_ind].field_op.src.bit_offset = doca_get_reg_bit_offset(REG_FIELD_CT_LABEL_ID);
    desc_array[desc_ind].field_op.dst.field_string = "outer.ipv4.src_ip";
    desc_array[desc_ind].field_op.dst.bit_offset = 0;
    desc_array[desc_ind].field_op.width = doca_get_reg_width(REG_FIELD_CT_LABEL_ID);
    desc_ind++;

    if (nv_mp) {
        /* Copy nv_mp */
        desc_array[desc_ind].type = DOCA_FLOW_ACTION_COPY;
        desc_array[desc_ind].field_op.src.field_string = "meta.plb_port_id";
        desc_array[desc_ind].field_op.src.bit_offset = 0;
        desc_array[desc_ind].field_op.dst.field_string = "outer.eth.src_mac";
        desc_array[desc_ind].field_op.dst.bit_offset =
            REV_BIT_OFFSETOF(union meta_eth_src, nv_mp.pid);
        desc_array[desc_ind].field_op.width = 3;
        desc_ind++;
        desc_array[desc_ind].type = DOCA_FLOW_ACTION_COPY;
        desc_array[desc_ind].field_op.src.field_string = "meta.plb_port_preferred";
        desc_array[desc_ind].field_op.src.bit_offset = 0;
        desc_array[desc_ind].field_op.dst.field_string = "outer.eth.src_mac";
        desc_array[desc_ind].field_op.dst.bit_offset =
            REV_BIT_OFFSETOF(union meta_eth_src, nv_mp.preferred);
        desc_array[desc_ind].field_op.width = 1;
        desc_ind++;
        desc_array[desc_ind].type = DOCA_FLOW_ACTION_COPY;
        desc_array[desc_ind].field_op.src.field_string = "meta.plb_port_strict";
        desc_array[desc_ind].field_op.src.bit_offset = 0;
        desc_array[desc_ind].field_op.dst.field_string = "outer.eth.src_mac";
        desc_array[desc_ind].field_op.dst.bit_offset =
            REV_BIT_OFFSETOF(union meta_eth_src, nv_mp.strict);
        desc_array[desc_ind].field_op.width = 1;
        desc_ind++;
        ovs_assert(desc_ind == n_descs);
    } else {
        ovs_assert(desc_ind + 3 == n_descs);
    }

    descs.nb_action_desc = desc_ind;

    pipe_ctx = &ctx->meta_transfer_ctx.meta_copy_pipe_ctx;
    return doca_offload_basic_pipe_create(netdev, &match, NULL, NULL, &actions, &actions_masks,
                                          &descs, &fwd, NULL, 1, UINT64_C(1) << AUX_QUEUE,
                                          "META_COPY", &pipe_ctx->pipe);
}

static int
doca_meta_copy_pipe_init(struct netdev *netdev, struct doca_offload_esw_ctx *ctx)
{
    struct doca_flow_pipe *jumbo;
    bool is_bf3;

    jumbo = ctx->hash_pipe_ctx->hashes[HASH_TYPE_IPV6_L3_JUMBO].pipe;
    if (jumbo) {
        if (netdev_doca_is_bf3(netdev, &is_bf3)) {
            VLOG_ERR("%s: Failed checking BF3 device type", netdev_get_name(netdev));
            return -1;
        }
        return doca_meta_copy_pipe_init__(netdev, ctx, !is_bf3);
    }
    return doca_meta_copy_pipe_init__(netdev, ctx, false);
}

static int
doca_meta_tag0_pipe_init(struct netdev *netdev, struct doca_offload_esw_ctx *ctx)
{
    struct ovs_doca_flow_actions actions_masks;
    struct ovs_doca_flow_actions actions;
    struct doca_basic_pipe_ctx *pipe_ctx;
    struct ovs_doca_flow_match match;
    struct doca_flow_fwd fwd = {
        .type = DOCA_FLOW_FWD_PIPE,
        .next_pipe = netdev_doca_ovs_doca_esw_ctx(netdev)->rss_pipe,
    };

    ovs_assert(ctx->meta_transfer_ctx.meta_tag0_pipe_ctx.pipe == NULL);

    memset(&match, 0, sizeof match);
    memset(&actions, 0, sizeof actions);
    memset(&actions_masks, 0, sizeof actions_masks);

    doca_set_reg_val_mask(&actions.d.meta, &actions_masks.d.meta, REG_FIELD_TAG0, 0);

    pipe_ctx = &ctx->meta_transfer_ctx.meta_tag0_pipe_ctx;
    return doca_offload_basic_pipe_create(netdev, &match, NULL, NULL, &actions, &actions_masks,
                                          NULL, &fwd, NULL, 1, UINT64_C(1) << AUX_QUEUE,
                                          "META_TAG0", &pipe_ctx->pipe);
}

static int
doca_meta_transfer_init(struct netdev *netdev, struct doca_offload_esw_ctx *ctx)
{
    if (doca_meta_tag0_pipe_init(netdev, ctx)) {
        return -1;
    }
    if (doca_meta_tag0_rule_init(netdev, ctx)) {
        return -1;
    }

    if (doca_meta_copy_pipe_init(netdev, ctx)) {
        return -1;
    }
    if (doca_meta_copy_rule_init(netdev, ctx)) {
        return -1;
    }

    if (doca_meta_push_pipe_init(netdev, ctx)) {
        return -1;
    }
    if (doca_meta_push_rule_init(netdev, ctx)) {
        return -1;
    }

    return 0;
}

static bool
get_n_active_ports_cb(struct netdev *netdev, odp_port_t port_no OVS_UNUSED, void *aux_)
{
    uint32_t *count = aux_;

    if (!netdev_doca_is_ethdev(netdev)) {
        return false;
    }

    (*count)++;
    return false;
}

static uint32_t
doca_offload_n_active_ports_get(void)
{
    uint32_t count = 0;

    netdev_ports_traverse("doca", get_n_active_ports_cb, &count);
    return count;
}

static int
doca_gpr_meter_config(struct doca_flow_port *port, uint32_t doca_mtr_id, uint32_t rate,
                      uint32_t burst_size)
{
    struct doca_flow_shared_resource_cfg meter_cfg = {
        .meter_cfg = {
            .limit_type = DOCA_FLOW_METER_LIMIT_TYPE_PACKETS,
            .cir = rate,
            .cbs = burst_size,
        },
    };
    int ret = 0;

    ret = doca_flow_port_shared_resource_set_cfg(port, DOCA_FLOW_SHARED_RESOURCE_METER, doca_mtr_id,
                                                 &meter_cfg);
    if (ret != DOCA_SUCCESS) {
        VLOG_ERR("Failed to configure core shared meter id %d  err %d - %s",
                 doca_mtr_id, ret, doca_error_get_descr(ret));
        return -1;
    }

    return 0;
}

struct gpr_meter_rate_set_cb_arg {
    uint32_t core_rate;
    uint32_t port_rate;
};

static void
doca_offload_gpr_meter_rate_set_cb(void *ctx_, void *key_ OVS_UNUSED, void *arg_)
{
    struct gpr_meter_rate_set_cb_arg *arg = arg_;
    struct doca_offload_esw_ctx *ctx = ctx_;
    uint32_t doca_mtr_id;
    int i;

    for (i = 0; i < OVS_DOCA_MAX_PORT_METERS_PER_ESW; i++) {
        /* Update port meter rate without considering GPR mode. If GPR mode is rep-only
           then the port meter rule of the eswitch is deleted from HW so it's safe to
           update the rate.
        */
        if (gpr_meter_id_get(ctx, i, &doca_mtr_id) ||
            doca_gpr_meter_config(ctx->esw_port, doca_mtr_id, arg->port_rate,
                                  arg->port_rate * 0.1)) {
            VLOG_ERR("%s: Failed to update port meter for eswitch %d",
                     netdev_get_name(ctx->esw_netdev), ctx->esw_id);
        }
    }
    /* Update core meter rate */
    if (gpr_meter_id_get(ctx, OVS_DOCA_MAX_PORT_METERS_PER_ESW, &doca_mtr_id) ||
        doca_gpr_meter_config(ctx->esw_port, doca_mtr_id, arg->core_rate, arg->core_rate * 0.1)) {
        VLOG_ERR("%s: Failed to update core meter for eswitch %d", netdev_get_name(ctx->esw_netdev),
                 ctx->esw_id);
    }
}

static struct refmap *doca_eswitch_rfm = NULL;

void
doca_offload_gpr_meter_rate_modify(uint32_t req_core_rate,
                                   uint32_t req_port_rate)
{
    uint32_t n_active_ports = doca_offload_n_active_ports_get();
    struct gpr_meter_rate_set_cb_arg set_cb_arg;

    if (OVS_UNLIKELY(!n_active_ports)) {
        n_active_ports = 1;
    }

    set_cb_arg.core_rate = req_core_rate * dpif_doca_get_n_pmd_threads();
    if (ovs_doca_port_rate_configured()) {
        set_cb_arg.port_rate = req_port_rate;
    } else {
        set_cb_arg.port_rate = set_cb_arg.core_rate / n_active_ports;
    }

    if (doca_eswitch_rfm) {
        refmap_for_each(doca_eswitch_rfm, doca_offload_gpr_meter_rate_set_cb, &set_cb_arg);
    }
}

static int
doca_port_meter_pipe_init(struct netdev *netdev, struct doca_offload_esw_ctx *ctx)
{
    struct doca_flow_fwd fwd = {
        .type = DOCA_FLOW_FWD_PIPE,
        .next_pipe = ctx->post_port_meter_pipe_ctx.pipe,
    };
    struct doca_flow_fwd fwd_miss = {
        .type = DOCA_FLOW_FWD_PIPE,
        .next_pipe = ctx->sw_meter_pipe_ctx.pipe,
    };
    struct doca_flow_monitor monitor = {
        .meter_type = DOCA_FLOW_RESOURCE_TYPE_SHARED,
        .shared_meter.shared_meter_id = UINT32_MAX,
    };
    struct ovs_doca_flow_match match = { .d = {
        .parser_meta.port_id = UINT16_MAX,
    }, };

    if (doca_offload_basic_pipe_create(netdev, &match, NULL, &monitor, NULL, NULL, NULL, &fwd,
                                       &fwd_miss, OVS_DOCA_MAX_PORT_METERS_PER_ESW,
                                       UINT64_C(1) << AUX_QUEUE, "PORT_METER",
                                       &ctx->port_meter_pipe_ctx.pipe)) {
        return -1;
    }

    return 0;
}

static int
doca_post_port_meter_pipe_init(struct netdev *netdev, struct doca_offload_esw_ctx *ctx)
{
    struct ovs_doca_flow_match match = { .d = {
        .parser_meta.meter_color = 0xff,
    }, };
    struct ovs_doca_flow_actions act;
    struct doca_flow_fwd fwd = {
        .type = DOCA_FLOW_FWD_PIPE,
        .next_pipe = ctx->core_meter_pipe_ctx.pipe,
    };

    memset(&act, 0, sizeof act);
    doca_set_reg_val(&act.d.meta, REG_FIELD_PORT_METER_COLOR, UINT32_MAX);

    return doca_offload_basic_pipe_create(netdev, &match, NULL, NULL, &act, NULL, NULL, &fwd, NULL,
                                          NUM_OVS_PORT_METER_COLOR, UINT64_C(1) << AUX_QUEUE,
                                          "POST_PORT_METER", &ctx->post_port_meter_pipe_ctx.pipe);
}

static int
doca_core_meter_pipe_init(struct netdev *netdev, struct doca_offload_esw_ctx *ctx)
{
    uint32_t core_rate = ovs_doca_per_core_meter_rate();
    struct doca_flow_monitor monitor = {
        .meter_type = DOCA_FLOW_RESOURCE_TYPE_SHARED,
    };
    struct ovs_doca_flow_match match;
    struct doca_flow_fwd fwd = {
        .type = DOCA_FLOW_FWD_PIPE,
        .next_pipe = ctx->post_core_meter_pipe_ctx.pipe,
    };

    if (gpr_meter_id_get(ctx, OVS_DOCA_MAX_PORT_METERS_PER_ESW,
                         &monitor.shared_meter.shared_meter_id)) {
        VLOG_ERR("%s: Failed to get core meter ID for eswitch %d",
                 netdev_get_name(netdev), ctx->esw_id);
        return -1;
    }

    memset(&match, 0, sizeof match);

    core_rate = core_rate * dpif_doca_get_n_pmd_threads();
    if (doca_gpr_meter_config(ctx->esw_port, monitor.shared_meter.shared_meter_id, core_rate,
                              core_rate * 0.1)) {
        VLOG_ERR("%s: Failed to configure core meter for eswitch %d",
                 netdev_get_name(netdev), ctx->esw_id);
        return -1;
    }

    return doca_offload_basic_pipe_create(netdev, &match, NULL, &monitor, NULL, NULL, NULL, &fwd,
                                          NULL, 1, UINT64_C(1) << AUX_QUEUE, "CORE_METER",
                                          &ctx->core_meter_pipe_ctx.pipe);
}

static int
doca_port_meter_rule_init(struct netdev *netdev, struct doca_offload_esw_ctx *ctx)
{
    uint32_t n_active_ports = doca_offload_n_active_ports_get();
    int port_id = netdev_doca_get_port_id(netdev);
    struct doca_flow_monitor monitor = {
        .meter_type = DOCA_FLOW_RESOURCE_TYPE_SHARED,
    };
    struct ovs_doca_flow_match match = { .d = {
        .parser_meta.port_id = port_id,
    }, };
    uint32_t port_rate;

    ovs_assert(port_id != -1);
    if (OVS_UNLIKELY(!n_active_ports)) {
        n_active_ports = 1;
    }

    if (gpr_meter_id_get(ctx, port_id, &monitor.shared_meter.shared_meter_id)) {
        VLOG_ERR("%s: Failed to get port meter ID for eswitch %d",
                 netdev_get_name(netdev), ctx->esw_id);
        return -1;
    }

    if (ovs_doca_port_rate_configured()) {
        port_rate = ovs_doca_per_port_meter_rate();
    } else {
        port_rate = ovs_doca_per_core_meter_rate() * dpif_doca_get_n_pmd_threads() / n_active_ports;
    }

    if (doca_gpr_meter_config(ctx->esw_port, monitor.shared_meter.shared_meter_id, port_rate,
                              port_rate * 0.1)) {
        VLOG_ERR("%s: Failed to configure port meter for eswitch %d",
                 netdev_get_name(netdev), ctx->esw_id);
        return -1;
    }

    if (doca_offload_add_entry(netdev, AUX_QUEUE, ctx->port_meter_pipe_ctx.pipe, &match, NULL,
                               &monitor, NULL, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                               &ctx->port_meter_pipe_ctx.entry[port_id])) {
        VLOG_ERR("%s: Failed to create port-meter rule", netdev_get_name(netdev));
        return -1;
    }

    return 0;
}

static int
doca_port_meter_rule_uninit(struct doca_offload_esw_ctx *ctx, int port_id)
{
    ovs_assert(port_id != -1);
    return doca_offload_remove_entry(ctx, AUX_QUEUE, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                     &ctx->port_meter_pipe_ctx.entry[port_id]);
}

static int
doca_post_core_meter_pipe_init(struct netdev *netdev, struct doca_offload_esw_ctx *ctx)
{
    struct ovs_doca_flow_match match, match_mask;
    struct doca_flow_fwd fwd_miss = {
        .type = DOCA_FLOW_FWD_PIPE,
        .next_pipe = ctx->sw_meter_pipe_ctx.pipe,
    };
    struct doca_flow_fwd fwd = {
        .type = DOCA_FLOW_FWD_DROP,
    };

    memset(&match, 0, sizeof match);
    memset(&match_mask, 0, sizeof match_mask);
    match.d.parser_meta.meter_color = DOCA_FLOW_METER_COLOR_RED;
    match_mask.d.parser_meta.meter_color = UINT8_MAX;
    doca_set_reg_val_mask(&match.d.meta, &match_mask.d.meta, REG_FIELD_PORT_METER_COLOR,
                          UINT32_MAX);

    return doca_offload_basic_pipe_create(netdev, &match, &match_mask, NULL, NULL, NULL, NULL, &fwd,
                                          &fwd_miss, 1, UINT64_C(1) << AUX_QUEUE, "POST_CORE_METER",
                                          &ctx->post_core_meter_pipe_ctx.pipe);
}

static void
doca_offload_gpr_mode_set_cb(void *ctx_, void *key_ OVS_UNUSED, void *arg_)
{
    enum ovs_doca_gpr_mode mode = *((enum ovs_doca_gpr_mode *) arg_);
    struct doca_offload_esw_ctx *ctx = ctx_;
    struct doca_flow_fwd miss_fwd;
    struct netdev *netdev;
    int port_id;

    netdev = ctx->esw_netdev;
    memset(&miss_fwd, 0, sizeof miss_fwd);
    miss_fwd.type = DOCA_FLOW_FWD_PIPE;
    miss_fwd.next_pipe = ctx->port_meter_pipe_ctx.pipe;

    port_id = netdev_doca_get_port_id(netdev);
    ovs_assert(port_id != -1);
    if (mode == OVS_DOCA_GPR_MODE_DISABLED) {
        miss_fwd.next_pipe = ctx->sw_meter_pipe_ctx.pipe;
        doca_flow_pipe_update_miss(ctx->pre_miss_ctx.pre_miss_pipe_ctx.pipe, &miss_fwd);
        return;
    } else if (mode == OVS_DOCA_GPR_MODE_REP_ONLY &&
               ctx->port_meter_pipe_ctx.entry[port_id]) {
        doca_offload_remove_entry(ctx, AUX_QUEUE, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                  &ctx->port_meter_pipe_ctx.entry[port_id]);
    } else if (mode == OVS_DOCA_GPR_MODE_ALL_PORTS &&
               !ctx->port_meter_pipe_ctx.entry[port_id]) {
        doca_port_meter_rule_init(netdev, ctx);
    }

    /* Update the miss path to the port-meter pipe. only if the existing GPR mode is disabled.
     * Otherwise, the miss path is already set to the port-meter pipe.
     */
    if (ovs_doca_gpr_mode() == OVS_DOCA_GPR_MODE_DISABLED) {
        doca_flow_pipe_update_miss(ctx->pre_miss_ctx.pre_miss_pipe_ctx.pipe, &miss_fwd);
    }
}

static void
doca_gpr_pipes_uninit(struct doca_offload_esw_ctx *ctx)
{
    int i;

    if (ctx->port_meter_pipe_ctx.pipe) {
        doca_pipe_group_pipe_destroy(ctx, ctx->port_meter_pipe_ctx.pipe);
    }
    if (ctx->post_port_meter_pipe_ctx.pipe) {
        doca_pipe_group_pipe_destroy(ctx, ctx->post_port_meter_pipe_ctx.pipe);
    }
    if (ctx->core_meter_pipe_ctx.pipe) {
        doca_pipe_group_pipe_destroy(ctx, ctx->core_meter_pipe_ctx.pipe);
    }
    if (ctx->post_core_meter_pipe_ctx.pipe) {
        doca_pipe_group_pipe_destroy(ctx, ctx->post_core_meter_pipe_ctx.pipe);
    }

    for (i = 0; i < OVS_DOCA_MAX_GPR_METERS_PER_ESW; i++) {
        if (!ctx->shared_meters_ctx.gpr_meter_ids[i].valid) {
            continue;
        }
        doca_flow_port_shared_resource_put(ctx->esw_port, DOCA_FLOW_SHARED_RESOURCE_METER,
                                           ctx->shared_meters_ctx.gpr_meter_ids[i].id);
        ctx->shared_meters_ctx.gpr_meter_ids[i].valid = false;
    }
}

static int
doca_gpr_pipes_init(struct netdev *netdev, struct doca_offload_esw_ctx *ctx)
{
    if (doca_post_core_meter_pipe_init(netdev, ctx)) {
        goto error;
    }

    if (doca_core_meter_pipe_init(netdev, ctx)) {
        goto error;
    }

    if (doca_post_port_meter_pipe_init(netdev, ctx)) {
        goto error;
    }

    if (doca_port_meter_pipe_init(netdev, ctx)) {
        goto error;
    }

    return 0;

error:
    doca_gpr_pipes_uninit(ctx);
    return -1;
}

static void
doca_gpr_rules_uninit(struct doca_offload_esw_ctx *ctx)
{
    int i;

    if (ctx->post_core_meter_pipe_ctx.entry) {
        doca_offload_remove_entry(ctx, AUX_QUEUE, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                  &ctx->post_core_meter_pipe_ctx.entry);
    }
    if (ctx->core_meter_pipe_ctx.entry) {
        doca_offload_remove_entry(ctx, AUX_QUEUE, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                  &ctx->core_meter_pipe_ctx.entry);
    }
    for (i = 0; i < NUM_OVS_PORT_METER_COLOR; i++) {
        if (ctx->post_port_meter_pipe_ctx.entry[i]) {
            doca_offload_remove_entry(ctx, AUX_QUEUE, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                      &ctx->post_port_meter_pipe_ctx.entry[i]);
        }
    }
    for (i = 0; i < OVS_DOCA_MAX_PORT_METERS_PER_ESW; i++) {
        if (ctx->port_meter_pipe_ctx.entry[i]) {
            doca_offload_remove_entry(ctx, AUX_QUEUE, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                      &ctx->port_meter_pipe_ctx.entry[i]);
        }
    }
}

static int
doca_gpr_rules_init(struct netdev *netdev, struct doca_offload_esw_ctx *ctx)
{
    struct {
        enum doca_flow_meter_color doca_color;
        enum ovs_port_meter_color ovs_color;
    } color_map[NUM_OVS_PORT_METER_COLOR] = {
        {DOCA_FLOW_METER_COLOR_RED, OVS_PORT_METER_COLOR_RED},
        {DOCA_FLOW_METER_COLOR_YELLOW, OVS_PORT_METER_COLOR_YELLOW},
        {DOCA_FLOW_METER_COLOR_GREEN, OVS_PORT_METER_COLOR_GREEN},
    };
    struct ovs_doca_flow_actions actions;
    struct ovs_doca_flow_match match;
    int i;

    memset(&match, 0, sizeof match);

    /* Match on red meter color in post-core meter */
    doca_set_reg_val(&match.d.meta, REG_FIELD_PORT_METER_COLOR,
                     color_map[OVS_PORT_METER_COLOR_RED].ovs_color);
    match.d.parser_meta.meter_color = color_map[OVS_PORT_METER_COLOR_RED].doca_color;
    if (doca_offload_add_entry(netdev, AUX_QUEUE, ctx->post_core_meter_pipe_ctx.pipe, &match,
                               NULL, NULL, NULL, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                               &ctx->post_core_meter_pipe_ctx.entry)) {
        VLOG_ERR("%s: Failed to create post-core-meter rule", netdev_get_name(netdev));
        goto error;
    }

    if (doca_offload_add_entry(netdev, AUX_QUEUE, ctx->core_meter_pipe_ctx.pipe, &match, NULL,
                               NULL, NULL, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                               &ctx->core_meter_pipe_ctx.entry)) {
        VLOG_ERR("%s: Failed to create core-meter rule", netdev_get_name(netdev));
        goto error;
    }

    /* Match and set meter colors in post-port meter */
    memset(&match, 0, sizeof match);
    memset(&actions, 0, sizeof actions);
    for (i = 0; i < NUM_OVS_PORT_METER_COLOR; i++) {
        match.d.parser_meta.meter_color = color_map[i].doca_color;
        doca_set_reg_val(&actions.d.meta, REG_FIELD_PORT_METER_COLOR, color_map[i].ovs_color);
        if (doca_offload_add_entry(netdev, AUX_QUEUE, ctx->post_port_meter_pipe_ctx.pipe, &match,
                                   &actions, NULL, NULL, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                   &ctx->post_port_meter_pipe_ctx.entry[i])) {
            VLOG_ERR("%s: Failed to create %d post-port-meter rule",
                     netdev_get_name(netdev), color_map[i].ovs_color);
            goto error;
        }
    }

    return 0;

error:
    doca_gpr_rules_uninit(ctx);
    doca_gpr_pipes_uninit(ctx);
    return -1;
}

static void
doca_offload_flush_destroy_pipe_lists(struct doca_offload_esw_ctx *ctx)
{
    for (unsigned int qid = 0; qid < MAX_OFFLOAD_QUEUE_NB; qid++) {
        struct destroy_pipe_elem *elem;

        LIST_FOR_EACH_POP (elem, node, &ctx->destroy_pipe_lists[qid]) {
            doca_flow_pipe_destroy(elem->pipe);
            free(elem);
        }
    }
}

static void
doca_offload_esw_ctx_uninit(void *ctx_)
{
    struct doca_offload_esw_ctx *ctx = ctx_;
    struct netdev_offload_dpdk_data *data;

    data = (struct netdev_offload_dpdk_data *)
        ovsrcu_get(void *, &ctx->esw_netdev->hw_info.offload_data);

    /* Eswitch uninit means that multiple pipes will get
     * destroyed. All resizes must have been completed by then. */
    doca_offload_resolve_pipe_resize(ctx);

    atomic_store_explicit(&ovs_doca_eswitch_active_ids[ctx->esw_id], false,
                          memory_order_release);

    /* First flush all destroys that may be pending from other queues other than AUX_QUEUE. */
    doca_offload_flush_destroy_pipe_lists(ctx);

    /* The fixed rule insertions were counted in the counters of
     * the netdev that issued the eswitch context init.
     *
     * Destroying the fixed rule is done only when the last netdev
     * using this eswitch context is being removed.
     *
     * We cannot keep track of the original init netdev without inducing
     * a circular dependency.
     *
     * So remove the fixed rules without counting the deletions
     * in the uninit netdev. As all netdevs related to this eswitch
     * are meant to be removed after this, the original counts will
     * have been removed once the uninit has finished.
     */
    doca_sample_root_del_entry__(ctx);
    doca_sample_pipe_del_entry__(ctx);
    doca_sample_pipe_miss_del_entry__(ctx);
    doca_sample_pipe_uninit(ctx);
    doca_sample_mirror_pipe_uninit(ctx);
    doca_sample_post_mirror_pipe_uninit(ctx);
    doca_post_meter_pipe_uninit(ctx);

    doca_ct_zones_uninit(ctx);
    doca_offload_ct_pipe_uninit(ctx);
    doca_ct_pipes_destroy(ctx);

    doca_hash_pipe_ctx_uninit(ctx);

    doca_mirrors_destroy(ctx->doca_mirrors);
    doca_shared_encaps_ctx_uninit(&ctx->shared_encaps_ctx);
    doca_shared_meters_ctx_uninit(&ctx->shared_meters_ctx);

    doca_pipe_group_ctx_unref(ctx->root_pipe_group_ctx);
    ctx->root_pipe_group_ctx = NULL;

    /* Destroying the miss path chain should be last. */
    doca_pre_miss_rules_uninit(ctx);
    doca_pre_miss_pipe_uninit(ctx);
    doca_gpr_rules_uninit(ctx);
    doca_gpr_pipes_uninit(ctx);
    doca_post_sw_meter_pipe_uninit(ctx);
    doca_sw_meter_pipe_uninit(ctx);
    doca_meta_transfer_uninit(ctx);

    doca_offload_flush_destroy_pipe_lists(ctx);

    doca_pipe_group_too_big_map_uninit(&ctx->too_big_map);

    /* DOCA doesn't provide an api to unbind shared counters
     * and they will remain bound until the port is destroyed.
     */
    if (ctx->shared_mtr_flow_id_pool) {
        /* DOCA doesn't provide an api to unbind shared meters
         * and they will remain bound until the port is destroyed.
         */
        id_fpool_destroy(ctx->shared_mtr_flow_id_pool);
        ctx->shared_mtr_flow_id_pool = NULL;
    }
    if (esw_id_pool) {
        id_pool_free_id(esw_id_pool, ctx->esw_id);
    }

    if (ctx->gnv_opt_parser.parser) {
        doca_flow_parser_geneve_opt_destroy(ctx->gnv_opt_parser.parser);
        ctx->gnv_opt_parser.parser = NULL;
    }

    ovs_mutex_destroy(&ctx->ct_zones_mutex);
    free(ctx->offload_queues);
    ctx->offload_queues = NULL;
    ctx->esw_port = NULL;
    ctx->esw_netdev = NULL;
    data->eswitch_ctx = NULL;
}

static int
doca_offload_esw_ctx_init(void *ctx_, void *arg_)
{
    struct netdev *netdev = (struct netdev *) arg_;
    struct doca_offload_esw_ctx *ctx = ctx_;
    struct netdev_offload_dpdk_data *data;
    struct doca_flow_port *doca_port;
    unsigned int n_queues;

    /* Set the esw-ctx reference early.
     * It is required by some inner calls.
     * Beware that is it incomplete and should be handled
     * with care. */
    data = (struct netdev_offload_dpdk_data *)
        ovsrcu_get(void *, &netdev->hw_info.offload_data);
    data->eswitch_ctx = ctx;

    doca_pipe_group_too_big_map_init(&ctx->too_big_map);
    n_queues = ovs_doca_n_offload_queues();
    if (conntrack_offload_doca_ct_enabled) {
        n_queues += dpif_doca_get_n_pmd_threads();
    }
    ctx->offload_queues = xcalloc(n_queues, sizeof(ctx->offload_queues[0]));

    for (unsigned int qid = 0; qid < MAX_OFFLOAD_QUEUE_NB; qid++) {
        ovs_list_init(&ctx->resized_pipe_lists[qid]);
        ovs_list_init(&ctx->destroy_pipe_lists[qid]);
    }

    ovs_refcount_init(&ctx->pipe_resizing);
    /* The shared resources are needed by the
     * sw-meter, which are needed by all other pipes.
     */
    doca_port = netdev_doca_port_get(netdev);
    ctx->esw_port = doca_flow_port_switch_get(doca_port);
    ctx->esw_netdev = netdev;

    doca_esw_id_pool_init(ctx);
    shared_mtr_flow_id_pool_init(ctx);

    /* This mutex is used on error rollback, it must be initialized
     * before any reference to the 'error' label below. */
    ovs_mutex_init(&ctx->ct_zones_mutex);

    ctx->doca_mirrors = doca_mirrors_create(netdev);
    if (!ctx->doca_mirrors) {
        goto error;
    }

    doca_shared_encaps_ctx_init(netdev, &ctx->shared_encaps_ctx);
    doca_shared_meters_ctx_init(netdev, &ctx->shared_meters_ctx);

    if (doca_hash_pipe_ctx_init(netdev, ctx)) {
        goto error;
    }

    /* The sw-meter pipes depend on the meta transfer infrastructure. */
    if (doca_meta_transfer_init(netdev, ctx)) {
        goto error;
    }

    /* The sw-meter pipe depends on the post-sw-meter pipe. */
    if (doca_post_sw_meter_pipe_init(netdev, ctx)) {
        goto error;
    }

    /* The sw-meter pipe is referenced by each control and
     * basic pipes as default miss target. Initialize it
     * as soon as possible. */
    if (doca_sw_meter_pipe_init(netdev, ctx)) {
        goto error;
    }

    if (doca_gpr_pipes_init(netdev, ctx)) {
        goto error;
    }

    if (doca_gpr_rules_init(netdev, ctx)) {
        goto error;
    }

    if (doca_pre_miss_pipe_init(netdev, ctx)) {
        goto error;
    }

    if (doca_pre_miss_rules_init(netdev, ctx)) {
        goto error;
    }
    ctx->root_pipe_group_ctx = doca_pipe_group_ctx_ref(netdev, 0, NULL);
    if (ctx->root_pipe_group_ctx == NULL) {
        goto error;
    }

    doca_port = netdev_doca_port_get(netdev);
    ctx->esw_port = doca_flow_port_switch_get(doca_port);
    ctx->gnv_opt_parser.parser = NULL;

    if (conntrack_offload_is_enabled()) {
        if (conntrack_offload_doca_ct_enabled) {
            if (doca_offload_ct_pipe_init(netdev, ctx)) {
                goto error;
            }
        }
        if (ovs_doca_max_ct_counters_per_esw()) {
            if (doca_ct_pipes_init(netdev, ctx)) {
                goto error;
            }
        }
    }

    if (doca_post_meter_pipe_init(netdev, ctx)) {
        goto error;
    }

    if (doca_sample_post_mirror_pipe_init(netdev, ctx)) {
        goto error;
    }

    if (doca_sample_mirror_pipe_init(netdev, ctx)) {
        goto error;
    }

    if (doca_sample_pipe_init(netdev, ctx)) {
        goto error;
    }
    atomic_store_explicit(&ovs_doca_eswitch_active_ids[ctx->esw_id], true,
                          memory_order_release);

    return 0;

error:
    /* Roll-back esw-ctx access on error. */
    VLOG_ERR("%s: Failed to init eswitch %d",
             netdev_get_name(netdev),
             netdev_doca_get_esw_mgr_port_id(netdev));
    doca_offload_esw_ctx_uninit(ctx);
    return -1;
}

static struct ds *
dump_doca_eswitch(struct ds *s, void *key_, void *ctx_ OVS_UNUSED,
                  void *arg_ OVS_UNUSED)
{
    struct doca_flow_port **esw_port = key_;

    ds_put_format(s, "esw_port=%p", *esw_port);

    return s;
}

static void
doca_eswitch_init(void)
{
    static struct ovsthread_once init_once = OVSTHREAD_ONCE_INITIALIZER;

    if (ovsthread_once_start(&init_once)) {
        doca_eswitch_rfm = refmap_create("doca-eswitch-map",
                                         sizeof(struct doca_flow_port *),
                                         sizeof(struct doca_offload_esw_ctx),
                                         doca_offload_esw_ctx_init,
                                         doca_offload_esw_ctx_uninit,
                                         dump_doca_eswitch, false);

        ovsthread_once_done(&init_once);
    }
}

/* Get the current eswitch context for this netdev,
 * /!\ without taking a reference, and without creating it!
 * The eswitch context must have been initialized once
 * beforehand using 'doca_offload_esw_ctx_ref()' for this netdev.
 */
struct doca_offload_esw_ctx *
doca_offload_esw_ctx_get(const struct netdev *netdev)
{
    struct netdev_offload_dpdk_data *data;

    data = (struct netdev_offload_dpdk_data *)
        ovsrcu_get(void *, &netdev->hw_info.offload_data);
    return data ? data->eswitch_ctx : NULL;
}

struct netdev *
doca_offload_get_esw_netdev(struct netdev *netdev)
{
    struct doca_offload_esw_ctx *esw_ctx;

    esw_ctx = doca_offload_esw_ctx_get(netdev);
    if (!esw_ctx) {
        return NULL;
    }

    return esw_ctx->esw_netdev;
}

static struct doca_offload_esw_ctx *
doca_offload_esw_ctx_ref(struct netdev *netdev)
{
    struct doca_flow_port *doca_port = netdev_doca_port_get(netdev);
    struct doca_flow_port *esw_port;

    esw_port = doca_flow_port_switch_get(doca_port);

    doca_eswitch_init();
    return refmap_ref(doca_eswitch_rfm, &esw_port, netdev);
}

static void
doca_offload_esw_ctx_unref(struct doca_offload_esw_ctx *ctx)
{
    refmap_unref(doca_eswitch_rfm, ctx);
}

void
doca_offload_gpr_mode_set(enum ovs_doca_gpr_mode mode)
{
    if (doca_eswitch_rfm) {
        refmap_for_each(doca_eswitch_rfm, doca_offload_gpr_mode_set_cb, &mode);
    }
}
static bool
add_sw_meter_match(struct netdev *netdev)
{
    struct netdev_offload_dpdk_data *data;
    struct doca_offload_esw_ctx *esw_ctx;
    struct doca_flow_monitor monitor;
    struct ovs_doca_flow_match match;
    struct dpdk_offload_handle *doh;
    struct doca_flow_fwd fwd;
    bool ret = true;

    data = (struct netdev_offload_dpdk_data *)
        ovsrcu_get(void *, &netdev->hw_info.offload_data);

    if (!data) {
        return false;
    }

    esw_ctx = doca_offload_esw_ctx_get(netdev);

    memset(&match, 0, sizeof match);
    memset(&monitor, 0, sizeof monitor);

    match.d.parser_meta.port_id = netdev_doca_get_port_id(netdev);
    match.d.parser_meta.meter_color = DOCA_FLOW_METER_COLOR_RED;
    monitor.counter_type = DOCA_FLOW_RESOURCE_TYPE_NON_SHARED;

   if (netdev_doca_sw_meter_get_dry_run(netdev)) {
        struct doca_basic_pipe_ctx *meta_push_pipe_ctx = NULL;

        meta_push_pipe_ctx = &esw_ctx->meta_transfer_ctx.meta_push_pipe_ctx;
        fwd.type = DOCA_FLOW_FWD_PIPE;
        fwd.next_pipe = meta_push_pipe_ctx->pipe;
    } else {
        fwd.type = DOCA_FLOW_FWD_DROP;
    }

    doh = &data->sw_meter_match;
    if (doca_offload_add_entry(netdev, AUX_QUEUE, esw_ctx->post_sw_meter_pipe_ctx.pipe, &match,
                               NULL, &monitor, &fwd, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                               &doh->dfh.flow)) {
        VLOG_ERR("%s: Failed to create sw-meter red rule",
                 netdev_get_name(netdev));
        ret = false;
    }

    return ret;
}

static bool
add_sw_meter_apply(struct netdev *netdev)
{
    struct netdev_offload_dpdk_data *data;
    struct doca_offload_esw_ctx *esw_ctx;
    struct doca_flow_monitor monitor;
    struct ovs_doca_flow_match match;
    struct dpdk_offload_handle *doh;
    bool ret = true;
    int port_id;

    data = (struct netdev_offload_dpdk_data *)
        ovsrcu_get(void *, &netdev->hw_info.offload_data);

    if (!data) {
        return false;
    }

    memset(&monitor, 0, sizeof monitor);
    memset(&match, 0, sizeof match);

    esw_ctx = doca_offload_esw_ctx_get(netdev);
    port_id = netdev_doca_get_port_id(netdev);

    match.d.parser_meta.port_id = port_id;
    monitor.meter_type = DOCA_FLOW_RESOURCE_TYPE_SHARED;

    if (sw_meter_id_get(esw_ctx, ovs_doca_sw_meter_dp_ext_id(port_id),
                        &monitor.shared_meter.shared_meter_id)) {
        VLOG_ERR("%s: Failed to get sw meter ID for eswitch %d",
                 netdev_get_name(netdev), esw_ctx->esw_id);
        return false;
    }

    doh = &data->sw_meter_apply;
    if (doca_offload_add_entry(netdev, AUX_QUEUE, esw_ctx->sw_meter_pipe_ctx.pipe, &match, NULL,
                               &monitor, NULL, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                               &doh->dfh.flow)) {
        VLOG_ERR("%s: Failed to create sw-meter rule",
                 netdev_get_name(netdev));
        ret = false;
    }

    return ret;
}

void
dpdk_offload_doca_sw_meter_uninit(struct netdev *netdev)
{
    struct netdev_offload_dpdk_data *data;
    struct doca_offload_esw_ctx *esw;

    data = (struct netdev_offload_dpdk_data *)
        ovsrcu_get(void *, &netdev->hw_info.offload_data);

    if (!data) {
        return;
    }
    esw = data->eswitch_ctx;

    if (data->sw_meter_apply.dfh.flow) {
        doca_offload_remove_entry(esw, AUX_QUEUE, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                  &data->sw_meter_apply.dfh.flow);
        dpdk_offload_counter_dec(netdev);
    }
    if (data->sw_meter_match.dfh.flow) {
        doca_offload_remove_entry(esw, AUX_QUEUE, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                  &data->sw_meter_match.dfh.flow);
        dpdk_offload_counter_dec(netdev);
    }
}

/* Configure the shared meter that is used for sw-meter,
 * Insert the 'red-match' rule in post-sw-meter.
 * Insert the 'apply-meter' rule in sw-meter.
 */
int
dpdk_offload_doca_sw_meter_init(struct netdev *netdev)
{
    struct netdev_offload_dpdk_data *data;
    struct ofputil_meter_config config;
    struct ofputil_meter_band band;
    ofproto_meter_id dp_meter_id;
    bool modification;
    int port_id;

    data = (struct netdev_offload_dpdk_data *)
        ovsrcu_get(void *, &netdev->hw_info.offload_data);

    if (!data) {
        return -1;
    }

    if (!netdev_doca_sw_meter_get_protected(netdev, &config, &band)) {
        return 0;
    }

    port_id = netdev_doca_get_port_id(netdev);
    dp_meter_id.uint32 = ovs_doca_sw_meter_dp_ext_id(port_id);
    config.meter_id = dp_meter_id.uint32;

    if (netdev_doca_meter_set(dp_meter_id, &config)) {
        VLOG_ERR("%s: Failed to configure shared meter resource for "
                 "sw-meter config.", netdev_get_name(netdev));
        return -1;
    }

    modification = data->sw_meter_match.dfh.flow != NULL;
    if (!modification) {
        if (!add_sw_meter_match(netdev)) {
            goto error;
        }
        if (!add_sw_meter_apply(netdev)) {
            goto error;
        }
    } else {
        struct doca_offload_esw_ctx *esw;

        /* In case there is a modification, remove the old rule and add a new one to make sure
         * dry-run sw-meter is taken into account.
         */
        esw = doca_offload_esw_ctx_get(netdev);
        doca_offload_remove_entry(esw, AUX_QUEUE, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                  &data->sw_meter_match.dfh.flow);
        dpdk_offload_counter_dec(netdev);
        if (!add_sw_meter_match(netdev)) {
            goto error;
        }
    }

    if (VLOG_IS_DBG_ENABLED()) {
        struct ds s = DS_EMPTY_INITIALIZER;

        if (config.flags & OFPMF13_PKTPS) {
            ds_put_cstr(&s, "pps:");
        } else if (config.flags & OFPMF13_KBPS) {
            ds_put_cstr(&s, "bps:");
            /* The sw-meter configuration is in bps, not Kbps. */
            band.rate *= 1000;
            band.burst_size *= 1000;
        }
        ds_put_format(&s, "%"PRIu32":%"PRIu32,
                      band.rate, band.burst_size);
        VLOG_DBG("%s: Installed sw-meter: %s",
                 netdev_get_name(netdev), ds_cstr(&s));
        ds_destroy(&s);
    }

    return 0;

error:
    dpdk_offload_doca_sw_meter_uninit(netdev);
    return -1;
}

static void
dpdk_offload_doca_aux_tables_uninit(struct netdev *netdev)
{
    struct netdev_offload_dpdk_data *data;

    if (!netdev_doca_is_ethdev(netdev)) {
        return;
    }

    data = (struct netdev_offload_dpdk_data *)
        ovsrcu_get(void *, &netdev->hw_info.offload_data);
    if (!data || !data->eswitch_ctx) {
        return;
    }

    dpdk_offload_doca_sw_meter_uninit(netdev);
    doca_port_meter_rule_uninit(data->eswitch_ctx,
                                netdev_doca_get_port_id(netdev));

    doca_offload_esw_ctx_unref(data->eswitch_ctx);
    data->eswitch_ctx = NULL;
}

static int
dpdk_offload_doca_aux_tables_init(struct netdev *netdev)
{
    struct netdev_offload_dpdk_data *data;
    struct doca_offload_esw_ctx *ctx;

    /* Skip initialization if the eSwitch manager is not yet available.
     * This check prevents premature referencing of the eSwitch manager.
     * Once the eSwitch manager is up, it will be responsible for re-invoking
     * this function for all its representors, at which point this check will pass.
     */
    if (netdev_doca_get_esw_mgr_port_id(netdev) == -1) {
        return 0;
    }

    ctx = doca_offload_esw_ctx_ref(netdev);
    if (!ctx) {
        VLOG_ERR("%s: Failed to get doca eswitch ctx", netdev_get_name(netdev));
        return -1;
    }

    data = (struct netdev_offload_dpdk_data *)
        ovsrcu_get(void *, &netdev->hw_info.offload_data);
    data->eswitch_ctx = ctx;

    if (doca_port_meter_rule_init(netdev, ctx)) {
        VLOG_ERR("%s: Failed to install port-meter rule.", netdev_get_name(netdev));
        return -1;
    }

    return 0;
}

static void
log_conn_rule(struct conn *conn, int dir, int error,
              uint32_t ct_action_label_id)
{
    struct ds s;

    if (error) {
        if (VLOG_DROP_ERR(&rl)) {
            return;
        }
    } else {
        if (VLOG_DROP_DBG(&rl)) {
            return;
        }
    }

    ds_init(&s);

    conntrack_offload_dump(&s, conn, dir, ct_action_label_id);
    if (error) {
        VLOG_ERR("conn create failed: %s", ds_cstr(&s));
    } else {
        VLOG_DBG("conn create: %s", ds_cstr(&s));
    }

    ds_destroy(&s);
}

static struct doca_split_prefix_ctx *
doca_insert_ip6_prefix_conn(struct netdev *netdev,
                            const struct ct_match *ct_match,
                            uint32_t ct_match_zone_id,
                            enum ct_tp_type tp_type,
                            uint32_t *prefix_id)
{
    struct doca_split_prefix_key prefix_key;
    struct doca_split_prefix_ctx *pctx;
    struct doca_split_prefix_arg args;
    struct ovs_doca_flow_match *dspec;
    struct doca_offload_esw_ctx *ctx;

    memset(&prefix_key, 0, sizeof prefix_key);
    dspec = &prefix_key.spec;

    /* IPv6 CT prefix match: port_id + zone + src_ip + next_proto. */
    doca_set_reg_val(&dspec->d.meta, REG_FIELD_CT_ZONE, ct_match_zone_id);
    dspec->d.parser_meta.port_id = netdev_doca_get_port_id(netdev);

    dspec->d.outer.ip6.next_proto = tp_type == CT_TP_TCP ? IPPROTO_TCP : IPPROTO_UDP;
    memcpy(dspec->d.outer.ip6.src_ip, &ct_match->key.src.addr.ipv6,
           sizeof dspec->d.outer.ip6.src_ip);

    ctx = doca_offload_esw_ctx_get(netdev);
    args.netdev = netdev;
    args.spec = dspec;
    args.prefix_is_group = false;
    args.prefix_pipe = ctx->ct_ip6_prefix.pipe;
    args.suffix_pipe = ctx->ct_pipes[CT_NW_IP6][tp_type].pipe;

    prefix_key.prefix_pipe = args.prefix_pipe;

    pctx = split_prefix_ctx_ref(&prefix_key, &args, prefix_id);
    if (!pctx) {
        VLOG_WARN_RL(&rl, "%s: Failed to associate a CT IPv6 prefix with a "
                     "prefix ID", netdev_get_name(netdev));
        return NULL;
    }

    return pctx;
}

static int
doca_offload_insert_conn_dir(struct doca_offload_esw_ctx *esw,
                             unsigned int qid,
                             struct netdev *netdev,
                             struct ct_flow_offload_item ct_offload[1],
                             struct conn_item *ci,
                             struct act_resources *act_res,
                             uint32_t ct_match_zone_id,
                             bool last)
{
    struct doca_split_prefix_ctx *pctx = NULL;
    struct doca_flow_header_format *dhdr;
    struct ovs_doca_flow_actions odacts;
    struct ovs_doca_flow_match dspec;
    const struct ct_match *ct_match;
    struct doca_flow_actions *dacts;
    struct doca_flow_monitor dmon;
    struct doca_flow_pipe *pipe;
    enum ct_nw_type nw_type;
    enum ct_tp_type tp_type;
    unsigned int prefix_id;
    uint32_t ct_state_spec;
    doca_error_t err;

    ct_match = &ct_offload->ct_match;
    dacts = &odacts.d;

    tp_type = ct_match->key.nw_proto == IPPROTO_TCP ? CT_TP_TCP : CT_TP_UDP;
    if (ct_match->key.dl_type == htons(ETH_TYPE_IP)) {
        nw_type = CT_NW_IP4;
    } else if (ct_match->key.dl_type == htons(ETH_TYPE_IPV6)) {
        nw_type = CT_NW_IP6;
        /* No IPv6 conn request should be generated if not enabled. */
        ovs_assert(conntrack_offload_ipv6_is_enabled());
    } else {
        VLOG_DBG_RL(&rl, "Unsupported CT network type.");
        return -1;
    }

    memset(&dspec.d.meta, 0, sizeof dspec.d.meta);

    dhdr = &dspec.d.outer;

    /* IPv4 */
    if (nw_type == CT_NW_IP4) {
        dhdr->ip4.src_ip = (OVS_FORCE doca_be32_t) ct_match->key.src.addr.ipv4;
        dhdr->ip4.dst_ip = (OVS_FORCE doca_be32_t) ct_match->key.dst.addr.ipv4;
        dhdr->ip4.next_proto = ct_match->key.nw_proto;
    /* IPv6 */
    } else {
        /* ip6 src match will be in the prefix rule while dst match in the
         * suffix rule
         */
        pctx = doca_insert_ip6_prefix_conn(netdev, ct_match, ct_match_zone_id,
                                           tp_type, &prefix_id);
        if (!pctx) {
            return -1;
        }

        memcpy(dhdr->ip6.dst_ip, &ct_match->key.dst.addr.ipv6,
               sizeof dhdr->ip6.dst_ip);
        doca_set_reg_val(&dspec.d.meta, REG_FIELD_SCRATCH, prefix_id);
    }

    if (tp_type == CT_TP_TCP) {
        dhdr->tcp.l4_port.src_port = (OVS_FORCE doca_be16_t) ct_match->key.src.port;
        dhdr->tcp.l4_port.dst_port = (OVS_FORCE doca_be16_t) ct_match->key.dst.port;
    } else {
        dhdr->udp.l4_port.src_port = (OVS_FORCE doca_be16_t) ct_match->key.src.port;
        dhdr->udp.l4_port.dst_port = (OVS_FORCE doca_be16_t) ct_match->key.dst.port;
    }

    doca_set_reg_val(&dspec.d.meta, REG_FIELD_CT_ZONE, ct_match_zone_id);

    dspec.d.parser_meta.port_id = netdev_doca_get_port_id(netdev);

    dhdr = &dacts->outer;

    /* Common part for all CT, plain and NAT. */

    if (nw_type == CT_NW_IP4) {
        dhdr->l3_type = DOCA_FLOW_L3_TYPE_IP4;
        dhdr->ip4.src_ip = (OVS_FORCE doca_be32_t) ct_match->key.src.addr.ipv4;
        dhdr->ip4.dst_ip = (OVS_FORCE doca_be32_t) ct_match->key.dst.addr.ipv4;
        dspec.d.parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV4;
    } else {
        dhdr->l3_type = DOCA_FLOW_L3_TYPE_IP6;
        memcpy(&dhdr->ip6.src_ip, &ct_match->key.src.addr.ipv6,
               sizeof dhdr->ip6.src_ip);
        memcpy(&dhdr->ip6.dst_ip, &ct_match->key.dst.addr.ipv6,
               sizeof dhdr->ip6.dst_ip);
        dspec.d.parser_meta.outer_l3_type = DOCA_FLOW_L3_META_IPV6;
    }

    if (tp_type == CT_TP_TCP) {
        dhdr->l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_TCP;
        dhdr->tcp.l4_port.src_port = (OVS_FORCE doca_be16_t) ct_match->key.src.port;
        dhdr->tcp.l4_port.dst_port = (OVS_FORCE doca_be16_t) ct_match->key.dst.port;
        dspec.d.parser_meta.outer_l4_type = DOCA_FLOW_L4_META_TCP;
    } else {
        dhdr->l4_type_ext = DOCA_FLOW_L4_TYPE_EXT_UDP;
        dhdr->udp.l4_port.src_port = (OVS_FORCE doca_be16_t) ct_match->key.src.port;
        dhdr->udp.l4_port.dst_port = (OVS_FORCE doca_be16_t) ct_match->key.dst.port;
        dspec.d.parser_meta.outer_l4_type = DOCA_FLOW_L4_META_UDP;
    }

    /* For NAT translate the relevant fields. */
    if (ct_offload->nat.mod_flags) {
        if (nw_type == CT_NW_IP4) {
            if (ct_offload->nat.mod_flags & NAT_ACTION_SRC) {
                dhdr->ip4.src_ip = (OVS_FORCE doca_be32_t) ct_offload->nat.key.src.addr.ipv4;
            } else if (ct_offload->nat.mod_flags & NAT_ACTION_DST) {
                dhdr->ip4.dst_ip = (OVS_FORCE doca_be32_t) ct_offload->nat.key.dst.addr.ipv4;
            }
        } else {
            if (ct_offload->nat.mod_flags & NAT_ACTION_SRC) {
                memcpy(&dhdr->ip6.src_ip, &ct_offload->nat.key.src.addr.ipv6,
                       sizeof dhdr->ip6.src_ip);
            } else if (ct_offload->nat.mod_flags & NAT_ACTION_DST) {
                memcpy(&dhdr->ip6.dst_ip, &ct_offload->nat.key.dst.addr.ipv6,
                       sizeof dhdr->ip6.dst_ip);
            }
        }
        if (ct_offload->nat.mod_flags & NAT_ACTION_SRC_PORT) {
            if (tp_type == CT_TP_TCP) {
                dhdr->tcp.l4_port.src_port = (OVS_FORCE doca_be16_t) ct_offload->nat.key.src.port;
            } else {
                dhdr->udp.l4_port.src_port = (OVS_FORCE doca_be16_t) ct_offload->nat.key.src.port;
            }
        } else if (ct_offload->nat.mod_flags & NAT_ACTION_DST_PORT) {
            if (tp_type == CT_TP_TCP) {
                dhdr->tcp.l4_port.dst_port = (OVS_FORCE doca_be16_t) ct_offload->nat.key.dst.port;
            } else {
                dhdr->udp.l4_port.dst_port = (OVS_FORCE doca_be16_t) ct_offload->nat.key.dst.port;
            }
        }
    }

    memset(&dmon, 0, sizeof dmon);
    dmon.counter_type = DOCA_FLOW_RESOURCE_TYPE_SHARED;
    dmon.shared_counter.shared_counter_id = act_res->shared_count_ctx->res_id;

    memset(&dacts->meta, 0, sizeof dacts->meta);

    /* CT MARK */
    doca_set_reg_val(&dacts->meta, REG_FIELD_CT_MARK, ct_offload->mark_key);

    /* CT LABEL */
    doca_set_reg_val(&dacts->meta, REG_FIELD_CT_LABEL_ID, act_res->ct_action_label_id);

    /* CT STATE */
    ct_state_spec = ct_offload->ct_state;
    /* If any of the NAT bits is set, set both of them.
     * As the NAT action is executed, the 5-tuple of the packet is modified.
     * A revisit with the post-NAT tuple would miss. */
    if (ct_offload->ct_state & OVS_CS_F_NAT_MASK) {
        ct_state_spec |= OVS_CS_F_NAT_MASK;
    }
    doca_set_reg_val(&dacts->meta, REG_FIELD_CT_STATE, ct_state_spec);

    memset(ci, 0, sizeof *ci);

    pipe = esw->ct_pipes[nw_type][tp_type].pipe;
    err = doca_offload_add_entry(netdev, qid, pipe, &dspec, &odacts, &dmon, NULL,
                                 last ? DOCA_FLOW_ENTRY_FLAGS_NO_WAIT :
                                 DOCA_FLOW_ENTRY_FLAGS_WAIT_FOR_BATCH,
                                 &ci->doca_flow);
    if (err) {
        VLOG_WARN_RL(&rl, "%s: Failed to create ct entry: Error %d (%s)",
                     netdev_get_name(netdev), err, doca_error_get_descr(err));
        return -1;
    }
    ci->valid = true;
    ci->curr_split_ctx = pctx;

    return 0;
}

static int
dpdk_offload_doca_destroy_conn_dir(struct netdev *netdev,
                                   struct conn_item *ci,
                                   struct rte_flow_error *error)
{
    struct doca_offload_esw_ctx *esw = doca_offload_esw_ctx_get(netdev);
    unsigned int qid = netdev_offload_thread_id();
    doca_error_t err;

    /* Deletion is always synchronous.
     *
     * If async deletion is implemented, aux-table uninit calls deleting
     * entries will use the offload queues in conflict with offload threads
     * polling them during upkeep. It should result in a crash or
     * in a lockup of the queues. */
    err = doca_offload_remove_entry(esw, qid, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT, &ci->doca_flow);
    if (err) {
        if (error) {
            error->type = RTE_FLOW_ERROR_TYPE_HANDLE;
            error->message = doca_error_get_descr(err);
        }
        if (err == DOCA_ERROR_IN_USE) {
            return EAGAIN;
        }
        return -1;
    }

    if (ci->curr_split_ctx) {
        split_prefix_ctx_unref(ci->curr_split_ctx);
    }

    /* Netdev can only be NULL during aux tables uninit. */
    if (netdev) {
        dpdk_offload_counter_dec(netdev);
    }

    return 0;
}

static void
doca_flow_ct_translate_match(struct ct_flow_offload_item *item, uint32_t ct_zone_id,
                             struct doca_flow_ct_match *m)
{
    const struct ct_match *ct_match;
    enum ct_nw_type nw_type;

    ct_match = &item->ct_match;
    nw_type = ct_match->key.dl_type == htons(ETH_TYPE_IP) ? CT_NW_IP4 : CT_NW_IP6;

    if (nw_type == CT_NW_IP4) {
        m->ipv4.metadata = DOCA_HTOBE32(reg_field_spec(REG_FIELD_CT_ZONE, ct_zone_id));
        m->ipv4.src_ip = (OVS_FORCE doca_be32_t) ct_match->key.src.addr.ipv4;
        m->ipv4.dst_ip = (OVS_FORCE doca_be32_t) ct_match->key.dst.addr.ipv4;
        m->ipv4.next_proto = ct_match->key.nw_proto;
        m->ipv4.l4_port.src_port = (OVS_FORCE doca_be16_t) ct_match->key.src.port;
        m->ipv4.l4_port.dst_port = (OVS_FORCE doca_be16_t) ct_match->key.dst.port;
    } else {
        m->ipv6.metadata = DOCA_HTOBE32(reg_field_spec(REG_FIELD_CT_ZONE, ct_zone_id));
        memcpy(&m->ipv6.src_ip, &ct_match->key.src.addr.ipv6, sizeof m->ipv6.src_ip);
        memcpy(&m->ipv6.dst_ip, &ct_match->key.dst.addr.ipv6, sizeof m->ipv6.dst_ip);
        m->ipv6.next_proto = ct_match->key.nw_proto;
        m->ipv6.l4_port.src_port = (OVS_FORCE doca_be16_t) ct_match->key.src.port;
        m->ipv6.l4_port.dst_port = (OVS_FORCE doca_be16_t) ct_match->key.dst.port;
    }
}

static void
doca_flow_meta_hton(struct doca_flow_meta *m)
{
    m->pkt_meta = DOCA_HTOBE32(m->pkt_meta);
    for (int i = 0; i < ARRAY_SIZE(m->u32); i++) {
        m->u32[i] = DOCA_HTOBE32(m->u32[i]);
    }
}

static void
doca_flow_ct_actions_set(struct doca_flow_ct_actions *a, enum ct_nw_type nw_type, bool nat,
                         uint32_t ct_label, uint32_t ct_mark, uint8_t ct_state)
{
    struct doca_flow_ct_meta ct_mask;

    memset(a, 0, sizeof *a);
    a->resource_type = DOCA_FLOW_RESOURCE_TYPE_NON_SHARED;
    a->data.action_idx = CT_ACTIONS_IDX(nw_type, nat);

    /* If any of the NAT bits is set, set both of them.
     * As the NAT action is executed, the 5-tuple of the packet is modified.
     * A revisit with the post-NAT tuple would miss. */
    if (ct_state & OVS_CS_F_NAT_MASK) {
        ct_state |= OVS_CS_F_NAT_MASK;
    }

    /* DOCA-CT has a significant API difference from DOCA-flow regarding
     * meta fields. All register writes must be packed together in adjacent
     * tags, and the offsets are handled by DOCA-CT internally:
     * they must not be handled by the application.
     *
     * In the following example:
     *   struct doca_flow_meta template = {
     *       .u32[2] = 0xffffffff,
     *       .u32[4] = 0xffffffff,
     *   };
     *
     *   struct doca_flow_meta mask = {
     *       .u32[2] = 0xffffffff,
     *       .u32[4] = 0xff00ff00,
     *   };
     *
     *   struct doca_flow_meta value = {
     *       .u32[2] = 0x12345678,
     *       .u32[4] = 0x00120034,
     *   };
     *
     * In DOCA-flow, value.u32[4] should have been 0x12003400;
     * In DOCA-CT, it must instead be set to 0x120034 and DOCA-CT will use
     * the CT pipe mask to set it properly.
     *
     * After all values have been written to the meta struct and binary OR-ed together,
     * move any field that has an offset where DOCA-CT expects it to be.
     *
     * Finally, once the CTZ-shifts are done, the doca_flow_meta structure must
     * be transformed to network-order. It must be done all at once after the above
     * transformations.
     */

    doca_set_reg_val_host_order(&a->data.meta.flow, REG_FIELD_CT_STATE, ct_state);
    doca_set_reg_val_host_order(&a->data.meta.flow, REG_FIELD_CT_MARK, ct_mark);
    doca_set_reg_val_host_order(&a->data.meta.flow, REG_FIELD_CT_LABEL_ID, ct_label);
    memset(&ct_mask, 0, sizeof ct_mask);
    doca_set_reg_mask_host_order(&ct_mask.flow, REG_FIELD_CT_STATE);
    doca_set_reg_mask_host_order(&ct_mask.flow, REG_FIELD_CT_MARK);
    doca_set_reg_mask_host_order(&ct_mask.flow, REG_FIELD_CT_LABEL_ID);
    for (int i = 0; i < ARRAY_SIZE(a->data.meta.flow.u32); i++) {
        int ctz = ctz32(ct_mask.flow.u32[i]);

        if (ctz && ctz != 32) {
            a->data.meta.flow.u32[i] >>= ctz;
        }
    }

    doca_flow_meta_hton(&a->data.meta.flow);
    a->data.meta.mark = DOCA_HTOBE32(a->data.meta.mark);
}

static bool
doca_flow_ct_shared_actions_create(struct doca_offload_esw_ctx *esw,
                                   unsigned int tid,
                                   struct shared_ct_actions_key *key,
                                   uint32_t *id, struct rte_flow_error *error)
{
    unsigned int qid = doca_offload_ct_queue_id(tid);
    struct doca_flow_ct_actions actions;
    doca_error_t err;

    /* For plain-CT, the packet header is not modified.
     * The network protocol version does not matter.
     * Here CT_NW_IP4 is arbitrary.
     * Similarly, 'nat' is always false for shared actions.
     */
    doca_flow_ct_actions_set(&actions, CT_NW_IP4, false,
                             key->ct_label, key->ct_mark, key->ct_state);

    err = doca_flow_ct_actions_add_shared(qid, esw->ct.pipe, &actions, 1, id);
    if (DOCA_IS_ERROR(err)) {
        error->type = RTE_FLOW_ERROR_TYPE_UNSPECIFIED;
        error->message = doca_error_get_descr(err);
        return false;
    }
    return true;
}

static void
doca_flow_ct_shared_actions_destroy(struct doca_offload_esw_ctx *esw, uint32_t id)
{
    unsigned int qid = doca_offload_ct_queue_id(netdev_offload_thread_id());
    doca_error_t err;

    err = doca_flow_ct_actions_rm_shared(qid, esw->ct.pipe, &id, 1);
    if (DOCA_IS_ERROR(err)) {
        VLOG_ERR("Failed to delete shared CT actions %" PRIu32": %s", id,
                 doca_error_get_descr(err));
    }
}

static void
doca_flow_ct_translate_actions(struct ct_flow_offload_item *item, uint32_t ct_label_id,
                               struct doca_flow_ct_actions *a)
{
    const struct ct_match *ct_match;
    enum ct_nw_type nw_type;
    bool nat;

    ct_match = &item->ct_match;
    nw_type = ct_match->key.dl_type == htons(ETH_TYPE_IP) ? CT_NW_IP4 : CT_NW_IP6;
    nat = item->nat.mod_flags;

    doca_flow_ct_actions_set(a, nw_type, nat, ct_label_id, item->mark_key, item->ct_state);

    if (!nat) {
        return;
    }

    if (item->nat.mod_flags & NAT_ACTION_SRC) {
        /* SNAT + PAT */
        if (nw_type == CT_NW_IP4) {
            a->data.ip4.src_ip = (OVS_FORCE doca_be32_t) item->nat.key.src.addr.ipv4;
            a->data.ip4.dst_ip = (OVS_FORCE doca_be32_t) ct_match->key.dst.addr.ipv4;
        } else {
            memcpy(&a->data.ip6.src_ip, &item->nat.key.src.addr.ipv6, sizeof a->data.ip6.src_ip);
            memcpy(&a->data.ip6.dst_ip, &ct_match->key.dst.addr.ipv6, sizeof a->data.ip6.dst_ip);
        }
        a->data.l4_port.src_port = (OVS_FORCE doca_be16_t) item->nat.key.src.port;
        a->data.l4_port.dst_port = (OVS_FORCE doca_be16_t) ct_match->key.dst.port;
    } else {
        /* DNAT + PAT */
        if (nw_type == CT_NW_IP4) {
            a->data.ip4.src_ip = (OVS_FORCE doca_be32_t) ct_match->key.src.addr.ipv4;
            a->data.ip4.dst_ip = (OVS_FORCE doca_be32_t) item->nat.key.dst.addr.ipv4;
        } else {
            memcpy(&a->data.ip6.src_ip, &ct_match->key.src.addr.ipv6, sizeof a->data.ip6.src_ip);
            memcpy(&a->data.ip6.dst_ip, &item->nat.key.dst.addr.ipv6, sizeof a->data.ip6.dst_ip);
        }
        a->data.l4_port.src_port = (OVS_FORCE doca_be16_t) ct_match->key.src.port;
        a->data.l4_port.dst_port = (OVS_FORCE doca_be16_t) item->nat.key.dst.port;
    }
}

static int
doca_offload_insert_conn_doca_ct(struct doca_offload_esw_ctx *esw,
                                 unsigned int qid,
                                 struct netdev *netdevs[CT_DIR_NUM],
                                 struct ct_flow_offload_item items[CT_DIR_NUM],
                                 struct conn_item *ci,
                                 struct act_resources *act_res,
                                 uint32_t ct_match_zone_id,
                                 bool last)
{
    struct doca_flow_pipe_entry *entry = NULL;
    doca_error_t err = DOCA_ERROR_UNKNOWN;
    uint32_t rep_fwd_handle;
    uint32_t ct_entry_flags;
    bool update = ci->valid;
    struct {
        struct doca_flow_ct_match match;
        struct doca_flow_ct_actions actions;
    } dir[CT_DIR_NUM];

    if (items->ct_match.key.dl_type == htons(ETH_TYPE_IPV6)) {
        if (!conntrack_offload_ipv6_is_enabled()) {
            VLOG_ERR_RL(&rl, "%s/%s: CT-IPv6 offload is disabled",
                        netdev_get_name(netdevs[CT_DIR_INIT]),
                        netdev_nullable_get_name(netdevs[CT_DIR_REP]));
            return -1;
        }
    }

    memset(dir, 0, sizeof dir);

    for (int i = 0; i < CT_DIR_NUM; i++) {
        if (!update) {
            doca_flow_ct_translate_match(&items[i], ct_match_zone_id, &dir[i].match);
        }
        if (act_res->shared_ct_actions_ctx[i]) {
            dir[i].actions.resource_type = DOCA_FLOW_RESOURCE_TYPE_SHARED;
            dir[i].actions.action_handle = act_res->shared_ct_actions_ctx[i]->res_id;
        } else if (netdevs[i]) {
            /* Should not add CT actions for dummy rule in REP in unidirectional UDP offload. */
            doca_flow_ct_translate_actions(&items[i], act_res->ct_action_label_id, &dir[i].actions);
        }
    }

    ct_entry_flags = last ? DOCA_FLOW_CT_ENTRY_FLAGS_NO_WAIT : 0;
    ct_entry_flags |= DOCA_FLOW_CT_ENTRY_FLAGS_COUNTER_ORIGIN;
    ct_entry_flags |= DOCA_FLOW_CT_ENTRY_FLAGS_COUNTER_REPLY;
    ct_entry_flags |= DOCA_FLOW_CT_ENTRY_FLAGS_COUNTER_SHARED;
    ct_entry_flags |= DOCA_FLOW_CT_ENTRY_FLAGS_DIR_ORIGIN;
    ct_entry_flags |= DOCA_FLOW_CT_ENTRY_FLAGS_DIR_REPLY;
    if (items->ct_match.key.dl_type == htons(ETH_TYPE_IPV6)) {
        ct_entry_flags |= DOCA_FLOW_CT_ENTRY_FLAGS_IPV6_ORIGIN;
        ct_entry_flags |= DOCA_FLOW_CT_ENTRY_FLAGS_IPV6_REPLY;
    }

    if (!update) {
        /* Prepare the entry with NULL matches: it will always miss and always allocate an entry. */
        err = doca_flow_ct_entry_prepare(qid, esw->ct.pipe, DOCA_FLOW_CT_ENTRY_FLAGS_ALLOC_ON_MISS,
                                         NULL, 0, NULL, 0, &entry, NULL);
        if (DOCA_IS_ERROR(err)) {
            VLOG_ERR_RL(&rl, "%s: failed to allocate DOCA-CT entry: %s",
                        netdev_get_name(netdevs[CT_DIR_INIT]), doca_error_get_descr(err));
            return err;
        }

        if (netdevs[CT_DIR_REP]) {
            rep_fwd_handle = esw->ct.fwd_handles[CT_FWD_POST];
        } else {
            rep_fwd_handle = esw->ct.fwd_handles[CT_FWD_MISS];
        }
        err = doca_flow_ct_add_entry(qid, esw->ct.pipe, ct_entry_flags,
                                     &dir[CT_DIR_INIT].match, &dir[CT_DIR_REP].match,
                                     &dir[CT_DIR_INIT].actions, &dir[CT_DIR_REP].actions,
                                     esw->ct.fwd_handles[CT_FWD_POST],
                                     rep_fwd_handle,
                                     0, esw->offload_queues, entry);
        if (DOCA_IS_ERROR(err)) {
            VLOG_ERR_RL(&rl, "%s: failed to insert DOCA-CT entry: %d (%s)",
                        netdev_get_name(netdevs[CT_DIR_INIT]),
                        err, doca_error_get_descr(err));
            goto rollback_entry;
        }

        dpdk_offload_counter_inc(netdevs[CT_DIR_INIT]);
        if (netdevs[CT_DIR_REP]) {
            dpdk_offload_counter_inc(netdevs[CT_DIR_REP]);
        }
        ci->doca_flow = entry;
        ci->valid = true;
    } else {
        err = doca_flow_ct_update_entry(qid, esw->ct.pipe, ct_entry_flags, ci->doca_flow,
                                        &dir[CT_DIR_INIT].actions, &dir[CT_DIR_REP].actions,
                                        esw->ct.fwd_handles[CT_FWD_POST],
                                        esw->ct.fwd_handles[CT_FWD_POST],
                                        0);
        if (DOCA_IS_ERROR(err)) {
            VLOG_ERR_RL(&rl, "%s: failed to update DOCA-CT entry: %d (%s)",
                        netdev_get_name(netdevs[CT_DIR_INIT]),
                        err, doca_error_get_descr(err));
            goto err_ret;
        }
        dpdk_offload_counter_inc(netdevs[CT_DIR_REP]);
    }
    esw->offload_queues[qid].n_waiting_entries++;

    return 0;

rollback_entry:
    /* Ignore potential error from this call, to keep the current value of 'err'. */
    doca_flow_ct_entry_prepare_rollback(qid, esw->ct.pipe, entry);
err_ret:
    return err;
}

static int
doca_offload_destroy_conn_dirs(struct netdev *netdevs[CT_DIR_NUM], struct conn *conn,
                               bool *aux_queue_needs_complete,
                               struct rte_flow_error *error)
{
    struct ct_offload_handle *coh;
    int rets[CT_DIR_NUM];

    memset(rets, 0, sizeof rets);

    coh = conntrack_offload_get(conn);

    for (int dir = 0; dir < CT_DIR_NUM; dir++) {
        if (!coh->dir[dir].conn_item.valid) {
            continue;
        }
        if (!netdevs[dir]) {
            continue;
        }

        if (coh->dir[dir].conn_item.curr_split_ctx) {
            *aux_queue_needs_complete = true;
        }
        rets[dir] = dpdk_offload_doca_destroy_conn_dir(netdevs[dir],
                                                       &coh->dir[dir].conn_item,
                                                       error);
    }
    netdev_offload_doca_free_conn_res(&coh->act_resources);
    return rets[CT_DIR_INIT] | rets[CT_DIR_REP];
}

static int
doca_offload_destroy_conn_doca_ct(struct netdev *netdevs[CT_DIR_NUM],
                                  struct conn *conn,
                                  struct rte_flow_error *error)
{
    struct doca_offload_esw_ctx *esw;
    doca_error_t err = DOCA_SUCCESS;
    struct ct_offload_handle *coh;
    struct conn_item *ci;
    unsigned int qid;
    int retries;

    coh = conntrack_offload_get(conn);
    ci = &coh->dir[0].conn_item;
    /* Read the thread ID from the connection. In the normal case, it is
     * strictly equivalent to reading the current thread ID. In the case of
     * an NDU synchronization, the main thread executes this code and it must
     * act in place of the original thread that previously inserted this offload
     * in the NDU server instance.
     */
    qid = doca_offload_ct_queue_id(conntrack_insertion_tid(conn));
    esw = doca_offload_esw_ctx_get(netdevs[CT_DIR_INIT] ? netdevs[CT_DIR_INIT]
                                                        : netdevs[CT_DIR_REP]);

    if (ci->doca_flow == NULL) {
        goto free_resources;
    }

    retries = 100;
    while (retries-- > 0) {
        err = doca_flow_ct_rm_entry(qid, esw->ct.pipe,
                                    DOCA_FLOW_CT_ENTRY_FLAGS_NO_WAIT,
                                    ci->doca_flow);
        if (err == DOCA_SUCCESS) {
            esw->offload_queues[qid].n_waiting_entries++;
            err = doca_offload_complete_queue_n(esw, qid,
                                                OVS_DOCA_CT_QUEUE_DEPTH);
            if (netdevs[CT_DIR_INIT]) {
                dpdk_offload_counter_dec(netdevs[CT_DIR_INIT]);
            }
            if (netdevs[CT_DIR_REP]) {
                dpdk_offload_counter_dec(netdevs[CT_DIR_REP]);
            }
        }
        if (err != DOCA_ERROR_IN_PROGRESS) {
            break;
        }
    }

free_resources:
    netdev_offload_doca_free_conn_res(&coh->act_resources);
    if (DOCA_IS_ERROR(err)) {
        error->type = RTE_FLOW_ERROR_TYPE_UNSPECIFIED;
        error->message = doca_error_get_descr(err);
        return -1;
    }
    return 0;
}

static int
dpdk_offload_doca_destroy_conn(struct netdev *netdevs[CT_DIR_NUM], struct conn *conn,
                               struct rte_flow_error *error)
{
    bool aux_queue_needs_complete = false;
    struct ct_offload_handle *coh;
    struct conn_item *ci;
    int ret;

    coh = conntrack_offload_get(conn);
    ci = &coh->dir[0].conn_item;

    if (ci->use_doca_ct) {
        return doca_offload_destroy_conn_doca_ct(netdevs, conn, error);
    } else {
        struct doca_offload_esw_ctx *esw;

        esw = doca_offload_esw_ctx_get(netdevs[CT_DIR_INIT] ? netdevs[CT_DIR_INIT]
                                       : netdevs[CT_DIR_REP]);
        ret = doca_offload_destroy_conn_dirs(netdevs, conn, &aux_queue_needs_complete, error);
        if (aux_queue_needs_complete) {
            doca_offload_complete_queue_n(esw, AUX_QUEUE, OVS_DOCA_QUEUE_DEPTH);
        }
        return ret;
    }
}

static void
dpdk_offload_doca_insert_conns__(struct netdev *netdevs[CT_DIR_NUM], struct batch *conns,
                                 bool batch_use_doca_ct)
{
    struct doca_offload_esw_ctx *esw = doca_offload_esw_ctx_get(netdevs[CT_DIR_INIT]);
    bool aux_queue_needs_complete = false;
    struct rte_flow_error error;
    struct conn *conn;
    unsigned int tid;
    unsigned int qid;
    size_t n_conns;

    /* If DOCA-CT is used:
     *   -> Do not allocate shared counters, they are handled by DOCA-CT.
     *   -> Do try to merge CT actions if possible (plain CT).
     * Else:
     *   -> Allocate one shared counter used in both directions.
     *   -> Impossible to share CT actions with basic pipe CT.
     */

    tid = !conns->md ? netdev_offload_thread_id() : ((struct conn_batch_md *) conns->md)->tid;
    netdev_offload_doca_get_conns_res(netdevs, conns, !batch_use_doca_ct, batch_use_doca_ct);

    qid = batch_use_doca_ct ? doca_offload_ct_queue_id(tid) : tid;

    n_conns = batch_size(conns);
    BATCH_FOREACH_POP (idx, conn, conns) {
        struct ct_flow_offload_item items[CT_DIR_NUM];
        bool last = idx == n_conns - 1;
        struct ct_offload_handle *coh;
        int ret;

        coh = conntrack_offload_get(conn);
        conntrack_offload_build_add_items(NULL, conn, items, 0);

        if (batch_use_doca_ct) {
            ret = doca_offload_insert_conn_doca_ct(esw, qid, netdevs, items,
                                                   &coh->dir[0].conn_item,
                                                   &coh->act_resources,
                                                   conn->key.zone,
                                                   last);
            coh->dir[0].conn_item.use_doca_ct = true;
            log_conn_rule(conn, CT_DIR_INIT, ret, coh->act_resources.ct_action_label_id);
            log_conn_rule(conn, CT_DIR_REP, ret, coh->act_resources.ct_action_label_id);
        } else {
            for (int dir = 0; dir < CT_DIR_NUM; dir++) {
                if (items[dir].ct_match.key.dl_type == htons(ETH_TYPE_IPV6)) {
                    /* IPV6 split can cause offload on AUX_QUEUE */
                    aux_queue_needs_complete = true;
                }

                ret = doca_offload_insert_conn_dir(esw, qid, netdevs[dir],
                                                   &items[dir],
                                                   &coh->dir[dir].conn_item,
                                                   &coh->act_resources,
                                                   conn->key.zone,
                                                   last && dir);
                coh->dir[dir].conn_item.use_doca_ct = false;
                log_conn_rule(conn, dir, ret, coh->act_resources.ct_action_label_id);
            }
        }

        if (ret) {
            dpdk_offload_doca_destroy_conn(netdevs, conn, &error);
            netdev_offload_doca_free_conn_res(&coh->act_resources);
            VLOG_ERR_RL(&rl, "%s/%s: Failed to insert connection %p offload.",
                        netdev_get_name(netdevs[CT_DIR_INIT]),
                        netdev_nullable_get_name(netdevs[CT_DIR_REP]), conn);
        } else {
            batch_add(conns, conn);
        }
    }

    doca_offload_complete_queue_n(esw, qid, batch_use_doca_ct ? OVS_DOCA_CT_QUEUE_DEPTH
                                                              : OVS_DOCA_QUEUE_DEPTH);
    if (aux_queue_needs_complete) {
        doca_offload_complete_queue_n(esw, AUX_QUEUE, OVS_DOCA_QUEUE_DEPTH);
    }
}

static int
dpdk_offload_doca_insert_conns(struct netdev *netdevs[CT_DIR_NUM], struct batch *conns)
{
    if (conntrack_offload_ipv6_is_enabled() &&
        conntrack_offload_doca_ct_enabled && !conntrack_offload_doca_ct_ipv6_enabled) {
        struct conn_batch_md *md = (struct conn_batch_md *) conns->md, v4md, v6md;
        struct batch v4conns, v6conns;
        struct conn *conn;
        int v4idx, v6idx;

        /* If DOCA-CT is enabled for IPv4 but not IPv6, it is possible to use
         * a mix of doca-ct or basic pipe implementation.
         * Dispatch the provided batch in two batches depending on which
         * implementation they require, then process one then the other.
         */
        batch_init(&v4conns);
        batch_init(&v6conns);
        memset(&v4md, 0, sizeof v4md);
        memset(&v6md, 0, sizeof v6md);
        v4md.tid = md->tid;
        v6md.tid = md->tid;
        v4conns.md = &v4md;
        v6conns.md = &v6md;
        v4idx = 0;
        v6idx = 0;
        BATCH_FOREACH_POP (idx, conn, conns) {
            struct ct_flow_offload_item items[CT_DIR_NUM];

            conntrack_offload_build_add_items(NULL, conn, items, 0);
            if (items->ct_match.key.dl_type == htons(ETH_TYPE_IPV6)) {
                v6md.unidir_update[v6idx++] = md->unidir_update[idx];
                batch_add(&v6conns, conn);
            } else {
                v4md.unidir_update[v4idx++] = md->unidir_update[idx];
                batch_add(&v4conns, conn);
            }
        }

        dpdk_offload_doca_insert_conns__(netdevs, &v4conns, conntrack_offload_doca_ct_enabled);
        dpdk_offload_doca_insert_conns__(netdevs, &v6conns, conntrack_offload_doca_ct_ipv6_enabled);

        BATCH_FOREACH_POP (conn, &v4conns) {
            batch_add(conns, conn);
        }
        BATCH_FOREACH_POP (conn, &v6conns) {
            batch_add(conns, conn);
        }
    } else {
        /* Use the same implementation for the whole batch, whether DOCA-CT or not. */
        dpdk_offload_doca_insert_conns__(netdevs, conns, conntrack_offload_doca_ct_enabled);
    }

    return -!batch_size(conns);
}

static int
dpdk_offload_doca_packet_hw_entropy(struct netdev *netdev,
                                    struct dp_packet *packet,
                                    uint16_t *entropy)
{
    struct doca_flow_entropy_format header;
    struct doca_flow_port *port;
    uint8_t ip_proto = 0;
    ovs_be16 eth_proto;
    doca_error_t err;
    void *nh;
    void *l4;

    parse_tcp_flags(packet, &eth_proto, NULL, NULL);
    nh = dp_packet_l3(packet);
    l4 = dp_packet_l4(packet);

    if (!netdev_doca_is_ethdev(netdev)) {
        return -1;
    }

    memset(&header, 0, sizeof header);

    if (eth_proto == htons(ETH_TYPE_IP) && nh) {
        struct ip_header *ip = nh;

        header.l3_type = DOCA_FLOW_L3_TYPE_IP4;
        header.ip4.src_ip = (OVS_FORCE doca_be32_t) get_16aligned_be32(&ip->ip_src);
        header.ip4.dst_ip = (OVS_FORCE doca_be32_t) get_16aligned_be32(&ip->ip_dst);
        ip_proto = ip->ip_proto;
        header.ip4.next_proto = ip_proto;
    } else if (eth_proto == htons(ETH_TYPE_IPV6) && nh) {
        struct ovs_16aligned_ip6_hdr *ip6 = nh;

        header.l3_type = DOCA_FLOW_L3_TYPE_IP6;
        memcpy(header.ip6.src_ip, &ip6->ip6_src, sizeof(ovs_be32[4]));
        memcpy(header.ip6.dst_ip, &ip6->ip6_dst, sizeof(ovs_be32[4]));
        ip_proto = ip6->ip6_nxt;
        header.ip6.next_proto = ip_proto;
    } else {
        header.l3_type = DOCA_FLOW_L3_TYPE_NONE;
    }

    if (ip_proto == IPPROTO_TCP && l4) {
        struct tcp_header *tcp = l4;

        header.is_transport = true;
        header.transport.src_port = (OVS_FORCE doca_be16_t) tcp->tcp_src;
        header.transport.dst_port = (OVS_FORCE doca_be16_t) tcp->tcp_dst;
    } else if (ip_proto == IPPROTO_UDP && l4) {
        struct udp_header *udp = l4;

        header.is_transport = true;
        header.transport.src_port = (OVS_FORCE doca_be16_t) udp->udp_src;
        header.transport.dst_port = (OVS_FORCE doca_be16_t) udp->udp_dst;
    }

    port = doca_flow_port_switch_get(netdev_doca_port_get(netdev));
    err = doca_flow_port_calc_entropy(port, &header, entropy);
    if (err != DOCA_SUCCESS) {
        return -1;
    }

    return 0;
}

struct shared_meter_set_cb_arg {
    uint32_t meter_id;
    struct doca_flow_resource_meter_cfg *meter_cfg;
};

static void
doca_offload_shared_meter_set_cb(void *ctx_, void *key_ OVS_UNUSED, void *arg_)
{
    struct shared_meter_set_cb_arg *arg = arg_;
    struct doca_offload_esw_ctx *ctx = ctx_;

    doca_get_shared_meter(arg->meter_id, arg->meter_cfg, ctx->esw_port, &ctx->shared_meters_ctx);
}

struct shared_meter_unset_cb_arg {
    uint32_t meter_id;
};

static void
doca_offload_shared_meter_unset_cb(void *ctx_, void *key_ OVS_UNUSED, void *arg_)
{
    struct shared_meter_unset_cb_arg *arg = arg_;
    struct doca_offload_esw_ctx *ctx = ctx_;

    doca_put_shared_meter(arg->meter_id, &ctx->shared_meters_ctx);
}

static int
query_shared_counter(struct doca_flow_port *port, uint32_t id,
                     struct doca_flow_resource_query *stats)
{
    doca_error_t ret;

    ret = doca_flow_port_shared_resources_query(port, DOCA_FLOW_SHARED_RESOURCE_COUNTER, &id,
                                                stats, 1);
    if (ret != DOCA_SUCCESS) {
        VLOG_ERR("Failed to query shared meter counter id %u: %s",
                 id, doca_error_get_descr(ret));
        return -1;
    }

    return 0;
}

struct shared_meter_stats_get_cb_arg {
    uint32_t meter_id;
    struct doca_flow_resource_query *red_stats;
    struct doca_flow_resource_query *green_stats;
};

static void
doca_offload_shared_meter_stats_get_cb(void *ctx_, void *key_ OVS_UNUSED, void *arg_)
{
    struct shared_meter_stats_get_cb_arg *arg = arg_;
    struct doca_offload_esw_ctx *ctx = ctx_;
    struct doca_shared_meter_ctx *meter_ctx;
    struct doca_flow_resource_query results;
    struct doca_shared_meter_key key = {
        .of_meter_id = arg->meter_id
    };
    int ret;

    meter_ctx = refmap_find(ctx->shared_meters_ctx.rfm, &key);
    if (!meter_ctx) {
        return;
    }

    ret = query_shared_counter(ctx->esw_port, meter_ctx->red_id, &results);
    if (!ret) {
        arg->red_stats->counter.total_bytes += results.counter.total_bytes;
        arg->red_stats->counter.total_pkts += results.counter.total_pkts;
    }
    ret = query_shared_counter(ctx->esw_port, meter_ctx->green_id, &results);
    if (!ret) {
        arg->green_stats->counter.total_bytes += results.counter.total_bytes;
        arg->green_stats->counter.total_pkts += results.counter.total_pkts;
    }
}

struct doca_flow_pipe_entry *
dpdk_offload_sample_entry_get(const struct netdev *netdev)
{
    struct doca_offload_esw_ctx *esw_ctx = doca_offload_esw_ctx_get(netdev);

    return esw_ctx ? esw_ctx->sample_ctx.sample_pipe_entry_mctx.entry : NULL;
}

struct ovs_list *
doca_offload_destroy_list_get(struct doca_offload_esw_ctx *esw_ctx)
{
    return &esw_ctx->destroy_pipe_lists[netdev_offload_thread_id()];
}

struct too_big_map *
doca_offload_too_big_map_get(struct doca_offload_esw_ctx *esw_ctx)
{
    return &esw_ctx->too_big_map;
}

void
doca_offload_shared_meter_set(uint32_t meter_id, struct doca_flow_resource_meter_cfg *meter_cfg)
{
    struct shared_meter_set_cb_arg set_cb_arg = {
        .meter_id = meter_id,
        .meter_cfg = meter_cfg
    };

    if (!doca_eswitch_rfm) {
        return;
    }

    /* Create meter for each eswitch if it doesn't exist or modify configuration of existing meter
     * if it was updated.
     */
    refmap_for_each(doca_eswitch_rfm, doca_offload_shared_meter_set_cb, &set_cb_arg);
}

void
doca_offload_shared_meter_unset(uint32_t meter_id)
{
    struct shared_meter_unset_cb_arg unset_cb_arg = { meter_id };

    if (!doca_eswitch_rfm) {
        return;
    }

    refmap_for_each(doca_eswitch_rfm, doca_offload_shared_meter_unset_cb, &unset_cb_arg);
}

void
doca_offload_shared_meter_stats_get(uint32_t meter_id, struct doca_flow_resource_query *red_stats,
                                    struct doca_flow_resource_query *green_stats)
{
    struct shared_meter_stats_get_cb_arg get_cb_arg = {
        .meter_id = meter_id,
        .red_stats = red_stats,
        .green_stats = green_stats
    };

    if (!doca_eswitch_rfm) {
        return;
    }

    memset(red_stats, 0, sizeof *red_stats);
    memset(green_stats, 0, sizeof *green_stats);
    refmap_for_each(doca_eswitch_rfm, doca_offload_shared_meter_stats_get_cb, &get_cb_arg);
}

struct dpdk_offload_api dpdk_offload_api_doca = {
    .upkeep = dpdk_offload_doca_upkeep,
    .create = dpdk_offload_doca_create,
    .destroy = dpdk_offload_doca_destroy,
    .query_count = dpdk_offload_doca_query_count,
    .get_packet_recover_info = dpdk_offload_doca_get_pkt_recover_info,
    .insert_conns = dpdk_offload_doca_insert_conns,
    .destroy_conn = dpdk_offload_doca_destroy_conn,
    .query_conn = dpdk_offload_doca_query_conn,
    .reg_fields = dpdk_offload_doca_get_reg_fields,
    .update_stats = dpdk_offload_doca_update_stats,
    .aux_tables_init = dpdk_offload_doca_aux_tables_init,
    .aux_tables_uninit = dpdk_offload_doca_aux_tables_uninit,
    .shared_create = dpdk_offload_doca_shared_create,
    .shared_destroy = dpdk_offload_doca_shared_destroy,
    .shared_query = dpdk_offload_doca_shared_query,
    .packet_hw_hash = dpdk_offload_doca_packet_hw_hash,
    .packet_hw_entropy = dpdk_offload_doca_packet_hw_entropy,
};
