/*
 * SPDX-FileCopyrightText: NVIDIA CORPORATION & AFFILIATES
 * Copyright (c) 2022-2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: LicenseRef-NvidiaProprietary
 *
 * NVIDIA CORPORATION, its affiliates and licensors retain all intellectual
 * property and proprietary rights in and to this material, related
 * documentation and any modifications thereto. Any use, reproduction,
 * disclosure or distribution of this material and related documentation
 * without an express license agreement from NVIDIA CORPORATION or
 * its affiliates is strictly prohibited.
 */

#ifndef CONNTRACK_OFFLOAD_H
#define CONNTRACK_OFFLOAD_H

#include "conntrack.h"
#include "openvswitch/types.h"
#include "smap.h"
#include "batch.h"

enum ct_timeout;
struct conn;
struct conntrack;
struct conntrack_offload_class;
struct dp_packet;
struct netdev;
struct ct_offload_handle;

enum ct_offload_flag {
    CT_OFFLOAD_NONE = 0,
    CT_OFFLOAD_INIT = 0x1 << 0,
    CT_OFFLOAD_REP  = 0x1 << 1,
    CT_OFFLOAD_SKIP_INIT = 0x1 << 2,
    CT_OFFLOAD_SKIP_REP  = 0x1 << 3,
    CT_OFFLOAD_SKIP_BOTH = (CT_OFFLOAD_SKIP_INIT | CT_OFFLOAD_SKIP_REP),
    CT_OFFLOAD_BOTH = (CT_OFFLOAD_INIT | CT_OFFLOAD_REP),
    CT_OFFLOAD_BYPASS_TCP_SEQ_CHK = 0x1 << 4,
    CT_OFFLOAD_DELETED = 1 << 5,
    CT_OFFLOAD_REQUESTED = 1 << 6,
};

struct ct_dir_info {
    odp_port_t port;
    ovs_u128 ufid;
    void *dp;
    uint8_t pkt_ct_state;
    uint32_t pkt_ct_mark;
    ovs_u128 pkt_ct_label;
    /* This pointer is atomic as it is sync
     * between PMD, ct_clean and hw_offload threads. */
    OVSRCU_TYPE(void *) offload_data;
};

extern bool conntrack_offload_doca_ct_enabled;
extern bool conntrack_offload_doca_ct_ipv6_enabled;

struct conn_batch_md {
    unsigned int tid;
    bool unidir_update[BATCH_SIZE];
};

static inline void *
ct_dir_info_data_get(const struct ct_dir_info *info)
{
    return ovsrcu_get(void *, &info->offload_data);
}

struct ct_offloads {
    uint8_t flags;
    bool unidir;
    unsigned int insertion_tid;
    struct ct_offload_handle *coh;
    struct ct_dir_info dir_info[CT_DIR_NUM];
};

struct ct_flow_offload_item {
    int  op;
    ovs_u128 ufid;
    void *ct;
    void *dp;
    uintptr_t ctid_key;
    long long int timestamp;

    /* matches */
    struct ct_match ct_match;

    /* actions */
    uint8_t ct_state;
    ovs_u128 label_key;
    uint32_t mark_key;

    struct {
        uint8_t mod_flags;
        struct conn_key  key;
    } nat;

    void *offload_data;
};

/* hw-offload callbacks */
struct conntrack_offload_class {
    void (*conn_get_ufid)(ovs_u128 *);
    void (*conns_add)(struct conntrack *, struct batch *);
    void (*conn_del)(struct conntrack *, struct conn *);
    int (*conn_active)(struct conntrack *, struct conn *, long long int, long long int);
};

bool conntrack_offload_valid(struct conntrack *ct, struct conn *conn, struct dp_packet *packet,
                             bool reply, bool *unidir_update);
void conntrack_offload_prepare(struct conntrack *ct, struct conn *conn,
                               struct dp_packet *packet, bool reply);
void conntrack_offload_add(struct conntrack *ct, struct batch *conns);

int
conn_hw_update(struct conntrack *ct,
               struct conntrack_offload_class *offload_class,
               struct conn *conn,
               long long now);
void
conntrack_set_offload_class(struct conntrack *,
                            struct conntrack_offload_class *);
void
conntrack_offload_del_conn(struct conntrack *ct,
                           struct conn *conn,
                           bool flush);


void
conntrack_offload_build_del_items(struct conntrack *ct, struct conn *conn,
                                  struct ct_flow_offload_item items[CT_DIR_NUM]);
void
conntrack_offload_build_add_items(struct conntrack *ct, struct conn *conn,
                                  struct ct_flow_offload_item items[CT_DIR_NUM],
                                  long long int now_us);
bool
conntrack_offload_build_stats_items(struct conntrack *ct, struct conn *conn,
                                    struct ct_flow_offload_item items[CT_DIR_NUM]);


unsigned int conntrack_offload_size(void);
bool conntrack_offload_is_enabled(void);
bool conntrack_offload_ipv6_is_enabled(void);
bool conntrack_offload_unidir_udp_is_enabled(void);
void conntrack_offload_config(const struct smap *other_config);
void
conntrack_offload_netdev_flush(struct conntrack *ct, struct netdev *netdev);

void
conntrack_offload_dump(struct ds *ds, struct conn *conn, int dir,
                       uint32_t ct_action_label_id);

struct ct_offload_handle *
conntrack_offload_get(struct conn *conn);

unsigned int
conntrack_offload_get_insertion_tid(struct conn *conn);

void
conntrack_offload_get_ports(struct conn *conn, odp_port_t ports[CT_DIR_NUM]);

void conntrack_offload_free(struct conn *conn);

bool
conntrack_offload_is_unidir(struct conn *conn);

#endif /* CONNTRACK_OFFLOAD_H */
