/*
 * SPDX-FileCopyrightText: NVIDIA CORPORATION & AFFILIATES
 * Copyright (c) 2023-2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: LicenseRef-NvidiaProprietary
 *
 * NVIDIA CORPORATION, its affiliates and licensors retain all intellectual
 * property and proprietary rights in and to this material, related
 * documentation and any modifications thereto. Any use, reproduction,
 * disclosure or distribution of this material and related documentation
 * without an express license agreement from NVIDIA CORPORATION or
 * its affiliates is strictly prohibited.
 */

#ifndef NETDEV_OFFLOAD_DOCA_H
#define NETDEV_OFFLOAD_DOCA_H

#include <sys/types.h>

#include "cmap.h"
#include "conntrack.h"
#include "conntrack-offload.h"
#include "dpdk-offload-provider.h"
#include "dpif.h"
#include "openvswitch/list.h"
#include "ovs-atomic.h"
#include "ovs-rcu.h"

struct unixctl_conn;
struct table_ctx;

struct act_resources {
    struct table_ctx *next_table_ctx;
    struct table_ctx *self_table_ctx;
    uint32_t flow_miss_ctx_id;
    uint32_t tnl_id;
    uint32_t flow_id;
    bool associated_flow_id;
    uint32_t ct_miss_ctx_id;
    uint32_t ct_match_zone_id;
    uint32_t ct_action_zone_id;
    uint32_t ct_match_label_id;
    uint32_t ct_action_label_id;
    uint32_t ct_action_label_ctx_id;
    struct ovs_shared_ctx *shared_age_ctx;
    struct ovs_shared_ctx *shared_count_ctx;
    struct ovs_shared_ctx *shared_ct_actions_ctx[CT_DIR_NUM];
    uint32_t sflow_id;
    uint32_t meter_ids[DPDK_OFFLOAD_MAX_METERS_PER_FLOW];
};

struct ufid_to_rte_flow_data {
    union {
        struct cmap_node node;
        struct ovs_list list_node;
    };
    ovs_u128 ufid;
    struct netdev *netdev;
    struct flow_item flow_item;
    struct dpif_flow_stats stats;
    struct dpif_flow_stats baseline_stats;
    struct netdev *physdev;
    struct ovs_mutex lock;
    unsigned int creation_tid;
    struct ovsrcu_gc_node gc_node;
    atomic_bool active;
    struct act_resources act_resources;
};

static inline bool
rte_flow_data_active(struct ufid_to_rte_flow_data *data)
{
    bool active;

    if (!data) {
        return false;
    }
    atomic_read(&data->active, &active);
    return active;
}

static inline void
rte_flow_data_active_set(struct ufid_to_rte_flow_data *data, bool state)
{
    atomic_store(&data->active, state);
}

struct ct_offload_handle {
    struct {
        struct conn_item conn_item;
        struct netdev *netdev;
        atomic_bool active;
    } dir[CT_DIR_NUM];
    struct ovs_list list_node;
    struct conn *conn;
    struct act_resources act_resources;
    struct dpif_flow_stats stats;
    struct ovs_spin lock;
    bool valid;
};

/* Only CT_OFFLOAD_INIT and CT_OFFLOAD_REP flags are considered in 'dir'.
 * Return true if the flagged direction(s) offload is(are) currently active.
 * If CT_OFFLOAD_BOTH is set, then return true if either direction is offloaded.
 */
static inline bool
ct_offload_handle_active(struct ct_offload_handle *coh, enum ct_offload_flag dir)
{
    bool ret = false;
    bool active;

    if (!coh) {
        return false;
    }

    if (dir & CT_OFFLOAD_INIT) {
        atomic_read(&coh->dir[CT_DIR_INIT].active, &active);
        ret |= active;
    }

    if (!ret && (dir & CT_OFFLOAD_REP)) {
        atomic_read(&coh->dir[CT_DIR_REP].active, &active);
        ret |= active;
    }

    return ret;
}

static inline bool
ct_offload_handle_active_ct_dir(struct ct_offload_handle *coh, enum ct_direction dir)
{
    enum ct_offload_flag dir_flag;

    if (dir == CT_DIR_INIT) {
        dir_flag = CT_OFFLOAD_INIT;
    } else if (dir == CT_DIR_REP) {
        dir_flag = CT_OFFLOAD_REP;
    } else {
        dir_flag = CT_OFFLOAD_BOTH;
    }

    return ct_offload_handle_active(coh, dir_flag);
}

/* Only CT_OFFLOAD_INIT and CT_OFFLOAD_REP flags are considered in 'dir'.
 * Sets the respective mark for each set direction to 'state'.
 */
static inline void
ct_offload_handle_active_set(struct ct_offload_handle *coh, enum ct_offload_flag dir, bool state)
{
    if (dir & CT_OFFLOAD_INIT) {
        atomic_store(&coh->dir[CT_DIR_INIT].active, state);
    }
    if (dir & CT_OFFLOAD_REP) {
        atomic_store(&coh->dir[CT_DIR_REP].active, state);
    }
}

void
netdev_offload_doca_get_conns_res(struct netdev *netdevs[CT_DIR_NUM], struct batch *conns,
                                  bool with_miss_ctx, bool with_shared_counters,
                                  bool with_shared_ct_actions);

void
netdev_offload_doca_free_conn_res(struct act_resources *act_resources);

#ifdef DPDK_NETDEV

#include <rte_flow.h>

struct table_ctx {
    struct netdev *netdev;
    struct dpdk_offload_handle miss_flow;
    uint32_t id;
};

/*
 * To avoid individual xrealloc calls for each new element, a 'curent_max'
 * is used to keep track of current allocated number of elements. Starts
 * by 8 and doubles on each xrealloc call.
 */
struct flow_patterns {
    struct rte_flow_item *items;
    int cnt;
    int current_max;
    struct netdev *physdev;
    struct ds s_tnl;
    /* Tag matches must be merged per-index. Keep track of
     * each index and use a single item for each. */
    struct rte_flow_item_tag *tag_spec[REG_TAG_INDEX_NUM];
    struct rte_flow_item_tag *tag_mask[REG_TAG_INDEX_NUM];
};

struct flow_actions {
    struct rte_flow_action *actions;
    int cnt;
    int current_max;
    struct ds s_tnl;
    int shared_age_action_pos;
    int shared_count_action_pos;
    int count_action_pos;
    bool has_encap;
};

extern struct dpdk_offload_api *dpdk_offload_api;

void
netdev_offload_doca_dump_traces(struct unixctl_conn *conn, int argc OVS_UNUSED,
                                const char *argv[] OVS_UNUSED, void *aux OVS_UNUSED);

OVS_ASSERT_PACKED(struct flow_miss_ctx,
    /* Manual padding must be used instead of PADDED_MEMBERS. */
    odp_port_t vport;
    uint32_t recirc_id;
    uint16_t esw_mgr_port_id;
    uint8_t has_dp_hash;
    bool is_miss_flow;
    uint8_t pad0[4];
    struct flow_tnl tnl;
);

/* Query the sw-meter installed on 'netdev' for the
 * number of dropped packets as a result of the rate-limiting.
 *
 * 'netdev': A netdev that has been initialized with the
 *           netdev-offload-dpdk offload provider.
 * 'query': The field 'hits' is set to the number of
 *          dropped packets, while 'bytes' is set to
 *          the number of dropped bytes.
 *
 * The return value is 0 on success, !0 otherwise.
 * If no sw-meter was configured on 'netdev', the value 0
 * is still returned and 'query' fields are set to 0.
 */
int
netdev_offload_doca_sw_meter_drops(const struct netdev *netdev,
                                   struct rte_flow_query_count *query);

int
netdev_offload_doca_aux_tables_init(struct netdev *netdev);

void
netdev_offload_doca_aux_tables_uninit(struct netdev *netdev);

int
netdev_offload_doca_hw_miss_packet_recover(struct netdev *netdev,
                                           struct dp_packet *packet,
                                           struct dpif_sflow_attr *sflow_attr,
                                           bool dump);
int
netdev_offload_doca_init_flow_api(struct netdev *netdev);
void
netdev_offload_doca_uninit_flow_api(struct netdev *netdev);

enum ct_mode {
    CT_MODE_NONE,
    CT_MODE_CT,
    CT_MODE_CT_NAT,
};

struct act_vars {
    enum ct_mode ct_mode;
    odp_port_t vport;
    uint32_t recirc_id;
    struct flow_tnl *tnl_key;
    struct flow_tnl tnl_mask;
    uintptr_t ct_counter_key;
    uintptr_t flows_counter_key;
    enum tun_type tun_type;
    bool is_outer_ipv4;
    uint8_t gnv_opts_cnt;
    bool is_ct_conn;
    rte_be16_t vlan_tpid;
    uint8_t vlan_pcp;
    uint8_t proto;
    bool has_dp_hash;
    bool has_decap;
    odp_port_t tnl_push_out_port;
    struct ds offload_error;
};

#define IS_REWRITE_ACTION(act_type) (\
    ((act_type) == RTE_FLOW_ACTION_TYPE_SET_MAC_SRC) || \
    ((act_type) == RTE_FLOW_ACTION_TYPE_SET_MAC_DST) || \
    ((act_type) == RTE_FLOW_ACTION_TYPE_SET_IPV4_SRC) || \
    ((act_type) == RTE_FLOW_ACTION_TYPE_SET_IPV4_DST) || \
    ((act_type) == RTE_FLOW_ACTION_TYPE_SET_IPV4_TTL) || \
    ((act_type) == RTE_FLOW_ACTION_TYPE_SET_IPV6_SRC) || \
    ((act_type) == RTE_FLOW_ACTION_TYPE_SET_IPV6_DST) || \
    ((act_type) == RTE_FLOW_ACTION_TYPE_SET_IPV6_HOP) || \
    ((act_type) == OVS_RTE_FLOW_ACTION_TYPE(SET_UDP_SRC)) || \
    ((act_type) == OVS_RTE_FLOW_ACTION_TYPE(SET_UDP_DST)) || \
    ((act_type) == OVS_RTE_FLOW_ACTION_TYPE(SET_TCP_SRC)) || \
    ((act_type) == OVS_RTE_FLOW_ACTION_TYPE(SET_TCP_DST)))

void per_thread_init(void);
void netdev_offload_doca_per_thread_init(void);
void *per_thread_xzalloc(size_t n);
void *per_thread_xcalloc(size_t n, size_t sz);
void *per_thread_xrealloc(void *old_p, size_t old_size, size_t new_size);
void per_thread_free(void *p);
struct ovs_list *per_thread_get_conn_list(int tid);



int
get_zone_id(uint16_t ct_zone, uint32_t *ct_zone_id);

int
add_pattern_match_reg_field(struct flow_patterns *patterns,
                            uint8_t reg_field_id, uint32_t val, uint32_t mask);

int
get_netdev_by_port(struct netdev *netdev,
                   const struct nlattr *nla,
                   int *outdev_id,
                   struct netdev **outdev);

struct ovs_shared_ctx *
get_shared_age_ctx(struct netdev *netdev,
                   uintptr_t app_counter_key,
                   bool create);

struct ovs_shared_ctx *
get_shared_count_ctx(struct netdev *netdev,
                     uintptr_t key,
                     enum ovs_shared_type type,
                     bool create);

int
netdev_offload_doca_get_table_ctx(odp_port_t vport,
                                  uint32_t recirc_id,
                                  struct netdev *physdev,
                                  struct table_ctx **ctx);

int
netdev_offload_doca_hw_miss_packet_recover_sflow(struct netdev *netdev,
                                                 struct dp_packet *packet,
                                                 int sflow_id,
                                                 struct dpif_sflow_attr *sflow_attr);

int
netdev_offload_doca_hw_miss_packet_recover_ext(struct netdev *netdev,
                                               struct dp_packet *packet,
                                               struct dpdk_offload_recovery_info *info);

int
create_rte_flow(struct netdev *netdev,
                const struct rte_flow_attr *attr,
                struct flow_patterns *flow_patterns,
                struct flow_actions *flow_actions,
                struct dpdk_offload_handle *doh,
                struct rte_flow_error *error);

int
create_pre_post_ct(struct netdev *netdev,
                   const struct rte_flow_attr *attr,
                   struct flow_patterns *flow_patterns,
                   struct flow_actions *flow_actions,
                   struct rte_flow_error *error,
                   struct act_resources *act_resources,
                   struct act_vars *act_vars,
                   struct flow_item *fi);

int
outer_encap_set_actions(struct raw_encap_data *raw_encap_data,
                        const struct nlattr *set_actions,
                        const size_t set_actions_len,
                        bool masked);

int
add_nested_action(struct netdev *flowdev,
                  struct netdev *netdev,
                  struct flow_actions *actions,
                  struct nlattr *nl_actions,
                  size_t nl_actions_len,
                  struct act_resources *act_resources,
                  struct act_vars *act_vars,
                  uint8_t nest_level);

int
parse_ct_actions(struct flow_actions *actions,
                 const struct nlattr *ct_actions,
                 const size_t ct_actions_len,
                 struct act_resources *act_resources,
                 struct act_vars *act_vars);

int
add_tnl_decap_action(struct flow_actions *actions,
                     struct act_vars *act_vars);

int
add_recirc_action(struct netdev *netdev,
                  struct flow_actions *actions,
                  const struct nlattr *nla,
                  struct act_resources *act_resources,
                  struct act_vars *act_vars);

int
parse_sample_action(struct flow_actions *actions,
                    const struct nlattr *nl_actions,
                    struct dpif_sflow_attr *sflow_attr,
                    struct act_resources *act_resources,
                    struct act_vars *act_vars);


int
add_meter_action(struct flow_actions *actions,
                 const struct nlattr *nla,
                 struct act_resources *act_resources,
                 struct act_vars *act_vars);

int
add_sflow_action(struct flow_actions *actions,
                 const struct nlattr *nla,
                 struct act_resources *act_resources,
                 struct act_vars *act_vars);

int
add_ct_clear_action(struct flow_actions *actions,
                    struct act_resources *act_resources);

int
get_flow_miss_ctx_id(struct flow_miss_ctx *flow_ctx_data,
                     struct netdev *netdev,
                     uint32_t table_id,
                     uint32_t recirc_id,
                     uint32_t tunnel_id,
                     uint32_t *miss_ctx_id);

int
parse_ct_label_match(struct match *match, struct act_resources *act_resources,
                     struct flow_patterns *patterns, struct flow *consumed_masks,
                     struct act_vars *act_vars);

int
parse_tnl_match_recirc(struct flow_patterns *patterns,
                       struct match *match,
                       struct act_resources *act_resources,
                       struct act_vars *act_vars);

void
offload_provider_api_init(void);

int
netdev_offload_doca_flow_flush_esw_mgr_members(struct netdev *netdev);


int
parse_geneve_match(struct flow_patterns *patterns,
                   struct match *match,
                   struct act_vars *act_vars);

int
netdev_offload_doca_get_stats(struct netdev *netdev,
                              struct netdev_offload_stats *stats);

int
netdev_offload_doca_ct_counter_query(struct netdev *netdev,
                                     uintptr_t counter_key,
                                     long long now,
                                     long long prev_now,
                                     struct dpif_flow_stats *stats);

int
netdev_offload_doca_flow_create(struct netdev *netdev,
                                const struct rte_flow_attr *attr,
                                struct flow_patterns *flow_patterns,
                                struct flow_actions *flow_actions,
                                struct rte_flow_error *error,
                                struct act_resources *act_resources,
                                struct act_vars *act_vars,
                                struct flow_item *fi);

int
netdev_offload_doca_conns_add(struct netdev *netdevs[CT_DIR_NUM], struct batch *conns);

int
netdev_offload_doca_conns_del(struct netdev *netdevs[CT_DIR_NUM], struct batch *conns);

int
netdev_offload_doca_conn_stats(struct netdev *netdevs[CT_DIR_NUM],
                               struct conn *conn,
                               struct dpif_flow_stats *stats,
                               struct dpif_flow_attrs *attrs,
                               long long int now);

void
netdev_offload_doca_upkeep(struct netdev *netdev, bool quiescing);

int
netdev_offload_doca_packet_hw_hash(struct netdev *netdev,
                                   struct dp_packet *packet,
                                   uint32_t seed,
                                   uint32_t *hash);

int
netdev_offload_doca_packet_hw_entropy(struct netdev *netdev,
                                      struct dp_packet *packet,
                                      uint16_t *entropy);

int
netdev_offload_doca_destroy_flow(struct netdev *netdev,
                                 struct dpdk_offload_handle *doh,
                                 const ovs_u128 *ufid, bool is_esw);

void
ufid_to_rte_flow_set(struct ufid_to_rte_flow_data *data,
                     struct netdev *netdev,
                     struct netdev *physdev,
                     struct act_resources *act_resources);

void
add_hairpin_action(struct flow_actions *actions);

int
netdev_offload_doca_insert_tun_header_with_vlan(
    struct raw_encap_data *raw_encap_data,
    const struct ovs_action_push_vlan *vlan);

void
netdev_offload_doca_zero_metadata_match_ignore(struct match *match);

uint32_t netdev_offload_flow_mark_alloc(void);
void netdev_offload_flow_mark_free(uint32_t mark);

struct reg_field *netdev_offload_doca_get_reg_field(uint32_t reg_field_id);

#else /* DPDK_NETDEV */

static inline void
netdev_offload_doca_per_thread_init(void)
{
}

#endif /* DPDK_NETDEV */

#endif /* NETDEV_OFFLOAD_DOCA_H */
