/*
 * SPDX-FileCopyrightText: NVIDIA CORPORATION & AFFILIATES
 * Copyright (c) 2023-2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: LicenseRef-NvidiaProprietary
 *
 * NVIDIA CORPORATION, its affiliates and licensors retain all intellectual
 * property and proprietary rights in and to this material, related
 * documentation and any modifications thereto. Any use, reproduction,
 * disclosure or distribution of this material and related documentation
 * without an express license agreement from NVIDIA CORPORATION or
 * its affiliates is strictly prohibited.
 */

#ifndef DPDK_OFFLOAD_DOCA_H
#define DPDK_OFFLOAD_DOCA_H

#include "dpdk-offload-provider.h"
#include "ovs-doca.h"

struct netdev;
struct doca_offload_esw_ctx;

struct destroy_pipe_elem {
    struct doca_flow_pipe *pipe;
    struct ovs_list node;
};

struct ovs_doca_flow_actions {
    struct doca_flow_actions d;
    uint32_t mark;
    struct nv_mp nv_mp;
    uint32_t encap_type;
    uint32_t encap_size;
};
BUILD_ASSERT_DECL(offsetof(struct ovs_doca_flow_actions, d) == 0);

struct ovs_doca_flow_match {
    struct doca_flow_match d;
    struct nv_mp nv_mp;
};
BUILD_ASSERT_DECL(offsetof(struct ovs_doca_flow_match, d) == 0);

/* Poll the queue 'qid' for completion. If 'sync' is true, wait until
 * all entries in the queue have been completed and all pipes have
 * completed their resizing. */
doca_error_t
doca_offload_complete_queue_esw(struct doca_offload_esw_ctx *esw, unsigned int qid,
                                bool sync);

struct doca_offload_esw_ctx *
doca_offload_esw_ctx_get(const struct netdev *netdev);

void
doca_offload_esw_ctx_pipe_resizing_ref(struct doca_offload_esw_ctx *esw_ctx);

void
doca_offload_esw_ctx_pipe_resizing_unref(struct doca_offload_esw_ctx *esw_ctx);

int
dpdk_offload_doca_sw_meter_init(struct netdev *netdev);

void
dpdk_offload_doca_sw_meter_uninit(struct netdev *netdev);

void
doca_offload_esw_ctx_resized_list_add(struct doca_offload_esw_ctx *esw_ctx,
                                      struct ovs_list *resized_list_node);

struct doca_flow_pipe_entry *
dpdk_offload_sample_entry_get(const struct netdev *netdev);

doca_error_t
doca_offload_add_entry(struct netdev *netdev,
                       unsigned int qid,
                       struct doca_flow_pipe *pipe,
                       const struct ovs_doca_flow_match *match,
                       const struct ovs_doca_flow_actions *actions,
                       const struct doca_flow_monitor *monitor,
                       const struct doca_flow_fwd *fwd,
                       uint32_t flags,
                       struct doca_flow_pipe_entry **pentry);

doca_error_t
doca_offload_remove_entry(struct doca_offload_esw_ctx *esw,
                          unsigned int qid, uint32_t flags,
                          struct doca_flow_pipe_entry **entry)
    OVS_EXCLUDED(mgmt_queue_lock);

doca_error_t
doca_offload_add_entry_hash(struct netdev *netdev,
                            unsigned int qid,
                            struct doca_flow_pipe *pipe,
                            uint32_t index,
                            const struct ovs_doca_flow_actions *actions,
                            const struct doca_flow_monitor *monitor,
                            const struct doca_flow_fwd *fwd,
                            uint32_t flags,
                            struct doca_flow_pipe_entry **pentry)
    OVS_EXCLUDED(mgmt_queue_lock);

struct ovs_list *
doca_offload_destroy_list_get(struct doca_offload_esw_ctx *esw_ctx);

struct doca_flow_pipe *
doca_offload_get_misspath_pipe(struct doca_offload_esw_ctx *esw_ctx);

void
doca_offload_entry_process(struct doca_flow_pipe_entry *entry,
                           uint16_t qid,
                           enum doca_flow_entry_status status,
                           enum doca_flow_entry_op op,
                           void *aux);

void
doca_offload_set_reg_template(struct doca_flow_meta *dmeta, enum dpdk_reg_id id);

void
doca_set_reg_val(struct doca_flow_meta *dmeta, uint8_t reg_field_id, uint32_t val);

void
doca_offload_set_reg_mask(struct doca_flow_meta *dmeta, enum dpdk_reg_id id);

void
doca_offload_gpr_meter_rate_modify(uint32_t req_core_rate,
                                   uint32_t req_port_rate);

void
doca_offload_gpr_mode_set(enum ovs_doca_gpr_mode mode);

struct too_big_map *
doca_offload_too_big_map_get(struct doca_offload_esw_ctx *esw_ctx);

void
doca_set_reg_val_mask(struct doca_flow_meta *dmeta,
                      struct doca_flow_meta *dmeta_mask,
                      uint8_t reg_field_id,
                      uint32_t val);

void
doca_offload_shared_meter_set(uint32_t meter_id, struct doca_flow_resource_meter_cfg *meter_cfg);

void
doca_offload_shared_meter_unset(uint32_t meter_id);

void
doca_offload_shared_meter_stats_get(uint32_t meter_id, struct doca_flow_resource_query *red_stats,
                                    struct doca_flow_resource_query *green_stats);

#endif /* DPDK_OFFLOAD_DOCA */
